/*
 * Copyright (c) 2005 Jeremy Erickson
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "verse.h"
#include <qdom.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qobject.h>
#include "versewatcher.h"
#include "bibleplugin.h"
#include <list>
#include <iostream>

namespace bmemcore
{

Verse::Verse()
{
    mUBook.testament = 0;
    mUBook.book = 0;
}

Verse::Verse(const QDomElement &element)
{
    mUBook.testament = 0;
    mUBook.book = 0;
    parseElement(element);
}


void Verse::parseElement(const QDomElement &element)
{
    QDomNode node = element.firstChild();
    while (!node.isNull())
    {
        if (node.isElement())
        {
            QDomElement elemNode = node.toElement();
            if (elemNode.tagName() == "reference")
            {
                QDomNode subNode = elemNode.firstChild();
                while (!subNode.isNull())
                {
                    if (subNode.isElement())
                    {
                        QDomElement elemSubNode = subNode.toElement();
                        if (elemSubNode.tagName() == "book")
                        {
                            setBook(getNodeText(elemSubNode));
                        }
                        else if (elemSubNode.tagName() == "ubook")
                        {
                            uBookType uBook;
                            QDomNode subSubNode = elemSubNode.firstChild();
                            while (!subSubNode.isNull())
                            {
                                if (subSubNode.isElement())
                                {
                                    QDomElement elemSubSubNode =
                                            subSubNode.toElement();
                                    if (elemSubSubNode.tagName() == "testament")
                                    {
                                        QString nodeText =
                                                getNodeText(elemSubSubNode).stripWhiteSpace();
                                        if (nodeText == "nt"){
                                            uBook.testament = NEW_TESTAMENT;
                                        }
                                        else if (nodeText == "ot"){
                                            uBook.testament = OLD_TESTAMENT;
                                        }
                                        else{
                                            uBook.testament = nodeText.toInt();
                                        }
                                    }
                                    else if (elemSubSubNode.tagName() == "book")
                                    {
                                        uBook.book =
                                            getNodeText(elemSubSubNode).toInt();
                                    }
                                }
                                subSubNode = subSubNode.nextSibling();
                            }
                            setUBook(uBook);
                        }
                        else if (elemSubNode.tagName() == "chapter")
                        {
                            setChapter(getNodeText(elemSubNode));
                        }
                        else if (elemSubNode.tagName() == "verses")
                        {
                            setVerses(getNodeText(elemSubNode));
                        }
                    }
                    subNode = subNode.nextSibling();
                }
            }
            else if (elemNode.tagName() == "translation")
            {
                setTranslation(getNodeText(elemNode));
            }
            else if (elemNode.tagName() == "text")
            {
                setText(getNodeText(elemNode));
            }
            else if (elemNode.tagName() == "categories")
            {
                setCategories(QStringList::split(";",getNodeText(elemNode)));
            }
        }
        node = node.nextSibling();
    }
}


QDomElement Verse::generateElement(QDomDocument* doc) const
{
    bool createdDoc = (doc == 0);
    if (createdDoc)
        doc = new QDomDocument();
    QDomElement verseElement = doc->createElement("verse");
    QDomElement referenceElement = doc->createElement("reference");
    verseElement.appendChild(referenceElement);
    QDomElement bookElement = doc->createElement("book");
    QDomText bookText = doc->createTextNode(getBook());
    referenceElement.appendChild(bookElement);
    bookElement.appendChild(bookText);
    QDomElement uBookElement = doc->createElement("ubook");
    referenceElement.appendChild(uBookElement);
    QDomElement uBookTestamentElement = doc->createElement("testament");
    QString uBookTestamentString;
    if (getUBook().testament == NEW_TESTAMENT){
        uBookTestamentString = "nt";
    }
    else if (getUBook().testament == OLD_TESTAMENT){
        uBookTestamentString = "ot";
    }
    else{
        uBookTestamentString = QString::number(getUBook().testament);
    }
    QDomText uBookTestamentText = doc->createTextNode(uBookTestamentString);
    uBookElement.appendChild(uBookTestamentElement);
    uBookTestamentElement.appendChild(uBookTestamentText);
    QDomElement uBookBookElement = doc->createElement("book");
    QDomText uBookBookText = 
            doc->createTextNode(QString::number(getUBook().book));
    uBookElement.appendChild(uBookBookElement);
    uBookBookElement.appendChild(uBookBookText);
    QDomElement chapterElement = doc->createElement("chapter");
    QDomText chapterText = doc->createTextNode(getChapter());
    referenceElement.appendChild(chapterElement);
    chapterElement.appendChild(chapterText);
    QDomElement versesElement = doc->createElement("verses");
    QDomText versesText = doc->createTextNode(getVerses());
    referenceElement.appendChild(versesElement);
    versesElement.appendChild(versesText);
    QDomElement translationElement = doc->createElement("translation");
    QDomText translationText = doc->createTextNode(getTranslation());
    verseElement.appendChild(translationElement);
    translationElement.appendChild(translationText);
    QDomElement textElement = doc->createElement("text");
    QDomText textText = doc->createTextNode(getText());
    verseElement.appendChild(textElement);
    textElement.appendChild(textText);
    QDomElement categoriesElement = doc->createElement("categories");
    QDomText categoriesText = doc->createTextNode(getCategories().join(";"));
    verseElement.appendChild(categoriesElement);
    categoriesElement.appendChild(categoriesText); 
    if (createdDoc)
        delete doc;
    return verseElement;
}

void Verse::setBookInfo(const QString& book, BiblePlugin& plugin)
{
    setBook(book);
    setUBook(plugin.uBook(book));
}

void Verse::setBookInfo(const uBookType uBook, BiblePlugin& plugin)
{
    setUBook(uBook);
    setBook(plugin.book(uBook));
}

void Verse::setBookInfo(const QString& book, const uBookType uBook)
{
    setBook(book);
    setUBook(uBook);
}

void Verse::setBook(const QString &newBook)
{
    mBook = newBook;
    notifyWatchers(VerseWatcher::CHANGE_BOOK);
}

QString Verse::getBook() const
{
    return mBook;
}

void Verse::setUBook(const uBookType uBook)
{
    mUBook = uBook;
    notifyWatchers(VerseWatcher::CHANGE_UBOOK);
}

uBookType Verse::getUBook() const
{
    return mUBook;
}

void Verse::setChapter(const QString &newChapter)
{
    mChapter = newChapter;
    notifyWatchers(VerseWatcher::CHANGE_CHAPTER);
}

QString Verse::getChapter() const
{
    return mChapter;
}

void Verse::setVerses(const QString &newVerses)
{
    mVerses = newVerses;
    notifyWatchers(VerseWatcher::CHANGE_VERSES);
}

QString Verse::getVerses() const
{
    return mVerses;
}

void Verse::setTranslation(const QString &newTranslation)
{
    mTranslation = newTranslation;
    notifyWatchers(VerseWatcher::CHANGE_TRANSLATION);
}

QString Verse::getTranslation() const
{
    return mTranslation;
}

void Verse::setText(const QString &newText)
{
    mText = newText;
    notifyWatchers(VerseWatcher::CHANGE_TEXT);
}

QString Verse::getText() const
{
    return mText;
}

void Verse::setCategories(const QStringList &newCategories)
{
    mCategories = newCategories;
    notifyWatchers(VerseWatcher::CHANGE_CATEGORIES);
}

QStringList Verse::getCategories() const
{
    return mCategories;
}

void Verse::addCategory(const QString& newCategory)
{
    QStringList::iterator it = mCategories.find(newCategory);
    if (it != mCategories.end())
    {
        mCategories.push_back(newCategory);
        notifyWatchers(VerseWatcher::CHANGE_CATEGORIES);
    }
}

bool Verse::removeCategory(const QString& oldCategory)
{
    QStringList::iterator it = mCategories.find(oldCategory);
    if (it != mCategories.end())
    {
        mCategories.erase(it);
        notifyWatchers(VerseWatcher::CHANGE_CATEGORIES);
        return true;
    }
    else
        return false;
}

bool Verse::replaceCategory(const QString& oldCategory, const QString&
        newCategory)
{
    QStringList::iterator it = mCategories.find(oldCategory);
    if (it != mCategories.end())
    {
        it = mCategories.erase(it);
        mCategories.insert(it, newCategory);
        return true;
    }
    else
        return false;
}

bool Verse::containsCategory(const QString& category) const
{
    QStringList::const_iterator it = mCategories.find(category);
    return (it != mCategories.end());
}

QString Verse::getReference() const
{
    QString toReturn;
    toReturn += getBook();
    toReturn += " ";
    if (!getChapter().stripWhiteSpace().isEmpty())
    {
        toReturn += getChapter();
        toReturn += ":";
    }
    toReturn += getVerses();
    toReturn = toReturn.simplifyWhiteSpace();
    if (toReturn.isEmpty())
    {
        toReturn = QObject::tr("<No Reference>");
    }
    return toReturn;
}

void Verse::addWatcher(VerseWatcher *newWatcher)
{
    mWatchers.push_back(newWatcher);
}

void Verse::removeWatcher(VerseWatcher *theWatcher)
{
    mWatchers.remove(theWatcher);
}

void Verse::setText(BiblePlugin& plugin)
{
    setText(plugin.getVerse(mBook, mUBook, mChapter, mVerses, mTranslation));
}

bool Verse::isTextAvailable(BiblePlugin& plugin)
{
    return plugin.verseAvailable(mBook, mUBook, mChapter, mVerses,
            mTranslation);
}

void Verse::notifyKilled()
{
    notifyWatchers(VerseWatcher::CHANGE_DESTROYED);
}

void Verse::notifyWatchers(VerseWatcher::ChangeType whatChanged)
{
    std::list<VerseWatcher*>::iterator it = mWatchers.begin();
    while (it != mWatchers.end())
    {
        (*it)->verseChanged(*this, whatChanged);
        it++;
    }
}

QString Verse::getNodeText(QDomElement &element)
{
    QDomNode node = element.firstChild();
    QString toReturn = "";
    while (!node.isNull())
    {
        if (node.isText())
        {
            toReturn = node.toText().data();
            break;
        }
        node = node.nextSibling();
    }
    return toReturn;
}

QStringList Verse::tokenize(const QString& text)
{
    return QStringList::split(" ", text, false);
}

bool Verse::checkText(const QString& text)
{
    return(tokenize(text) == tokenize(mText));
}

std::list<CorrectionWord> Verse::correctText(const QString& text)
{
    QStringList first = tokenize(mText);
    QStringList second = tokenize(text);
    //We are iterating through both at once, adding words as either
    //correct, missing, or wrong (i.e. only in user text)
    QStringList::const_iterator it1 = first.begin();
    QStringList::const_iterator it2 = second.begin();
    //Used as a search distance when not found.
    //This is guaranteed to be a bigger distance than for any actual search.
    int combinedSize = first.size() + second.size();
    std::list<CorrectionWord> toReturn;
    while (it1 != first.end() && it2 != second.end())
    {
        //When both are the same, we have a correct word.
        if (*it1 == *it2)
        {
            CorrectionWord nextCorrection(*it1,
                    CorrectionWord::WORD_STATE_CORRECT);
            toReturn.push_back(nextCorrection);
            ++it1;
            ++it2;
        }
        else
        {
            //Here we are determining how far the correct word is in the
            //user's text, and vice versa.
            QStringList::const_iterator it1Find = it1;
            QStringList::const_iterator it2Find = it2;
            int twoInOne = 0;
            int oneInTwo = 0;
            //Iterate until we find it.
            while (it1Find != first.end() && *it1Find != *it2)
            {
                ++twoInOne;
                ++it1Find;
            }
            //If we didn't find it, use the proper search distance.
            if (it1Find == first.end())
                twoInOne = combinedSize;
            //Same procedure as above.
            while (it2Find != second.end() && *it2Find != *it1)
            {
                ++oneInTwo;
                ++it2Find;
            }
            if (it2Find == second.end())
                oneInTwo = combinedSize;
            //Word from user's text is closer, approach it in correct text.
            if (twoInOne < oneInTwo)
            {
                CorrectionWord nextWord(*it1,
                        CorrectionWord::WORD_STATE_MISSING);
                toReturn.push_back(nextWord);
                ++it1;
            }
            //Word from correct text is closer, approach it in user's text.
            //Alternatively, they might have been at equal distances, or
            //neither found.  In this case, defaulting to this shows crossing
            //out before the correct word.
            else
            {
                CorrectionWord nextWord(*it2,
                        CorrectionWord::WORD_STATE_WRONG);
                toReturn.push_back(nextWord);
                ++it2;
            }
        }
    }
    //Pick up anything we missed.
    while (it1 != first.end())
    {
        CorrectionWord nextWord(*it1,
                CorrectionWord::WORD_STATE_MISSING);
        toReturn.push_back(nextWord);
        ++it1;
    }
    while (it2 != second.end())
    {
        CorrectionWord nextWord(*it2,
                CorrectionWord::WORD_STATE_WRONG);
        toReturn.push_back(nextWord);
        ++it2;
    }
    return toReturn;
}

int Verse::compareTo(const Verse& other) const
{
    if (getUBook().testament != other.getUBook().testament){
        return getUBook().testament - other.getUBook().testament;
    }
    else if (getUBook().book != other.getUBook().book){
        return getUBook().book - other.getUBook().book;
    }
    else if (getChapter() != other.getChapter()){
        return getChapter().toInt() - other.getChapter().toInt();
    }
    else{
        QString myVerses = getVerses().stripWhiteSpace();
        for (int i = 0; i < myVerses.length(); i++){
            if (!myVerses.ref(i).isDigit()){
                myVerses.remove(i, myVerses.length());
                break;
            }
        }
        QString otherVerses = other.getVerses().stripWhiteSpace();
        for (int i = 0; i < otherVerses.length(); i++){
            if (!otherVerses.ref(i).isDigit()){
                otherVerses.remove(i, otherVerses.length());
            }
        }
        return myVerses.toInt() - otherVerses.toInt();
    }
}

Verse::~Verse()
{
    notifyWatchers(VerseWatcher::CHANGE_DESTROYED);
}

}

