/*=====================================================================*/
/*    .../prgm/project/bigloo/api/fthread/src/Posix/bglfthread.c       */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Fri Feb 22 12:12:04 2002                          */
/*    Last change :  Fri Jun  2 11:42:48 2006 (serrano)                */
/*    Copyright   :  2002-06 Manuel Serrano                            */
/*    -------------------------------------------------------------    */
/*    C utilities for native Bigloo fair threads implementation.       */
/*=====================================================================*/
#include <pthread.h>
#include <sched.h>
#include <stdlib.h>
#include <gc.h>

#define GC_PRIVATE_H
#include <bglfthread.h>

/*---------------------------------------------------------------------*/
/*    Runtime system                                                   */
/*---------------------------------------------------------------------*/
BGL_IMPORT char *strerror();

/*---------------------------------------------------------------------*/
/*    Imports                                                          */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DECL void bgl_multithread_dynamic_denv_register();
extern void bglfth_mutexes_unlock( bglfthread_t );

/*---------------------------------------------------------------------*/
/*    Global token                                                     */
/*---------------------------------------------------------------------*/
pthread_key_t bglkey;
static pthread_key_t bglidkey;

static bglfthread_t token;

static pthread_mutex_t first_lock;
static pthread_cond_t first_cv;

/*---------------------------------------------------------------------*/
/*    bgldenv_t                                                        */
/*    bglfth_single_thread_denv ...                                    */
/*---------------------------------------------------------------------*/
static bgldenv_t bglfth_single_thread_denv = 0L;

/*---------------------------------------------------------------------*/
/*    pthread_key_t                                                    */
/*    bgldenv_key ...                                                  */
/*---------------------------------------------------------------------*/
static pthread_key_t bgldenv_key;

/*---------------------------------------------------------------------*/
/*    bgldenv_t                                                        */
/*    bglfth_dynamic_env ...                                           */
/*---------------------------------------------------------------------*/
bgldenv_t
bglfth_dynamic_env() {
   return (bgldenv_t)pthread_getspecific( bgldenv_key );
}
 
/*---------------------------------------------------------------------*/
/*    bgldenv_t                                                        */
/*    bglfth_dynamic_env_set ...                                       */
/*---------------------------------------------------------------------*/
bgldenv_t
bglfth_dynamic_env_set( bgldenv_t env ) {
   pthread_setspecific( bgldenv_key, env );

   return env;
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    bglfth_thread_cleanup ...                                        */
/*---------------------------------------------------------------------*/
static void
bglfth_thread_cleanup( void *arg ) {
   bglfthread_t self = (bglfthread_t)arg;

   /* lock the internal state of the thread */
   pthread_mutex_lock( &(self->lock) );

   /* unlock all locked mutexes */
   bglfth_mutexes_unlock( self );
 
   /* unlock the internal state of the thread */
   pthread_mutex_unlock( &(self->lock) );
}

/*---------------------------------------------------------------------*/
/*    static void *                                                    */
/*    bglfth_thread_run ...                                            */
/*---------------------------------------------------------------------*/
static void *
bglfth_thread_run( void *arg ) {
   bglfthread_t self = (bglfthread_t)arg;

   /* store the thread's stack bottom address */
   /* in the dynamic env per thread structure */
   bglfth_dynamic_env_set( self->env );
   self->env->stack_bottom = (char *)&arg;
   self->env->current_thread = self;

   /* prepare the cleanup of this thread */
   pthread_cleanup_push( bglfth_thread_cleanup, arg );
   
   /* we have to store in the thread, the pointer to the bglfth_thread */
   pthread_setspecific( bglkey, arg );
   
   /* Mark the id for bmem */
   bglfth_thread_id_set( self, self->name );
   
   bglfth_thread_wait( self );

   PROCEDURE_ENTRY( self->thunk )( self->thunk, BEOA );
   pthread_cleanup_pop( 1 );
       
   return (void *)arg;
}

/*---------------------------------------------------------------------*/
/*    bglfthread_t                                                     */
/*    bglfth_thread_new ...                                            */
/*---------------------------------------------------------------------*/
bglfthread_t
bglfth_thread_new( obj_t proc ) {
   bglfthread_t thread = (bglfthread_t)GC_MALLOC( sizeof(struct bglfthread) );
   
   thread->thunk = proc;
   thread->name = BUNSPEC;

   if( pthread_mutex_init( &(thread->lock), 0L ) )
      goto err;
   if( pthread_cond_init( &(thread->cv), 0L ) )
      goto err;

   return thread;

err:
   FAILURE( string_to_bstring( "make-thread" ),
	    string_to_bstring( "Cannot create thread" ),
	    string_to_bstring( strerror( errno ) ) );

   return 0L;
}

/*---------------------------------------------------------------------*/
/*    bglfthread_t                                                     */
/*    bglfth_thread_name_new ...                                       */
/*---------------------------------------------------------------------*/
bglfthread_t
bglfth_thread_new_with_name( obj_t proc, obj_t name ) {
   bglfthread_t th = bglfth_thread_new( proc );
   th->name = name;
   return th;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bglfth_thread_id_set ...                                         */
/*---------------------------------------------------------------------*/
obj_t
bglfth_thread_id_set( bglfthread_t thread, obj_t id ) {
   pthread_setspecific( bglidkey, (void *)id );

   return id;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bglfth_thread_id_get ...                                         */
/*---------------------------------------------------------------------*/
obj_t
bglfth_thread_id_get() {
   return (obj_t)pthread_getspecific( bglidkey );
}

/*---------------------------------------------------------------------*/
/*    static bglfthread_t                                              */
/*    bglfth_current_thread ...                                        */
/*---------------------------------------------------------------------*/
bglfthread_t
bglfth_current_thread() {
   return (bglfthread_t)pthread_getspecific( bglkey );
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglfth_thread_start ...                                          */
/*---------------------------------------------------------------------*/
void
bglfth_thread_start( bglfthread_t thread, obj_t bglfth_thread ) {
   pthread_attr_t a;

   pthread_attr_init( &a );
   pthread_attr_setdetachstate( &a, PTHREAD_CREATE_DETACHED );

   thread->env = bgl_dup_dynamic_env( BGL_DYNAMIC_ENV() );
   thread->bglobj = bglfth_thread;
   
   if( pthread_create( &(thread->pthread), &a, bglfth_thread_run, thread ) )
      FAILURE( string_to_bstring( "thread-start!" ),
	       string_to_bstring( "Cannot start thread" ),
	       string_to_bstring( strerror( errno ) ) );
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglfth_thread_wait ...                                           */
/*---------------------------------------------------------------------*/
void
bglfth_thread_wait( bglfthread_t this ) {
   pthread_mutex_lock( &(this->lock) );
   while( token != this ) {
      pthread_cond_wait( &(this->cv), &(this->lock) );
   }
   pthread_mutex_unlock( &(this->lock) );
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglfth_thread_switch ...                                         */
/*---------------------------------------------------------------------*/
void
bglfth_thread_switch( bglfthread_t this, bglfthread_t next ) {
   pthread_mutex_lock( &(next->lock) );
   
   token = next;
   
   pthread_cond_signal( &(next->cv) );
   pthread_mutex_unlock( &(next->lock) );
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglfth_thread_enter_scheduler ...                                */
/*---------------------------------------------------------------------*/
void
bglfth_thread_enter_scheduler( bglfthread_t scdl ) {
   pthread_mutex_t *this_lock;
   pthread_cond_t *this_cv;
   bglfthread_t this;

   this = bglfth_current_thread();

   scdl->parent = this;

   if( this ) {
      this_lock = &(this->lock);
      this_cv = &(this->cv);
   } else {
      this_lock = &first_lock;
      this_cv = &first_cv;
   }

   /* switch into the scdl thread */
   bglfth_thread_switch( this, scdl );

   /* wait for the token */
   pthread_mutex_lock( this_lock );
   while( token != this ) pthread_cond_wait( this_cv, this_lock );
   pthread_mutex_unlock( this_lock );
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglfth_thread_leave_scheduler ...                                */
/*---------------------------------------------------------------------*/
void
bglfth_thread_leave_scheduler( bglfthread_t scdl ) {
   pthread_mutex_t *parent_lock;
   pthread_cond_t *parent_cv;

   if( scdl->parent ) {
      parent_lock = &(scdl->parent->lock);
      parent_cv = &(scdl->parent->cv);
   } else {
      parent_lock = &first_lock;
      parent_cv = &first_cv;
   }
   
   token = 0L;

   pthread_mutex_lock( parent_lock );
   token = scdl->parent;
   pthread_cond_signal( parent_cv );
   pthread_mutex_unlock( parent_lock );

   pthread_mutex_lock( &(scdl->lock) );
   while( token != scdl ) pthread_cond_wait( &(scdl->cv), &(scdl->lock) );
   pthread_mutex_unlock( &(scdl->lock) );
}
   
/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglfth_setup_thread ...                                          */
/*---------------------------------------------------------------------*/
void
bglfth_setup_thread() {
   /* Can't handle this code with MINGW */
   /* Don't know of the impact */
#if HAVE_SIGACTION
   struct sigaction sigact;
#endif
     
   token = 0L;

   /* Can't handle this code with MINGW */
   /* Don't know of the impact */
#if HAVE_SIGACTION
   sigemptyset( &(sigact.sa_mask) );
   sigact.sa_handler = SIG_IGN;
   sigact.sa_flags = SA_RESTART;
   sigaction( SIGPIPE, &sigact, NULL );
#endif
   if( pthread_key_create( &bgldenv_key, 0L ) )
      goto err;

   if( pthread_key_create( &bglkey, 0L ) )
      goto err;

   if( pthread_key_create( &bglidkey, 0L ) )
      goto err;

   if( pthread_mutex_init( &first_lock, 0L ) )
      goto err;
      
   if( pthread_cond_init( &first_cv, 0L ) )
      goto err;

   /* main dynamic env init */
   bgl_init_dynamic_env();

   /* Keep the environment in a global for the GC */
   bglfth_single_thread_denv = single_thread_denv;

   bglfth_dynamic_env_set( single_thread_denv );
   single_thread_denv = 0;
 
   bgl_multithread_dynamic_denv_register( &bglfth_dynamic_env );

   return;

 err:
   FAILURE( string_to_bstring( "bglfth_thread_init" ),
	    string_to_bstring( "Cannot initialize" ),
	    string_to_bstring( strerror( errno ) ) );
}
