
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <fcntl.h>

#include "brighton.h"
#include "brightonMini.h"

static int pro10Init();
static int pro10Configure();
static int pro10Callback(void *, int, int, float);
static int lmCallback(void *, int, int, float);
static int lmmodCallback(void *, int, int, float);

extern guimain global;
/*
 * These have to be integrated into the synth private structures, since we may
 * have multiple invocations of any given synth.
static guiSynth *lower;
 */
static guimain manual;

#include "brightonKeys.h"

#define FIRST_DEV 0
#define DEVICE_COUNT (66 + FIRST_DEV)
#define ACTIVE_DEVS (47 + FIRST_DEV)
#define DISPLAY_DEV (65 + FIRST_DEV)
#define RADIOSET_1 50

#define KEY_PANEL 1

#define R1 140
#define RB1 160
#define R2 425
#define RB2 445
#define R3 710
#define RB3 730

#define C1 20
#define C2 70
#define CB1 60
#define CB2 90
#define C3 120
#define C4 150
#define C5 180

#define C6 235
#define CB6 275
#define CB7 305
#define C7 340
#define CB8 390

#define C9 280
#define CB9 330
#define CB10 360
#define CB11 390
#define C10 430
#define CB12 480
#define CB13 510

#define C11 442
#define C12 490
#define C13 538

#define C14 595
#define C15 642
#define C16 688
#define C17 735

#define C18 800
#define C19 850
#define C20 900
#define C21 950

#define C22 877

#define S1 120

#define B1 14
#define B2 80

/*
 * This structure is for device definition. The structure is defined in 
 * include/brighton.h, further definitions in brighton/brightonDevtable.h and
 * include/brightoninternals.h
 *
 *	typedef int (*brightonCallback)(int, float);
 *	typedef struct BrightonLocations {
 *		int device; 0=rotary, 1=scale, etc.
 *		float relx, rely; relative position with regards to 1000 by 1000 window
 *		float relw, relh; relative height.
 *		int from, to;
 *		brightonCallback callback; specific to this dev
 *		char *image; bitmap. If zero take a device default.
 *		int flags;
 *	} brightonLocations;
 *
 * This example is for a pro10Bristol type synth interface.
 */
static brightonLocations locations[DEVICE_COUNT] = {
/* poly mod */
	{"", 0, C1, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C2, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, C3, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, C4, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, C5, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
/* lfo */
	{"", 0, C2, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, C3, RB2, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, C4, RB2, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, C5, RB2, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
/* wheel mod */
	{"", 0, C1, R3, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, CB1, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, CB2, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, C3, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, C4, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, C5, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
/* osc a */
	{"", 0, C6, R1, S1, S1, 0, 5, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, CB6, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, CB7, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 0, C7, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, CB8, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
/* osc b */
	{"", 0, C6, R2, S1, S1, 0, 5, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C9, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, CB9, RB2, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, CB10, RB2, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, CB11, RB2, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 0, C10, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, CB12, RB2, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 2, CB13, RB2, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
/* glide, etc */
	{"", 0, C6, R3, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, CB7, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
/* mixer */
	{"", 0, C11, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C12, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C13, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
/* filter + env */
	{"", 0, C14, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C15, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C16, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, C17, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	{"", 0, C14, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C15, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C16, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C17, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
/* main env */
	{"", 0, C18, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C19, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C20, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C21, R2, S1, S1, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
/* tune + vol */
	{"", 0, C18, R1, S1, S1, 0, 1, 0, "bitmaps/knobs/knob3.xpm", 0, 0},
	{"", 0, C22, R3, S1, S1, 0, 1, 0, "bitmaps/knobs/knob3.xpm", 0, 0},
	{"", 2, C18 + 7, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", BRIGHTON_CHECKBUTTON},
	{"", 2, C19 + 7, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", 0},
	{"", 2, C21 + 10, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_CHECKBUTTON},
/* memories */
	{"", 2, C14 + 10, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_RADIOBUTTON},
	{"", 2, C14 + 30, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_RADIOBUTTON},
	{"", 2, C14 + 50, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_RADIOBUTTON},
	{"", 2, C14 + 70, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_RADIOBUTTON},
	{"", 2, C14 + 90, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_RADIOBUTTON},
	{"", 2, C14 + 110, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_RADIOBUTTON},
	{"", 2, C14 + 130, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_RADIOBUTTON},
	{"", 2, C14 + 150, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_RADIOBUTTON},
	{"", 2, CB8, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_CHECKBUTTON},
	{"", 2, CB8 + 30, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffo.xpm", "bitmaps/buttons/pressono.xpm", BRIGHTON_CHECKBUTTON},
	{"", 2, CB8 + 60, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_CHECKBUTTON},
/* Midi, perhaps eventually file import/export buttons */
	{"", 2, C20 + 10, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_CHECKBUTTON},
	{"", 2, C21 + 10, RB1, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoffg.xpm", "bitmaps/buttons/pressong.xpm", BRIGHTON_CHECKBUTTON},
	// Synth select
	{"", 2, CB8 - 30, RB3, B1, B2, 0, 1, 0, "bitmaps/buttons/pressoff.xpm", "bitmaps/buttons/presson.xpm", 0},
	// LOGO
	{"", 4, 857, 1025, 143, 150, 0, 1, 0, "bitmaps/images/pro10.xpm", 0, 0},
	// Display
	{"", 3, CB8 + 85, RB3, 120, B2, 0, 1, 0, 0, 0, 0}
};

/*
 */

static int
pro10destroy(int cid)
{
	guiSynth *synth = findSynth(global.synths, cid);
	bristolMidiMsg msg;

//printf("pro10destroy(%i): %i\n", cid, synth->midichannel);
	/*
	 * Since we registered two synths, we now need to remove the upper
	 * manual.
	 */
	bristolMidiSendMsg(manual.controlfd, synth->sid2, 127, 0,
		BRISTOL_EXIT_ALGO);
	if (bristolMidiRead(manual.controlfd, &msg) < 0)
		printf("socket closed\n");
	bristolMidiSendMsg(global.controlfd, synth->sid, 127, 0,
		BRISTOL_EXIT_ALGO);
	if (bristolMidiRead(global.controlfd, &msg) < 0)
		printf("socket closed\n");
}

/*
 * This is a set of globals for the main window rendering. Again taken from
 * include/brighton.h
 */
brightonApp pro10App = {
	"prophet10",
	0, // no blueprint on wood background.
	"bitmaps/textures/wood3.xpm",
	0, // BRIGHTON_STRETCH, //flags
	pro10Init,
	pro10Configure, // 3 callbacks, unused?
	0,
	pro10destroy,
	745, 410,
	7, // one panel only
	{
		{
			"Pro10",
			"bitmaps/blueprints/prophet10.xpm",
			"bitmaps/textures/metal5.xpm",
			0, //BRIGHTON_STRETCH, // flags
			0,
			0,
			pro10Callback,
			15, 25, 970, 400,
			DEVICE_COUNT,
			locations
		},
/*
		{
			"Pro10",
			"bitmaps/blueprints/prophet10.xpm",
			"bitmaps/textures/metal5.xpm",
			0, //BRIGHTON_STRETCH, // flags
			0,
			0,
			pro10Callback,
			15, 25, 970, 400,
			DEVICE_COUNT,
			locations
		},
*/
		{
			"Keybaord",
			0,
			"bitmaps/keys/kbg.xpm", // flags
			0x020|BRIGHTON_STRETCH,
			0,
			0,
			keyCallback,
			110, 505, 870, 240,
			KEY_COUNT,
			keys
		},
		{
			"Mods",
			"bitmaps/blueprints/prophetmod.xpm",
			"bitmaps/textures/metal5.xpm", // flags
			0,
			0,
			0,
			modCallback,
			15, 505, 95, 240,
			2,
			mods
		},
		{
			"Keybaord",
			0,
			"bitmaps/keys/kbg.xpm", // flags
			0x020|BRIGHTON_STRETCH,
			0,
			0,
			lmCallback,
			110, 745, 870, 240,
			KEY_COUNT,
			keys
		},
		{
			"Mods",
			"bitmaps/blueprints/prophetmod.xpm",
			"bitmaps/textures/metal5.xpm", // flags
			0,
			0,
			0,
			lmmodCallback,
			15, 745, 95, 240,
			2,
			mods
		},
		{
			"Pro10",
			0,
			"bitmaps/textures/wood3.xpm",
			BRIGHTON_STRETCH|BRIGHTON_VERTICAL, // flags
			0,
			0,
			0,
			0, 0, 15, 1000,
			0,
			0
		},
		{
			"Pro10",
			0,
			"bitmaps/textures/wood3.xpm",
			BRIGHTON_STRETCH|BRIGHTON_VERTICAL, //flags
			0,
			0,
			0,
			985, 0, 15, 1000,
			0,
			0
		}
	}
};

//static dispatcher dispatch[DEVICE_COUNT];

static int
pro10MidiSendMsg(void *synth, int fd, int chan, int c, int o, int v)
{
	bristolMidiSendMsg(fd, chan, c, o, v);
}

static int
multiA440(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	if (v != 0) {
		bristolMidiSendMsg(fd, synth->sid, 126, 3, 1);
		bristolMidiSendMsg(fd, synth->sid2, 126, 3, 1);
	} else {
		bristolMidiSendMsg(fd, synth->sid, 126, 3, 0);
		bristolMidiSendMsg(fd, synth->sid2, 126, 3, 0);
	}
}

static int
multiTune(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	brightonEvent event;
	int hold = synth->mem.param[63];

//printf("multiTune()\n");
	/*
	 * Configures all the tune settings to zero (ie, 0.5). Do the Osc-A first,
	 * since it is not visible, and then request the GUI to configure Osc-B.
	 */
	synth->mem.param[63] = 0;
	bristolMidiSendMsg(global.controlfd, synth->sid, 0, 2, 8191);
	event.type = BRIGHTON_FLOAT;
	event.value = 0.49;
	brightonParamChange((void *) synth->connid, 0, FIRST_DEV + 45, &event);
	brightonParamChange((void *) synth->connid, 0, FIRST_DEV + 21, &event);
	event.value = 0.50;
	brightonParamChange((void *) synth->connid, 0, FIRST_DEV + 45, &event);
	brightonParamChange((void *) synth->connid, 0, FIRST_DEV + 21, &event);

	synth->mem.param[63] = 1;
	bristolMidiSendMsg(global.controlfd, synth->sid2, 0, 2, 8191);
	event.value = 0.49;
	brightonParamChange((void *) synth->connid, 0, FIRST_DEV + 45, &event);
	brightonParamChange((void *) synth->connid, 0, FIRST_DEV + 21, &event);
	event.value = 0.50;
	brightonParamChange((void *) synth->connid, 0, FIRST_DEV + 45, &event);
	brightonParamChange((void *) synth->connid, 0, FIRST_DEV + 21, &event);

	synth->mem.param[63] = hold;

	((guiSynth *) synth->lower)->mem.param[49] = 0.5;
	synth->mem.param[49] = 0.5;
}

static int
midiRelease(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	if (!global.libtest)
	{
		bristolMidiMsg msg;

		/*
		 * Midi release is ALL notes, ALL synths. If this behaviour (in the
		 * engine) changes we may need to put in an all_notes_off on the
		 * lower manual as well.
		 */
		bristolMidiSendMsg(fd, synth->sid, 127, 0,
			BRISTOL_ALL_NOTES_OFF);

		if (bristolMidiRead(fd, &msg) != BRISTOL_OK)
			cleanupBristol();
	}
}

static int
lmmodCallback(void *cid, int panel, int index, float value)
{
	guiSynth *synth = findSynth(global.synths, (int) cid);

//	printf("modCallback(%x, %i, %i, %f)\n", synth, panel, index, value);

	/*
	 * If this is controller 0 it is the frequency control, otherwise a 
	 * generic controller 1.
	 */
	if (index == 0)
		bristolMidiSendMsg(global.controlfd, synth->midichannel + 1,
			BRISTOL_EVENT_PITCH, 0, (int) (value * C_RANGE_MIN_1));
	else {
		bristolMidiControl(global.controlfd, synth->midichannel + 1,
			0, 1, ((int) (value * C_RANGE_MIN_1)) >> 7);
	}
}

static int
lmCallback(void *cid, int panel, int index, float value)
{
	guiSynth *synth = findSynth(global.synths, (int) cid);

//	printf("keycallback(%x, %i, %i, %f): %i %i\n", synth, panel, index, value,
//		synth->transpose, global.controlfd);

	/*
	 * Want to send a note event, on or off, for this index + transpose.
	 */
	if (value)
		bristolMidiSendMsg(global.controlfd, synth->midichannel + 1,
			BRISTOL_EVENT_KEYON, 0, index + synth->transpose);
	else
		bristolMidiSendMsg(global.controlfd, synth->midichannel + 1,
			BRISTOL_EVENT_KEYOFF, 0, index + synth->transpose);
}

static int
dualTune(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
//	printf("dualTune\n");

	bristolMidiSendMsg(global.controlfd, synth->sid, 126, 2, v);
	bristolMidiSendMsg(global.controlfd, synth->sid2, 126, 2, v);
}

static int
pro10PanelSwitch(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	brightonEvent event;
	int i;

//	printf("pro10PanelSwitch(%x, %i, %i, %i, %i, %1.0f)\n",
//		synth, fd, chan, c, o, global.synths->mem.param[63]);

	/*
	 * Prevent param changes when we are switching panels.
	 */
	synth->flags |= SUPPRESS;

	/*
	 * Go through all the active devices, and force the GUI to represent
	 * the new value.
	 */
	if (synth->mem.param[63] == 0)
	{
		/*
		 * Go through all the params, and request an update to the GUI
		 */
		for (i = 0; i < ACTIVE_DEVS; i++)
		{
			event.type = BRIGHTON_FLOAT;
			event.value = ((guiSynth *) synth->lower)->mem.param[i] + 1;
			brightonParamChange((void *) synth->connid, 0, i, &event);
			event.value = ((guiSynth *) synth->lower)->mem.param[i];
			brightonParamChange((void *) synth->connid, 0, i, &event);
		}

		displayPanelText(synth, "LOWER", ((guiSynth *) synth->lower)->bank + ((guiSynth *) synth->lower)->location,
			0, DISPLAY_DEV);
		synth->mem.param[63] = 1;
	} else {
		for (i = 0; i < ACTIVE_DEVS; i++)
		{
			event.type = BRIGHTON_FLOAT;
			event.value = synth->mem.param[i] + 1;
			brightonParamChange((void *) synth->connid, 0, i, &event);
			event.value = synth->mem.param[i];
			brightonParamChange((void *) synth->connid, 0, i, &event);
		}

		displayPanelText(synth, "UPPER", synth->bank + synth->location,
			0, DISPLAY_DEV);
		synth->mem.param[63] = 0;
	}

	synth->flags &= ~SUPPRESS;
}

static int
pro10Memory(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	brightonEvent event;

	if (synth->flags & SUPPRESS)
		return;

	if (synth->dispatch[RADIOSET_1].other2)
	{
		synth->dispatch[RADIOSET_1].other2 = 0;
		return;
	}

//printf("pro10Memory(%x)\n", synth);

	if (synth->mem.param[63] != 0)
		/*
		 * take the second synth
		 */
		synth = ((guiSynth *) synth->lower);

	switch (c) {
		default:
		case 0:
			/*
			 * We want to make these into memory buttons. To do so we need to
			 * know what the last active button was, and deactivate its 
			 * display, then send any message which represents the most
			 * recently configured value. Since this is a memory button we do
			 * not have much issue with the message, but we are concerned with
			 * the display.
			 */
			if (synth->dispatch[RADIOSET_1].other1 != -1)
			{
				synth->dispatch[RADIOSET_1].other2 = 1;

				if (synth->dispatch[RADIOSET_1].other1 != o)
					event.value = 0;
				else
					event.value = 1;

				brightonParamChange((void *) synth->connid, synth->panel,
					synth->dispatch[RADIOSET_1].other1 + 49, &event);
			}
			synth->dispatch[RADIOSET_1].other1 = o;

			if (synth->flags & BANK_SELECT) {
				if ((synth->bank * 10 + o * 10) >= 1000)
				{
					synth->location = o;
					synth->flags &= ~BANK_SELECT;
					if (loadMemory(synth, "prophet", 0,
						synth->bank + synth->location, synth->mem.active,
							FIRST_DEV, BRISTOL_STAT) < 0)
						displayText(synth, "FRE", synth->bank + synth->location,
							DISPLAY_DEV);
					else
						displayText(synth, "PRG", synth->bank + synth->location,
							DISPLAY_DEV);
				} else {
					synth->bank = synth->bank * 10 + o * 10;
					if (loadMemory(synth, "prophet", 0,
						synth->bank + synth->location, synth->mem.active,
							FIRST_DEV, BRISTOL_STAT) < 0)
						displayText(synth, "FRE", synth->bank + synth->location,
							DISPLAY_DEV);
					else
						displayText(synth, "BANK",
							synth->bank + synth->location, DISPLAY_DEV);
				}
			} else {
				if (synth->bank < 10)
					synth->bank = 10;
				synth->location = o;
				if (loadMemory(synth, "prophet", 0,
					synth->bank + synth->location, synth->mem.active,
						FIRST_DEV, BRISTOL_STAT) < 0)
					displayText(synth, "FRE", synth->bank + synth->location,
						DISPLAY_DEV);
				else
					displayText(synth, "PRG", synth->bank + synth->location,
						DISPLAY_DEV);
			}
			break;
		case 1:
			if (synth->bank < 10)
				synth->bank = 10;
			if (synth->location == 0)
				synth->location = 1;
			if (loadMemory(synth, "prophet", 0, synth->bank + synth->location,
				synth->mem.active, FIRST_DEV, 0) < 0)
				displayText(synth, "FRE", synth->bank + synth->location,
					DISPLAY_DEV);
			else
				displayText(synth, "PRG", synth->bank + synth->location,
					DISPLAY_DEV);
			synth->flags &= ~BANK_SELECT;
			break;
		case 2:
			if (synth->bank < 10)
				synth->bank = 10;
			if (synth->location == 0)
				synth->location = 1;
			saveMemory(synth, "prophet", 0, synth->bank + synth->location,
				FIRST_DEV);
			displayText(synth, "PRG", synth->bank + synth->location,
				DISPLAY_DEV);
			synth->flags &= ~BANK_SELECT;
			break;
		case 3:
			if (synth->flags & BANK_SELECT) {
				synth->flags &= ~BANK_SELECT;
				if (loadMemory(synth, "prophet", 0,
					synth->bank + synth->location, synth->mem.active,
						FIRST_DEV, BRISTOL_STAT) < 0)
					displayText(synth, "FRE", synth->bank + synth->location,
						DISPLAY_DEV);
				else
					displayText(synth, "PRG", synth->bank + synth->location,
						DISPLAY_DEV);
			} else {
				synth->bank = 0;
				displayText(synth, "BANK", synth->bank, DISPLAY_DEV);
				synth->flags |= BANK_SELECT;
			}
			break;
	}
//	printf("	pro10Memory(B: %i L %i: %i)\n",
//		synth->bank, synth->location, o);
}

static int
pro10Midi(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	bristolMidiMsg msg;
	int newchan;

	if (c == 1) {
		if ((newchan = synth->midichannel - 1) < 0)
		{
			synth->midichannel = 0;
			return;
		}
	} else {
		if ((newchan = synth->midichannel + 1) > 14)
		{
			synth->midichannel = 14;
			return;
		}
	}

	if (global.libtest == 0)
	{
		/*
		 * Do lower manual, then upper.
		 */
		bristolMidiSendMsg(global.controlfd, synth->sid2,
			127, 0, BRISTOL_MIDICHANNEL|newchan + 1);
		if (bristolMidiRead(global.controlfd, &msg) != BRISTOL_OK)
			cleanupBristol();
		bristolMidiSendMsg(global.controlfd, synth->sid,
			127, 0, BRISTOL_MIDICHANNEL|newchan);
		if (bristolMidiRead(global.controlfd, &msg) != BRISTOL_OK)
			cleanupBristol();
	}

	synth->midichannel = newchan;

//printf("P: going to display: %x, %x\n", synth, synth->connid);
	displayText(synth, "MIDI", synth->midichannel + 1, DISPLAY_DEV);
}

/*
 * For the sake of ease of use, links have been placed here to be called
 * by any of the devices created. They would be better in some other file,
 * perhaps with this as a dispatch.
 *
 * Param refers to the device index in the locations table given below.
 */
static int
pro10Callback(void *cid, int panel, int index, float value)
{
	guiSynth *synth = findSynth(global.synths, (int) cid);
	int sendvalue, sid;

//	printf("pro10Callback(%i, %f): %x, %1.0f\n", index, value, synth,
//		global.synths->mem.param[63]);

	if (synth->flags & SUPPRESS)
		return;

	if (synth == 0)
		return;

	if ((index >= DEVICE_COUNT) || (synth->flags & OPERATIONAL == 0))
		return;

	if (pro10App.resources[0].devlocn[index].to == 1.0)
		sendvalue = value * (CONTROLLER_RANGE - 1);
	else
		sendvalue = value;

	if (synth->mem.param[63] == 0)
	{
		synth->mem.param[index] = value;
		sid = synth->sid;
	} else {
		((guiSynth *) synth->lower)->mem.param[index] = value;
		sid = synth->sid2;
	}


	if ((!global.libtest) || (index >= ACTIVE_DEVS))
		synth->dispatch[index].routine(synth,
			global.controlfd,
			sid,
			synth->dispatch[index].controller,
			synth->dispatch[index].operator,
			sendvalue);

#ifdef DEBUG
	else
		printf("dispatch[%x,%i](%i, %i, %i, %i, %i)\n", synth, index,
			global.controlfd, synth->sid,
			synth->dispatch[index].controller,
			synth->dispatch[index].operator,
			sendvalue);
#endif

	return(0);
}

static int
pro10Passthrough(float value)
{
	printf("pro10Passthrough %f\n", value);
}

/*
 * Any location initialisation required to run the callbacks. For bristol, this
 * will connect to the engine, and give it some base parameters.
 * May need to generate some application specific menus.
 * Will also then make specific requests to some of the devices to alter their
 * rendering.
 */
static int
pro10Init(void *reference)
{
	guiSynth *synth = findSynth(global.synths, (int) reference);
	dispatcher *dispatch;
	int i;

	if (synth == 0)
	{
		synth = findSynth(global.synths, (int) 0);
		if (synth == 0)
		{
			printf("cannot init\n");
			return(0);
		}
	}

	synth->connid = (int) reference;

	printf("Initialise the pro10 link to bristol: %x %x\n",
		synth, synth->connid);

	synth->mem.param = (float *) brightonmalloc(DEVICE_COUNT * sizeof(float));
	synth->mem.count = DEVICE_COUNT;
	synth->mem.active = ACTIVE_DEVS;
	synth->dispatch = (dispatcher *)
		brightonmalloc(DEVICE_COUNT * sizeof(dispatcher));
	dispatch = synth->dispatch;

	((guiSynth *) synth->lower) = (guiSynth *) brightonmalloc(sizeof(guiSynth));
	bcopy(synth, ((guiSynth *) synth->lower), sizeof(guiSynth));
	((guiSynth *) synth->lower)->mem.param = (float *) brightonmalloc(DEVICE_COUNT * sizeof(float));
	((guiSynth *) synth->lower)->mem.count = DEVICE_COUNT;
	((guiSynth *) synth->lower)->mem.active = ACTIVE_DEVS;
	((guiSynth *) synth->lower)->dispatch = synth->dispatch;

	/*
	 * We really want to have three connection mechanisms. These should be
	 *	1. Unix named sockets.
	 *	2. UDP sockets.
	 *	3. MIDI pipe.
	 */
	if (!global.libtest)
	{
		synth->synthtype = BRISTOL_PROPHET;
		if ((synth->sid = initConnection(&global, synth)) < 0)
			exit(0);
printf("UM CID: %i\n", synth->sid);

		manual.flags |= BRISTOL_CONN_FORCE|BRIGHTON_NOENGINE;
		synth->midichannel++;
		if ((synth->sid2 = initConnection(&manual, synth)) < 0)
			exit(0);
printf("LM CID: %i\n", synth->sid2);
		synth->midichannel--;
	} else {
		synth->sid = 5;
		synth->sid2 = 10;
	}

	for (i = 0; i < DEVICE_COUNT; i++)
	{
		synth->dispatch[i].routine = pro10MidiSendMsg;
	}

	dispatch[FIRST_DEV].controller = 126;
	dispatch[FIRST_DEV].operator = 6;
	dispatch[FIRST_DEV + 1].controller = 126;
	dispatch[FIRST_DEV + 1].operator = 7;
	dispatch[FIRST_DEV + 2].controller = 126;
	dispatch[FIRST_DEV + 2].operator = 8;
	dispatch[FIRST_DEV + 3].controller = 126;
	dispatch[FIRST_DEV + 3].operator = 9;
	dispatch[FIRST_DEV + 4].controller = 126;
	dispatch[FIRST_DEV + 4].operator = 10;

	dispatch[FIRST_DEV + 5].controller = 2;
	dispatch[FIRST_DEV + 5].operator = 2;
	dispatch[FIRST_DEV + 6].controller = 2;
	dispatch[FIRST_DEV + 6].operator = 4;
	dispatch[FIRST_DEV + 7].controller = 2;
	dispatch[FIRST_DEV + 7].operator = 5;
	dispatch[FIRST_DEV + 8].controller = 2;
	dispatch[FIRST_DEV + 8].operator = 6;

	dispatch[FIRST_DEV + 9].controller = 126;
	dispatch[FIRST_DEV + 9].operator = 11;
	dispatch[FIRST_DEV + 10].controller = 126;
	dispatch[FIRST_DEV + 10].operator = 12;
	dispatch[FIRST_DEV + 11].controller = 126;
	dispatch[FIRST_DEV + 11].operator = 13;
	dispatch[FIRST_DEV + 12].controller = 126;
	dispatch[FIRST_DEV + 12].operator = 14;
	dispatch[FIRST_DEV + 13].controller = 126;
	dispatch[FIRST_DEV + 13].operator = 15;
	dispatch[FIRST_DEV + 14].controller = 126;
	dispatch[FIRST_DEV + 14].operator = 16;

	dispatch[FIRST_DEV + 15].controller = 0;
	dispatch[FIRST_DEV + 15].operator = 1;
	dispatch[FIRST_DEV + 16].controller = 0;
	dispatch[FIRST_DEV + 16].operator = 4;
	dispatch[FIRST_DEV + 17].controller = 0;
	dispatch[FIRST_DEV + 17].operator = 6;
	dispatch[FIRST_DEV + 18].controller = 0;
	dispatch[FIRST_DEV + 18].operator = 0;
	dispatch[FIRST_DEV + 19].controller = 0;
	dispatch[FIRST_DEV + 19].operator = 7;

	dispatch[FIRST_DEV + 20].controller = 1;
	dispatch[FIRST_DEV + 20].operator = 1;
	dispatch[FIRST_DEV + 21].controller = 1;
	dispatch[FIRST_DEV + 21].operator = 2;
	dispatch[FIRST_DEV + 22].controller = 1;
	dispatch[FIRST_DEV + 22].operator = 4;
	dispatch[FIRST_DEV + 23].controller = 1;
	dispatch[FIRST_DEV + 23].operator = 5;
	dispatch[FIRST_DEV + 24].controller = 1;
	dispatch[FIRST_DEV + 24].operator = 6;
	dispatch[FIRST_DEV + 25].controller = 1;
	dispatch[FIRST_DEV + 25].operator = 0;
	dispatch[FIRST_DEV + 26].controller = 126;
	dispatch[FIRST_DEV + 26].operator = 18;
	dispatch[FIRST_DEV + 27].controller = 126;
	dispatch[FIRST_DEV + 27].operator = 19;

	dispatch[FIRST_DEV + 28].controller = 126;
	dispatch[FIRST_DEV + 28].operator = 0;
	dispatch[FIRST_DEV + 29].controller = 126;
	dispatch[FIRST_DEV + 29].operator = 1;

	dispatch[FIRST_DEV + 30].controller = 126;
	dispatch[FIRST_DEV + 30].operator = 20;
	dispatch[FIRST_DEV + 31].controller = 126;
	dispatch[FIRST_DEV + 31].operator = 21;
	dispatch[FIRST_DEV + 32].controller = 126;
	dispatch[FIRST_DEV + 32].operator = 22;

	dispatch[FIRST_DEV + 33].controller = 4;
	dispatch[FIRST_DEV + 33].operator = 0;
	dispatch[FIRST_DEV + 34].controller = 4;
	dispatch[FIRST_DEV + 34].operator = 1;
	dispatch[FIRST_DEV + 35].controller = 126;
	dispatch[FIRST_DEV + 35].operator = 23;
	dispatch[FIRST_DEV + 36].controller = 4;
	dispatch[FIRST_DEV + 36].operator = 3;
	dispatch[FIRST_DEV + 37].controller = 3;
	dispatch[FIRST_DEV + 37].operator = 0;
	dispatch[FIRST_DEV + 38].controller = 3;
	dispatch[FIRST_DEV + 38].operator = 1;
	dispatch[FIRST_DEV + 39].controller = 3;
	dispatch[FIRST_DEV + 39].operator = 2;
	dispatch[FIRST_DEV + 40].controller = 3;
	dispatch[FIRST_DEV + 40].operator = 3;

	dispatch[FIRST_DEV + 41].controller = 5;
	dispatch[FIRST_DEV + 41].operator = 0;
	dispatch[FIRST_DEV + 42].controller = 5;
	dispatch[FIRST_DEV + 42].operator = 1;
	dispatch[FIRST_DEV + 43].controller = 5;
	dispatch[FIRST_DEV + 43].operator = 2;
	dispatch[FIRST_DEV + 44].controller = 5;
	dispatch[FIRST_DEV + 44].operator = 3;

	dispatch[FIRST_DEV + 45].controller = 126;
	dispatch[FIRST_DEV + 45].operator = 2;
	dispatch[FIRST_DEV + 45].routine = (synthRoutine) dualTune;
	dispatch[FIRST_DEV + 46].controller = 5;
	dispatch[FIRST_DEV + 46].operator = 4;

	dispatch[FIRST_DEV + 47].controller = 1;
	dispatch[FIRST_DEV + 47].operator = 1;
	dispatch[FIRST_DEV + 47].routine = (synthRoutine) midiRelease;
	dispatch[FIRST_DEV + 48].controller = 126;
	dispatch[FIRST_DEV + 48].operator = 3;
	dispatch[FIRST_DEV + 48].routine = (synthRoutine) multiA440;
	dispatch[FIRST_DEV + 49].controller = 1;
	dispatch[FIRST_DEV + 49].operator = 1;
	dispatch[FIRST_DEV + 49].routine = (synthRoutine) multiTune;

	dispatch[FIRST_DEV + 50].operator = 1;
	dispatch[FIRST_DEV + 51].operator = 2;
	dispatch[FIRST_DEV + 52].operator = 3;
	dispatch[FIRST_DEV + 53].operator = 4;
	dispatch[FIRST_DEV + 54].operator = 5;
	dispatch[FIRST_DEV + 55].operator = 6;
	dispatch[FIRST_DEV + 56].operator = 7;
	dispatch[FIRST_DEV + 57].operator = 8;

	dispatch[FIRST_DEV + 58].controller = 1;
	dispatch[FIRST_DEV + 59].controller = 2;
	dispatch[FIRST_DEV + 60].controller = 3;

	dispatch[FIRST_DEV + 50].routine =
		dispatch[FIRST_DEV + 51].routine =
		dispatch[FIRST_DEV + 52].routine =
		dispatch[FIRST_DEV + 53].routine =
		dispatch[FIRST_DEV + 54].routine =
		dispatch[FIRST_DEV + 55].routine =
		dispatch[FIRST_DEV + 56].routine =
		dispatch[FIRST_DEV + 57].routine =
		dispatch[FIRST_DEV + 58].routine =
		dispatch[FIRST_DEV + 59].routine =
		dispatch[FIRST_DEV + 60].routine
			= (synthRoutine) pro10Memory;

	dispatch[FIRST_DEV + 61].routine = dispatch[FIRST_DEV + 62].routine =
		(synthRoutine) pro10Midi;
	dispatch[FIRST_DEV + 61].controller = 1;
	dispatch[FIRST_DEV + 62].controller = 2;

	dispatch[FIRST_DEV + 63].routine = (synthRoutine) pro10PanelSwitch;

	dispatch[RADIOSET_1].other1 = -1;

	/* Tune osc-1 */
	bristolMidiSendMsg(global.controlfd, synth->sid, 0, 2, 8192);
	bristolMidiSendMsg(global.controlfd, synth->sid2, 0, 2, 8192);
	/* Gain on filter contour */
	bristolMidiSendMsg(global.controlfd, synth->sid, 3, 4,
		CONTROLLER_RANGE - 1);
	bristolMidiSendMsg(global.controlfd, synth->sid2, 3, 4,
		CONTROLLER_RANGE - 1);
}

memForce(guiSynth *synth)
{
	brightonEvent event;
	int i;

	synth->flags |= SUPPRESS;

	event.type = BRIGHTON_FLOAT;
	event.value = 1;

	for (i = 0; i < ACTIVE_DEVS; i++)
	{
		brightonParamChange((void *) synth->connid, 0, i, &event);
	}
	event.value = 0;
	for (i = 0; i < ACTIVE_DEVS; i++)
	{
		brightonParamChange((void *) synth->connid, 0, i, &event);
	}

	synth->flags &= ~SUPPRESS;
}

/*
 * This will be called to make any routine specific parameters available.
 */
static int
pro10Configure(int cid)
{
	guiSynth *synth = findSynth(global.synths, cid);
	brightonEvent event;

	if (synth == 0)
	{
		printf("problems going operational\n");
		return(-1);
	}

	if (synth->flags & OPERATIONAL)
		return;

printf("	going operational: %x, %x\n", synth, cid);

	synth->flags |= OPERATIONAL;
	synth->transpose = 36;

	synth->mem.param[63] = 0;

	synth->bank = 10;
	synth->location = 2;
	memForce(synth);
	loadMemory(synth, "prophet", 0, 12, synth->mem.active, FIRST_DEV, 0);

	synth->mem.param[63] = 1;

	((guiSynth *) synth->lower)->resources = synth->resources;
	((guiSynth *) synth->lower)->connid = synth->connid;
	((guiSynth *) synth->lower)->bank = 10;
	((guiSynth *) synth->lower)->location = 1;
	((guiSynth *) synth->lower)->panel = 0;
	((guiSynth *) synth->lower)->mem.active = ACTIVE_DEVS;
	((guiSynth *) synth->lower)->transpose = 36;

	memForce(((guiSynth *) synth->lower));

	loadMemory(((guiSynth *) synth->lower), "prophet", 0, 11, synth->mem.active, FIRST_DEV, 0);

	synth->mem.param[63] = 0;

	/*
	 * Hm. This is a hack for a few bits of bad rendering of a keyboard. Only
	 * occurs on first paint, so we suppress the first paint, and then request
	 * an expose here.
	 */
	event.type = BRIGHTON_EXPOSE;
	event.intvalue = 1;
	brightonParamChange((void *) synth->connid, KEY_PANEL, -1, &event);
	event.type = BRIGHTON_EXPOSE;
	event.intvalue = 1;
	brightonParamChange((void *) synth->connid, KEY_PANEL + 2, -1, &event);
}

