/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "FileJoiner.h"
#include "Events.h"
#include "Shared.h"
#include "PleaseWait.h"
#include <qfile.h>
#include <qfileinfo.h>
#include <qmessagebox.h>
#include <qevent.h>
#include <qapplication.h>
using namespace std;


/*------- local constants:
-------------------------------------------------------------------*/
const QString FileJoiner::NotWritableDir = QT_TR_NOOP( "You can't wite to this directory: %1.\nCheck permission access." );
const QString FileJoiner::Caption        = QT_TR_NOOP( "Files joining" );
const QString FileJoiner::CanOverwarite  = QT_TR_NOOP( "Destination file:%1\nalready exist. Would you like overwrite?" );


//*******************************************************************
// FileJoiner                                            CONSTRUCTOR
//*******************************************************************
FileJoiner::FileJoiner( const ViewTable::SelectedItems& in_items,
                        const QString& in_dst_dir,
                        const QString& in_dst_fname )
: QObject      ()
, d_items      ( in_items )
, d_dst_dir    ( in_dst_dir )
, d_dst_fname  ( in_dst_fname )
, d_please_wait( 0 )
{}
// end of FileJoiner

//*******************************************************************
// ~FileJoiner                                            DESTRUCTOR
//*******************************************************************
FileJoiner::~FileJoiner()
{
    if( d_please_wait ) {
        delete d_please_wait;
        d_please_wait = 0;
    }
}
// end of ~FileJoiner

//*******************************************************************
// run                                                        PUBLIC
//-------------------------------------------------------------------
// Funkcja wykonawcza.
//*******************************************************************
void FileJoiner::run()
{
    const QFileInfo fi( d_dst_dir );
    if( FALSE == fi.isWritable() ) {
       QMessageBox::warning( 0, Shared::ProgramName, tr(NotWritableDir).arg( d_dst_dir ));
    }
    else {
        QString fpath = d_dst_dir;
        if( fpath != "/" ) fpath += "/";
        fpath += d_dst_fname;
        
        if( can_write( fpath ) ) {
            d_please_wait = new PleaseWait( 0 );
            if( d_please_wait ) {
                d_please_wait->caption( tr(Caption) );
                connect( d_please_wait, SIGNAL( break_work() ), this, SLOT( break_slot() ));
                d_please_wait->show();

                QFile out( fpath );
                if( out.open( IO_WriteOnly | IO_Truncate ) ) {
                    Shared::d_break = FALSE;
                    QTextStream dst_stream( &out );
                    ViewTable::SelectedItems::const_iterator it = d_items.begin();
                    while( !Shared::d_break && ( it != d_items.end() )) {
                        if( (*it)->is_readable() ) {
                            join( (*it)->path(), dst_stream );
                        }
                        ++it;
                    }
                    out.close();
                }
                delete d_please_wait;
                d_please_wait = 0;
            }
        }
    }
}
// end of run

//*******************************************************************
// join                                                      PRIVATE
//-------------------------------------------------------------------
// Dolaczenie kolejnego pliku.
// Przed operacja wysylamy informacje (jako event) do dialogu
// 'PleaseWait'.
//*******************************************************************
void FileJoiner::join( const QString& in_src_fpath, QTextStream& in_dst_stream )
{
    BscEvent* event = new BscEvent( JoinFileEvent );
    if( event ) {
        event->m_message = in_src_fpath;
        QApplication::postEvent( d_please_wait, event );
        Shared::idle();
        
        QFile src( in_src_fpath );
        if( src.open( IO_ReadOnly ) ) {
            QTextStream src_stream( &src );
            QString line;
            while ( !Shared::d_break && !src_stream.atEnd() ) {
                line = src_stream.readLine();
                in_dst_stream << line << "\n";
            }
            src.close();
        }
    }
}
// end of join

//*******************************************************************
// can_write                                                 PRIVATE
//-------------------------------------------------------------------
// Sprawdzenie czy plik wynikowy juz instnieje.
// Jesli istnieje pytamy uzytkownika o zgode na nadpisanie pliku.
//*******************************************************************
bool FileJoiner::can_write( const QString& in_fpath )
{
    bool retval = TRUE;
    
    const QFile file( in_fpath );
    if( file.exists() ) {
        const int answer = QMessageBox::question( 0,
                            tr(Caption),
                            tr(CanOverwarite).arg(in_fpath),
                            Shared::YesBtnLabel,
                            Shared::NoBtnLabel,
                            QString::null,
                            1, 1);
        retval = ( 0 == answer );
    }
    
    return retval;
}
// end of can_write

//*******************************************************************
// break_slot                                           PRIVATE slot
//-------------------------------------------------------------------
// Obsluga zdarzenia wygenerowanego przez nacisniecie klawisza
// 'break' w dialogu 'PleaseWait'.
//*******************************************************************
void FileJoiner::break_slot()
{
    Shared::d_break = TRUE;
}
// end of break_slot
