/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/
 
/*------- include files:
-------------------------------------------------------------------*/
#include "SystemCall.h"
#include <qprocess.h>
#include <qtimer.h>
using namespace std;

//*******************************************************************
// SystemCall                                            CONSTRUCTOR
//********************************************************************
SystemCall::SystemCall( QWidget* const in_parent )
: QDialog( in_parent )
, d_process( 0 )
, d_mode( OFF_LINE )
, d_stdout_buffer( "" )
, d_stderr_buffer( "" )
{
	setHidden( TRUE );
}
// end of SystemCall

//*******************************************************************
// ~SystemCall                                            DESTRUCTOR
//*******************************************************************
SystemCall::~SystemCall()
{
	if( d_process ) {
		d_process->kill();
		delproc();
	}
}
// end of ~SystemCall

//*******************************************************************
// run                                                        PUBLIC
//*******************************************************************
bool SystemCall::run( const vector<QString>& in_parameters, const Mode in_mode )
{
	bool retval = FALSE;
	
	d_mode = in_mode;
	d_cout_strings.clear();
	d_cerr_strings.clear();

	d_process = new QProcess( this );
	if( d_process ) {
		connect( d_process, SIGNAL( readyReadStdout() ), this, SLOT( read_stdout()    ));
		connect( d_process, SIGNAL( readyReadStderr() ), this, SLOT( read_stderr()    ));
		connect( d_process, SIGNAL( processExited()   ), this, SLOT( process_exited() ));
		
		d_process->clearArguments();
		
		vector<QString>::const_iterator it = in_parameters.begin();
		while( it != in_parameters.end() ) {
			d_process->addArgument( *it );
			++it;
		}

		retval = d_process->start();
	}
	
	return retval;
}
// end of run

//*******************************************************************
// keyPressEvent                                   PRIVATE inherited
//*******************************************************************
void SystemCall::keyPressEvent( QKeyEvent* e )
{
	if( Qt::Key_Escape == e->key() ) {
		d_process->tryTerminate();
		QTimer::singleShot( 500, d_process, SLOT( kill() ) );
	}
	QDialog::keyPressEvent( e );
}
// end of keyPressEvent

//*******************************************************************
// result                                                     PUBLIC
//-------------------------------------------------------------------
// Za pomoca tej funkcji mozna pobrac wyniki operacji.
// Wyniki sa dwojakiego rodzaju: cout i cerr.
// Funkcje nalezy uzywac dopiero po zakonczeniu calej operacji.
//*******************************************************************
void SystemCall::result( vector<QString>& out_cout_strings, vector<QString>& out_cerr_strings )
{
	out_cout_strings = d_cout_strings;
	out_cerr_strings = d_cerr_strings;
}
// end of result

//*******************************************************************
// read_stdout                                          PRIVATE slot
//*******************************************************************
void SystemCall::read_stdout()
{
	d_stdout_buffer += d_process->readStdout();
	
	if( d_stdout_buffer.endsWith( "\n" ) ) {
		if( ON_LINE == d_mode ) {
			emit stdout_line( d_stdout_buffer );
		}
		else {
			d_cout_strings.push_back( d_stdout_buffer );
		}
		d_stdout_buffer = "";
	}
}
// end of read_stdout

//*******************************************************************
// read_stderr                                          PRIVATE slot
//*******************************************************************
void SystemCall::read_stderr()
{
	d_stderr_buffer += d_process->readStderr();
	
	if( d_stderr_buffer.endsWith( "\n" ) ) {
		if( ON_LINE == d_mode ) {
			emit stderr_line( d_stderr_buffer );
		}
		else {
			d_cerr_strings.push_back( d_stderr_buffer );
		}
		d_stderr_buffer = "";
	}
}
// end of read_stderr

//*******************************************************************
// process_exited                                       PRIVATE slot
//*******************************************************************
void SystemCall::process_exited()
{
	const int status = d_process->exitStatus();
	delproc();
	emit finished( status );
}
// end of process_exited

//*******************************************************************
// delproc                                                   PRIVATE
//*******************************************************************
void SystemCall::delproc()
{
	disconnect( d_process, SIGNAL( readyReadStdout() ), this, SLOT( read_stdout()    ));
	disconnect( d_process, SIGNAL( readyReadStderr() ), this, SLOT( read_stderr()    ));
	disconnect( d_process, SIGNAL( processExited()   ), this, SLOT( process_exited() ));
	delete d_process;
	d_process = 0;
}
// end of delproc
