# Copyright (C) 2005, 2006, 2007 by Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""Functionality for controlling a Patch Queue Manager (pqm).
"""

from bzrlib.commands import Command, register_command
from bzrlib.option import Option

from bzrlib.bzrdir import BzrDir


version_info = (1, 0, 0, 'dev', 0)

if version_info[3] == 'final':
    version_string = '%d.%d.%d' % version_info[:3]
else:
    version_string = '%d.%d.%d.%s.%d' % version_info
__version__ = version_string


class cmd_pqm_submit(Command):
    """Submit the parent tree to the pqm.

    This acts like:
        $ echo "star-merge $PARENT $TARGET"
            | gpg --cl
            | mail pqm@somewhere -s "merge text"

    But it pays attention to who the local committer is
    (using their e-mail address), and uses the local
    gpg signing configuration. (As well as target pqm
    settings, etc.)

    The reason we use 'parent' instead of the local branch
    is that most likely the local branch is not a public
    branch. And the branch must be available to the pqm.

    This can be configured at the branch level using ~/.bazaar/locations.conf.
    Here is an example:
        [/home/emurphy/repo]
        pqm_email = PQM <pqm@example.com>
        pqm_user_email = User Name <user@example.com>
        submit_branch = http://code.example.com/code/project/devel
        # Set public_branch appropriately for all branches in repository:
        public_branch = http://code.example.com/code/emurphy/project
        public_branch:policy = appendpath
        [/home/emurphy/repo/branch]
        # Override public_branch for this repository:
        public_branch = http://alternate.host.example.com/other/public/branch

        smtp_server = host:port
        smtp_username =
        smtp_password =

    If you don't specify the smtp server, the message will be sent via localhost.
    """

    takes_args = ['location?']
    takes_options = [
        Option('message',
               help='Message to use on merge to pqm.  '
                    'Currently must be a single line because of pqm limits.',
               short_name='m',
               type=unicode),
        Option('dry-run', help='Print request instead of sending.'),
        Option('public-location', type=str,
               help='Use this url as the public location to the pqm.'),
        Option('submit-branch', type=str,
               help='Use this url as the target submission branch.'),
        ]

    def run(self, location=None, message=None, public_location=None,
            dry_run=False, submit_branch=None):
        if __name__ != 'bzrlib.plugins.pqm':
            from bzrlib import trace
            trace.warning('The bzr-pqm plugin needs to be called'
                          ' "bzrlib.plugins.pqm" not "%s"\n'
                          'Please rename the plugin.',
                          __name__)
            return 1
        from bzrlib.plugins.pqm.pqm_submit import submit

        if location is None:
            location = '.'
        tree, b, relpath = BzrDir.open_containing_tree_or_branch(location)
        if relpath and not tree:
            from bzrlib import errors
            raise errors.BzrCommandError('No working tree was found, but we'
                                          ' were not given the exact path to'
                                          ' the branch.\n'
                                          ' We found the branch at: %s'
                                          % (b.base,))
        submit(b, message=message, dry_run=dry_run,
               public_location=public_location,
               submit_location=submit_branch,
               tree=tree)


register_command(cmd_pqm_submit)


def test_suite():
    from bzrlib.tests import TestLoader
    import test_pqm_submit

    loader = TestLoader()
    return loader.loadTestsFromModule(test_pqm_submit)
