/*
 * Danpei -- a GTK+ based Image Viewer
 * Copyright (C) 2001-2003 Shinji Moiino
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
/* file_menu.c */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <gdk/gdkx.h>

#include "config.h"
#include "dialog.h"
#include "edit_menu.h"
#include "file_menu.h"
#include "image_cache.h"
#include "intl.h"
#include "main.h"
#include "option_menu.h"
#include "thumbnail_table.h"
#include "version.h"
#include "viewer.h"

/* Static function declarations. */
static void file_menu_print_start_dialog      (TopLevel*);

static gint file_menu_print_dialog_time_count (gpointer );

/* Function definitions. */
/*
 * @file_menu_cb_simple_viewer
 *
 *  Start simple viewer.
 *
 */
void file_menu_cb_simple_viewer(GtkWidget *widget,
                                gpointer  data   ,
                                guint     action   ) {
  viewer_start((TopLevel*)data, ((TopLevel*)data)->op_thumb);
  ((TopLevel*)data)->op_thumb = NULL;

  return;
}

/*
 * @file_menu_cb_view
 *
 *  Start the viewer.
 *  -- view all selected thumbnails.
 *
 */
void file_menu_cb_view(GtkWidget *widget,
                       gpointer  data   ,
                       guint     action   ) {
  TopLevel  *tp;
  Thumbnail *thumb;
  gchar     *com, *filename, *tmp_files, *arg_files;

  /* Initialize the local variables. */
  tp    = (TopLevel*)data;
  thumb = NULL;
  com   = filename = tmp_files = arg_files = NULL;

  if (tp->from_2button == TRUE) {
    thumb = tp->op_thumb;
  }
  else {
    thumb = tp->thumbnail_table.top_thumbnail;
  }

  while (thumb != NULL) {
    /* When called by double click, excuted only once. */
    if ((tp->from_2button == TRUE) && (thumb != tp->op_thumb)) { 
      thumb = NULL;
      break;
     }

    if ((thumb->selected == TRUE) && (thumb->disabled == FALSE)) {
      filename = (gchar*)malloc(sizeof(gchar) *
                                (strlen(thumb->path)     +
                                 strlen("/")             +
                                 strlen(thumb->filename) + 1));
      if (filename == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        file_menu.c: error -- 01.\n");
        gtk_exit(-1);
      }
      else {
        sprintf(filename, "%s/%s", thumb->path, thumb->filename);
      }

      if (access(filename, R_OK) == 0) {
        if ((tp->from_2button == TRUE                         ) && 
            ((tp->app_option).viewer.default_external == FALSE)     ) {
          viewer_start(tp, thumb);
        }
        else {
          if ((tp->app_option).is_viewer_multi_process == FALSE) {
            if (arg_files == NULL) {
              arg_files = g_strconcat("'", filename, "'", NULL);
            }
            else {
              tmp_files = g_strdup(arg_files);
              if (tmp_files == NULL) {
                /* Out of memory. */
                fprintf(stderr, "danpei: Out of memory.\n");
                fprintf(stderr, "        file_menu.c: error -- 02.\n");
                gtk_exit(-1);
              }
              arg_files = g_strconcat(tmp_files, " '", filename, "'",  NULL);
              g_free(tmp_files);
            }
            if (arg_files == NULL) {
              /* Out of memory. */
              fprintf(stderr, "danpei: Out of memory.\n");
              fprintf(stderr, "        file_menu.c: error -- 03.\n");
              gtk_exit(-1);
            }
          }
          else {
            com = (gchar*)malloc(sizeof(gchar) *
                                 (strlen(tp->app_option.path.viewer)  +
                                  strlen(" '")                        +
                                  strlen(filename)                    +
                                  strlen("' &")                       + 1));
            if (com == NULL) {
              /* Out of memory. */
              fprintf(stderr, "danpei: Out of memory.\n");
              fprintf(stderr, "        file_menu.c: error -- 04.\n");
              gtk_exit(-1);
            }
            else {
              sprintf(com, "%s '%s' &",
                      tp->app_option.path.viewer, filename);
              system(com);
              free(com);
            }
          }
        }
      }
      else {
        /* That file no longer exists or changed it's permission,
         * so make that thumbnail disable and delete from the cache.
         */
        if (tp->app_option.image_cache.cache_on == TRUE) {
          image_cache_delete(&(tp->cache), filename);
        }
        thumbnail_set_sensitive(thumb, FALSE);
        thumbnail_count_selected_thumbnails(tp, -1);
      }
      free(filename);
    }
    thumb = thumb->next;
  }

  if ((arg_files != NULL) &&
      ((tp->app_option).is_viewer_multi_process == FALSE)) {
    com = (gchar*)malloc(sizeof(gchar) *
                         (strlen(tp->app_option.path.viewer)  +
                          strlen(" ")                         +
                          strlen(arg_files)                   +
                          strlen(" &")                        + 1));
    if (com == NULL) {
      /* Out of memory. */
      fprintf(stderr, "danpei: Out of memory.\n");
      fprintf(stderr, "        file_menu.c: error -- 05.\n");
      gtk_exit(-1);
    }
    else {
      sprintf(com, "%s %s &", tp->app_option.path.viewer, arg_files);
      system(com);
      free(com);
      free(arg_files);
    }
  }

  return;
}

/*
 * @file_menu_cb_edit
 *
 *  Start the editor.
 *  -- edit all selected thumbnails.
 *
 */
void file_menu_cb_edit(GtkWidget *widget, 
                       gpointer  data   , 
                       guint     action  ) {
  TopLevel  *tp;
  Thumbnail *thumb;
  gchar     *com, *filename, *tmp_files, *arg_files;

  /* Initialize the local variables. */
  tp    = (TopLevel*)data;
  thumb = tp->thumbnail_table.top_thumbnail;
  com   = filename = tmp_files = arg_files = NULL;

  while (thumb != NULL) {
    if ((thumb->selected == TRUE) && (thumb->disabled == FALSE)) {
      filename = (gchar*)malloc(sizeof(gchar) *
                                (strlen(thumb->path)     +
                                 strlen("/")             +
                                 strlen(thumb->filename) + 1));
      if (filename == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        file_menu.c: error -- 06.\n");
        gtk_exit(-1);
      }
      else {
        sprintf(filename, "%s/%s", thumb->path, thumb->filename);
      }

      if (access(filename, R_OK) == 0) {
        if ((tp->app_option).is_editor_multi_process == FALSE) {
          if (arg_files == NULL) {
            arg_files = g_strconcat("'", filename, "'", NULL);
          }
          else {
            tmp_files = g_strdup(arg_files);
            if (tmp_files == NULL) {
              /* Out of memory. */
              fprintf(stderr, "danpei: Out of memory.\n");
              fprintf(stderr, "        file_menu.c: error -- 07.\n");
              gtk_exit(-1);
            }
            arg_files = g_strconcat(tmp_files, " '", filename, "'",  NULL);
            g_free(tmp_files);
          }
          if (arg_files == NULL) {
            /* Out of memory. */
            fprintf(stderr, "danpei: Out of memory.\n");
            fprintf(stderr, "        file_menu.c: error -- 08.\n");
            gtk_exit(-1);
          }
        }
        else {
          com = (gchar*)malloc(sizeof(gchar) *
                               (strlen(tp->app_option.path.editor)  +
                                strlen(" '")                        +
                                strlen(filename)                    +
                                strlen("' &")                       + 1));
          if (com == NULL) {
            /* Out of memory. */
            fprintf(stderr, "danpei: Out of memory.\n");
            fprintf(stderr, "        file_menu.c: error -- 09.\n");
            gtk_exit(-1);
          }
          else {
            sprintf(com, "%s '%s' &",
                    tp->app_option.path.editor, filename);
            system(com);
            free(com);
          }
        }
      }
      else {
        /* That file no longer exists or changed it's permission,
         * so make that thumbnail disable and delete from the cache.
         */
        if (tp->app_option.image_cache.cache_on == TRUE) {
          image_cache_delete(&(tp->cache), filename);
        }
        thumbnail_set_sensitive(thumb, FALSE);
        thumbnail_count_selected_thumbnails(tp, -1);
      }
      free(filename);
    }
    thumb = thumb->next;
  }

  if ((arg_files != NULL) &&
      ((tp->app_option).is_editor_multi_process == FALSE)) {
    com = (gchar*)malloc(sizeof(gchar) *
                         (strlen(tp->app_option.path.editor)  +
                          strlen(" ")                         +
                          strlen(arg_files)                   +
                          strlen(" &")                        + 1));
    if (com == NULL) {
      /* Out of memory. */
      fprintf(stderr, "danpei: Out of memory.\n");
      fprintf(stderr, "        file_menu.c: error -- 10.\n");
      gtk_exit(-1);
    }
    else {
      sprintf(com, "%s %s &", tp->app_option.path.editor, arg_files);
      system(com);
      free(com);
      free(arg_files);
    }
  }

  return;
}

/*
 * @file_menu_cb_prev
 *
 *  Display the previous thumbnail page.
 *
 */
void file_menu_cb_prev(GtkWidget *widget, gpointer data, guint action) {
  TopLevel *tp;

  tp = (TopLevel*)data;

  if (tp->thumbnail_table.start_pos < 1) { return; }

  thumbnail_table_display_prev_page(tp);

  return;
}

/*
 * @file_menu_cb_next
 *
 *  Display the next thumbnail page.
 *
 */
void file_menu_cb_next(GtkWidget *widget, gpointer data, guint action) {
  TopLevel *tp;

  tp = (TopLevel*)data;

  if (tp->thumbnail_table.end_pos + 2 > tp->thumbnail_table.file_num)
     return; 

  thumbnail_table_display_next_page(tp);

  return;
}

/*
 * @file_menu_cb_reload_selected
 *
 *  Refresh the current thumbnail table.
 *  Same as file_menu_cb_reload, except reload only selected thumbnails.
 *
 */
void file_menu_cb_reload_selected(GtkWidget *widget,
                                  gpointer  data   ,
                                  guint     action   ) {
  TopLevel  *tp;

  tp = (TopLevel*)data;

  thumbnail_table_reload_all(tp, FALSE);

  return;
}

/*
 * @file_menu_cb_reload
 *
 *  Refresh the current thumbnail table.
 *  Re-create the ThumbnailTable structure object and cache files.
 *
 */
void file_menu_cb_reload(GtkWidget *widget,
                         gpointer  data   ,
                         guint     action   ) {
  TopLevel  *tp;

  tp = (TopLevel*)data;

  thumbnail_table_reload_all(tp, TRUE);

  return;
}

/*
 * @file_menu_cb_refresh
 *
 *  Refresh the current thumbnail table.
 *  Re-create the ThumbnailTable structure object.
 *
 */
void file_menu_cb_refresh(GtkWidget *widget,
                          gpointer  data   ,
                          guint     action   ) {
  TopLevel  *tp;

  tp = (TopLevel*)data;

  thumbnail_table_refresh(tp);

  return;
}

/*
 * @file_menu_cb_goto_home
 *                                
 */
void file_menu_cb_goto_home(GtkWidget *widget,
                            gpointer  data   ,
                            guint     action   ) {
  TopLevel *tp;

  tp = (TopLevel*)data;

  dirtree_directory_set_selected(tp, getenv("HOME"), TRUE);
  dirtree_directory_selected(tp, getenv("HOME"));

  return;
}

/*
 * @file_menu_cb_print
 *
 *  Start the print command.
 *  -- all selected thumbnails will be printed .
 *
 */
void file_menu_cb_print(GtkWidget *widget,
                       gpointer  data    ,
                       guint     action    ) {
  TopLevel  *tp;
  Thumbnail *thumb;
  gchar     *com, *filename;

  /* Initialize the local variables. */
  tp    = (TopLevel*)data;
  thumb = tp->thumbnail_table.top_thumbnail;
  com = filename = NULL;

  while (thumb != NULL) {
    if ((thumb->selected == TRUE) && (thumb->disabled == FALSE)) {
      filename = (gchar*)malloc(sizeof(gchar) *
                                (strlen(thumb->path)     +
                                 strlen("/")             +
                                 strlen(thumb->filename) + 1));
      if (filename == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        file_menu.c: error -- 11.\n");
        gtk_exit(-1);
      }
      else {
        sprintf(filename, "%s/%s", thumb->path, thumb->filename);
      }

      if (access(filename, R_OK) == 0) {
        com = (gchar*)malloc(sizeof(gchar) *
                             (strlen("cat '")                     +
                              strlen(filename)                    +
                              strlen("' | ")                      +
                              strlen(tp->app_option.path.printer) +
                              strlen(" &")                        + 1));
        if (com == NULL) {
          /* Out of memory. */
          fprintf(stderr, "danpei: Out of memory.\n");
          fprintf(stderr, "        file_menu.c: error -- 12.\n");
          gtk_exit(-1);
        }
        else {
          sprintf(com, "%s '%s' | %s &", 
                  "cat", filename, tp->app_option.path.printer);
          system(com);
          free(com);
        }
      }
      else {
        /* That file no longer exists or changed it's permission,
         * so make that thumbnail disable and delete from the cache.
         */
        if (tp->app_option.image_cache.cache_on == TRUE) {
          image_cache_delete(&(tp->cache), filename);
        }
        thumbnail_set_sensitive(thumb, FALSE);
        thumbnail_count_selected_thumbnails(tp, -1);
      }
      free(filename);
    }
    thumb = thumb->next;
  }

  file_menu_print_start_dialog(tp);

  return;
}

/*
 * @file_menu_cb_remove
 *
 *  remove the current thumbnail table.
 *  Re-create the ThumbnailTable structure object.
 *
 */
void file_menu_cb_remove(GtkWidget *widget,
                         gpointer  data   ,
                         guint     action   ) {
  TopLevel        *tp;
  Thumbnail       *thumb, *thumb_next;
  gchar           *filename;
  AppDialogResult ret;
  gint            count;

  /* Initialize the local variables. */
  tp         = (TopLevel*)data;
  thumb      = tp->thumbnail_table.top_thumbnail;
  thumb_next = NULL;
  filename   = NULL;
  count      = 0;
  ret        = APP_RET_CANCEL;

  tp->all_yes = FALSE;
  while (thumb != NULL) {
    thumb_next = thumb->next;
    if ((thumb->selected == TRUE) && (thumb->disabled == FALSE)) {
      filename = (gchar*)malloc(sizeof(gchar) *
                                (strlen(thumb->path)     +
                                 strlen("/")             +
                                 strlen(thumb->filename) + 1));
      if (filename == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        file_menu.c: error -- 13.\n");
        gtk_exit(-1);
      }
      else {
        sprintf(filename, "%s/%s", thumb->path, thumb->filename);
      }

      if (access(filename, F_OK) == 0) {
        /* Check dialog. */
        if ((count == 0) &&
            (tp->app_option.dialog.remove == TRUE)) {
          ret = dialog_message_dialog_create(FILE_REMOVED, NULL,
                                             APP_OK_CANCEL_DIALOG,
                                             APP_OK_BUTTON   ,
                                             GTK_WINDOW(tp->window));
          if (ret != APP_RET_OK) { 
            free(filename);
            return;
          }
        }
        /* Remove */
        if (remove(filename) == 0) {
          image_cache_delete(&(tp->cache), filename);
          thumbnail_change_link(&(tp->thumbnail_table), thumb); 
          gtk_widget_destroy(thumb->ev_box);
          thumbnail_free_structure(thumb);
          thumbnail_count_selected_thumbnails(tp, -1);
        }
        else {
          switch(errno) {
            case ENOENT:
              dialog_error_dialog_create(FILE_NOT_EXISTS, thumb->filename,
                                         GTK_WINDOW(tp->window));
              if (tp->app_option.image_cache.cache_on == TRUE) {
                image_cache_delete(&(tp->cache), filename);
              }
              thumbnail_set_sensitive(thumb, FALSE);
              thumbnail_count_selected_thumbnails(tp, -1);
              break;
            case EACCES:
            case EPERM :
              dialog_error_dialog_create(PERMISSION_DENIED, thumb->filename,
                                         GTK_WINDOW(tp->window));
              break;
            case EROFS:
              dialog_error_dialog_create(READ_ONLY_FS, thumb->filename,
                                         GTK_WINDOW(tp->window));
              thumbnail_entry_set_sensitive(thumb, FALSE);
              break;
            case ENAMETOOLONG:
              dialog_error_dialog_create(TOO_LONG_FILE_NAME, thumb->filename,
                                         GTK_WINDOW(tp->window));
              break;
            default:
              dialog_error_dialog_create(REMOVE_ERROR, thumb->filename,
                                         GTK_WINDOW(tp->window));
              break;
          }
        }
      }
      else {
        /* That file no longer exists or changed it's permission,
         * so make that thumbnail disable and delete from the cache.
         */
        if (tp->app_option.image_cache.cache_on == TRUE) {
          image_cache_delete(&(tp->cache), filename);
        }
        thumbnail_set_sensitive(thumb, FALSE);
        thumbnail_count_selected_thumbnails(tp, -1);
      }
      free(filename);
      count++;
    }
    thumb = thumb_next;
  }

  if (tp->app_option.auto_refresh == TRUE) {
    thumbnail_table_refresh(tp);
  }

  return;
}

/*
 * @file_menu_cb_quit
 *
 *  Quit the 'Danpei'.
 *
 */
void file_menu_cb_quit(GtkWidget *widget, 
                       gpointer  data   , 
                       guint     action   ) {
  toplevel_cb_destroy(NULL, data);

  return;
}

/* Static function definitions. */
/*
 * @file_menu_print_start_dialog
 *
 *  Display the dialog, "Print command started.".
 *  -- This dialog is automatically removed after passed about 1 second.
 *
 */
static void file_menu_print_start_dialog(TopLevel *tp) {
  GtkWidget *dialog;
  GtkWidget *label;
  gint      tm_tag;

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), _("MESSAGE"));
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_transient_for(GTK_WINDOW(dialog), 
                               GTK_WINDOW(tp->window));

  label = gtk_label_new("        ");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
                     label, TRUE, TRUE, 0);
  gtk_widget_show(label);
  label = gtk_label_new(_("  Print command started.  "));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
                     label, TRUE, TRUE, 0);
  gtk_widget_show(label);
  label = gtk_label_new("        ");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
                     label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  gtk_widget_show(dialog);

  tm_tag = gtk_timeout_add(500, 
                           (GtkFunction)file_menu_print_dialog_time_count, 
                           NULL);

  gtk_main();

  gtk_timeout_remove(tm_tag);

  gtk_widget_destroy(dialog);

  return;
}

/*
 * @file_menu_print_dialog_time_count
 *
 *  Count up to about 1 second.
 *  -- a function for file_menu_print_start_dialog().
 *
 */
static gint file_menu_print_dialog_time_count(gpointer data) {
  static int counter = 0;

  counter++;
  if(counter > 1) {
    counter = 0;
    gtk_main_quit();
    return 0;
  }
  else {
    return 1;
  }

  return 1;
}


