/***************************************************************************
 *   Copyright (C) 2005 by Adam Treat                                      *
 *   treat@kde.org                                                         *
 *                                                                         *
 *   Copyright (C) 2004 by Scott Wheeler                                   *
 *   wheeler@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kdebug.h>
#include <kcombobox.h>
#include <klineedit.h>
#include <ktextedit.h>
#include <kpushbutton.h>
#include <klocale.h>

#include <qradiobutton.h>
#include <qvgroupbox.h>
#include <qlabel.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qhbuttongroup.h>
#include <qcheckbox.h>

#include "advancedsearchdialog.h"
#include "searchwidget.h"

AdvancedSearchDialog::AdvancedSearchDialog( DataTableList dataTables,
        const DataTableSearch &defaultSearch,
        QWidget *parent,
        const char *name ) :
        KDialogBase( parent, name, true, i18n( "Advanced Search" ), Ok | Cancel ),
        m_dataTables( dataTables ),
        m_search( defaultSearch )
{
    makeVBoxMainWidget();

    QHBox *box = new QHBox( mainWidget() );
    box->setSpacing( 5 );

    m_saveCheckBox = new QCheckBox( i18n( "Save search as:" ), box );
    connect( m_saveCheckBox, SIGNAL( toggled( bool ) ), SLOT( saveToggled( bool ) ) );

    m_dataTableNameLineEdit = new KLineEdit( "", box );
    m_dataTableNameLineEdit->setEnabled( false );

    QVGroupBox *criteriaGroupBox = new QVGroupBox( i18n( "Search Criteria" ), mainWidget() );
    static_cast<QHBox *>( mainWidget() ) ->setStretchFactor( criteriaGroupBox, 1 );

    QHButtonGroup *group = new QHButtonGroup( criteriaGroupBox );
    m_matchAllButton = new QRadioButton( i18n( "Match all of the following" ), group );
    m_matchAnyButton = new QRadioButton( i18n( "Match any of the following" ), group );

    m_criteria = new QVBox( criteriaGroupBox );
    m_advanced = new QVBox( criteriaGroupBox );

    QWidget *buttons = new QWidget( criteriaGroupBox );
    QBoxLayout *l = new QHBoxLayout( buttons, 0, 5 );

    m_advancedButton = new KPushButton( i18n( "Advanced >>" ), buttons );
    connect( m_advancedButton, SIGNAL( clicked() ), SLOT( custom() ) );
    l->addWidget( m_advancedButton );
    l->addStretch( 1 );

    KPushButton *clearButton = new KPushButton( KStdGuiItem::clear(), buttons );
    connect( clearButton, SIGNAL( clicked() ), SLOT( clear() ) );
    l->addWidget( clearButton );

    m_moreButton = new KPushButton( i18n( "More" ), buttons );
    connect( m_moreButton, SIGNAL( clicked() ), SLOT( more() ) );
    l->addWidget( m_moreButton );

    m_fewerButton = new KPushButton( i18n( "Fewer" ), buttons );
    connect( m_fewerButton, SIGNAL( clicked() ), SLOT( fewer() ) );
    l->addWidget( m_fewerButton );

    m_dataTableNameLineEdit->setFocus();
    if ( m_search.isNull() )
    {
        m_searchLines.append( new SearchLine( m_criteria, m_dataTables ) );
        m_matchAllButton->setChecked( true );

        m_advancedLines.append( new AdvancedLine( m_advanced, m_dataTables ) );
        m_advanced->hide();
    }
    else
    {
        if ( !m_search.name().isEmpty() )
        {
            m_saveCheckBox->setChecked( true );
            m_dataTableNameLineEdit->setText( m_search.name() );
        }

        if ( m_search.searchLevel() == DataTableSearch::Custom )
        {
            DataTableSearch::ComponentList components = m_search.components();
            for ( DataTableSearch::ComponentList::ConstIterator it = components.begin();
                    it != components.end();
                    ++it )
            {
                AdvancedLine *s = new AdvancedLine( m_advanced, m_dataTables );
                s->setSearchComponent( *it );
                m_advancedLines.append( s );
            }
            m_searchLines.append( new SearchLine( m_criteria, m_dataTables ) );
            custom();
        }
        else
        {
            DataTableSearch::ComponentList components = m_search.components();
            for ( DataTableSearch::ComponentList::ConstIterator it = components.begin();
                    it != components.end();
                    ++it )
            {
                SearchLine *s = new SearchLine( m_criteria, m_dataTables );
                s->setSearchComponent( *it );
                s->setPrompt( ( *it ).isPrompt() );
                m_searchLines.append( s );
            }
            if ( m_search.searchMode() == DataTableSearch::MatchAny )
                m_matchAnyButton->setChecked( true );
            else
                m_matchAllButton->setChecked( true );

            m_advancedLines.append( new AdvancedLine( m_advanced, m_dataTables ) );
            m_advanced->hide();
        }
    }

    updateButtons();

    resize( QSize( 640, 0 ).expandedTo( minimumSizeHint() ) );
}

AdvancedSearchDialog::~AdvancedSearchDialog()
{}

AdvancedSearchDialog::Result AdvancedSearchDialog::exec()
{
    Result r;
    r.result = DialogCode( KDialogBase::exec() );
    r.search = m_search;
    r.dataTableName = m_dataTableName;
    return r;
}

void AdvancedSearchDialog::saveToggled( bool toggle )
{
    if ( toggle )
        m_dataTableNameLineEdit->setEnabled( true );
    else
        m_dataTableNameLineEdit->setEnabled( false );
}

void AdvancedSearchDialog::accept()
{
    m_search.clearComponents();

    m_search.setDataTables( m_dataTables );

    if ( !m_advanced->isVisible() )
    {
        QValueListConstIterator<SearchLine *> it = m_searchLines.begin();
        for ( ; it != m_searchLines.end(); ++it )
            m_search.addComponent( ( *it ) ->searchComponent() );

        DataTableSearch::SearchMode m = DataTableSearch::SearchMode( !m_matchAnyButton->isChecked() );
        m_search.setSearchMode( m );
    }
    else
    {
        QValueListConstIterator<AdvancedLine *> it = m_advancedLines.begin();
        for ( ; it != m_advancedLines.end(); ++it )
            m_search.addComponent( ( *it ) ->searchComponent() );

        m_search.setSearchLevel( DataTableSearch::Custom );
    }

    m_search.search();

    m_dataTableName = m_dataTableNameLineEdit->text();

    m_search.setName( m_dataTableNameLineEdit->text() );

    KDialogBase::accept();
}

void AdvancedSearchDialog::clear()
{
    if ( !m_advanced->isVisible() )
    {
        QValueListConstIterator<SearchLine *> it = m_searchLines.begin();
        for ( ; it != m_searchLines.end(); ++it )
            ( *it ) ->clear();
    }
    else
    {
        QValueListConstIterator<AdvancedLine *> it = m_advancedLines.begin();
        for ( ; it != m_advancedLines.end(); ++it )
            ( *it ) ->clear();
    }
}

void AdvancedSearchDialog::more()
{
    if ( !m_advanced->isVisible() )
    {
        SearchLine * searchLine = new SearchLine( m_criteria, m_dataTables );
        m_searchLines.append( searchLine );
        searchLine->show();
    }
    else
    {
        AdvancedLine * advancedLine = new AdvancedLine( m_advanced, m_dataTables );
        m_advancedLines.append( advancedLine );
        advancedLine->show();
    }
    updateButtons();
}

void AdvancedSearchDialog::fewer()
{
    if ( !m_advanced->isVisible() )
    {
        SearchLine * searchLine = m_searchLines.last();
        m_searchLines.remove( searchLine );
        delete searchLine;
    }
    else
    {
        AdvancedLine * advancedLine = m_advancedLines.last();
        m_advancedLines.remove( advancedLine );
        delete advancedLine;
    }
    updateButtons();
}

void AdvancedSearchDialog::custom()
{
    if ( !m_advanced->isVisible() )
    {
        m_advancedButton->setText( i18n( "<< Basic" ) );
        m_criteria->hide();
        m_advanced->show();
    }
    else
    {
        m_advancedButton->setText( i18n( "Advanced >>" ) );
        m_advanced->hide();
        m_criteria->show();
    }
    updateButtons();
}

void AdvancedSearchDialog::updateButtons()
{
    if ( !m_advanced->isVisible() )
    {
        m_moreButton->setEnabled( m_searchLines.count() < 16 );
        m_fewerButton->setEnabled( m_searchLines.count() > 1 );
    }
    else
    {
        m_moreButton->setEnabled( m_advancedLines.count() < 16 );
        m_fewerButton->setEnabled( m_advancedLines.count() > 1 );
    }
}

#include "advancedsearchdialog.moc"
