/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   Copyright (C) 2004 by Scott Wheeler                                   *
*   wheeler@kde.org                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#include <kapplication.h>
#include <kiconloader.h>
#include <kurldrag.h>
#include <kmessagebox.h>
#include <ksplashscreen.h>
#include <kstandarddirs.h>
#include <kparts/partmanager.h>
#include <kpopupmenu.h>
#include <kaction.h>
#include <kdebug.h>

#include <qheader.h>
#include <qpainter.h>
#include <qwidgetstack.h>

#include "datatablebox.h"
#include "datatable.h"
#include "datakiosk.h"
#include "dataclip.h"

#include "project.h"
#include "datareport.h"
#include "viewmode.h"
#include "actioncollection.h"
#include "databaseconnection.h"

#define widget (kapp->mainWidget())

using namespace ActionCollection;

DataTableBox::DataTableBox( QWidget *parent, QWidgetStack *dataTableStack,
                            const char *name ) :
        KListView( parent, name ),
        Project( dataTableStack ),
        m_viewModeIndex( 0 ),
        m_hasSelection( false ),
        m_doingMultiSelect( false ),
        m_treeViewSetup( false ),
        m_dropItem( 0 ),
        m_clipShown( false )
{
    setFocusPolicy( QWidget::ClickFocus );

    readConfig();
    addColumn( "DataTables", width() );

    header() ->blockSignals( true );
    header() ->hide();
    header() ->blockSignals( false );

    setSorting( 0 );
    setFullWidth( true );
    setItemMargin( 3 );

    setAcceptDrops( false );
    setDragEnabled( false );
    setSelectionModeExt( Single );

    m_listMenu = new KPopupMenu( this );

    action( "create_childtable" ) ->plug( m_listMenu );
    action( "create_search" ) ->plug( m_listMenu );
    action( "create_report" ) ->plug( m_listMenu );
    m_listMenu->insertSeparator();
    action( "edit_table" ) ->plug( m_listMenu );
    action( "add_virtual" ) ->plug( m_listMenu );
    m_listMenu->insertSeparator();
    action( "table_up" ) ->plug( m_listMenu );
    action( "table_down" ) ->plug( m_listMenu );
    m_listMenu->insertSeparator();
    action( "remove_table" ) ->plug( m_listMenu );

    m_reportMenu = new KPopupMenu( this );
    action( "refresh_report" ) ->plug( m_reportMenu );
    action( "print" ) ->plug( m_reportMenu );
    m_reportMenu->insertSeparator();
    action( "edit_report" ) ->plug( m_reportMenu );
    m_reportMenu->insertSeparator();
    action( "remove_report" ) ->plug( m_reportMenu );

    // add the view modes stuff

    KSelectAction *viewModeAction =
        new KSelectAction( i18n( "View Modes" ), "view_choose", KShortcut(), actions(), "view_mode_menu" );

    m_viewModes.append( new ViewMode( this ) );
    m_viewModes.append( new CompactViewMode( this ) );
    m_viewModes.append( new TreeViewMode( this ) );

    QStringList modeNames;

    for ( QValueListIterator<ViewMode *> it = m_viewModes.begin(); it != m_viewModes.end(); ++it )
        modeNames.append( ( *it ) ->name() );

    viewModeAction->setItems( modeNames );

    QPopupMenu *p = viewModeAction->popupMenu();

    p->changeItem( 0, SmallIconSet( "view_detailed" ), modeNames[ 0 ] );
    p->changeItem( 1, SmallIconSet( "view_text" ), modeNames[ 1 ] );
    p->changeItem( 2, SmallIconSet( "view_tree" ), modeNames[ 2 ] );

    viewModeAction->setCurrentItem( m_viewModeIndex );
    m_viewModes[ m_viewModeIndex ] ->setShown( true );

    connect( viewModeAction, SIGNAL( activated( int ) ), this, SLOT( slotSetViewMode( int ) ) );

    connect( this, SIGNAL( selectionChanged() ),
             this, SLOT( slotWidgetChanged() ) );

    connect( this, SIGNAL( contextMenuRequested( QListViewItem *, const QPoint &, int ) ),
             this, SLOT( slotShowContextMenu( QListViewItem *, const QPoint &, int ) ) );

    //UGLY HACK
    uglyInitialize();
}

DataTableBox::~DataTableBox()
{
    saveConfig();
}

void DataTableBox::raise( DataTable *dataTable )
{
    if ( !dataTable )
        return ;

    Item *i = m_dataTableDict.find( dataTable );

    clearSelection();
    setSelected( i, true );

    setSingleItem( i );
    ensureItemVisible( currentItem() );
}

void DataTableBox::raise( DataReport *dataReport )
{
    Project::raise( dataReport );
    if ( !dataReport )
        return ;

    Item *i = m_dataTableDict.find( dataReport );

    clearSelection();
    setSelected( i, true );

    setSingleItem( i );
    ensureItemVisible( currentItem() );
}

void DataTableBox::initialize()
{
    QPixmap pm;
    pm.load( locate( "appdata", "pics/datakiosk-splash.png" ) );
    KSplashScreen *splash = new KSplashScreen( pm );

    connect( this, SIGNAL( splashMessage( const QString &, int, const QColor & ) ),
            splash, SLOT( message( const QString &, int, const QColor & ) ) );

    if ( !fileName().isEmpty() )
        splash->show();

    loadConnections();

    if ( loadDataTables() )
    {
        setSelected( currentItem(), false );
        setCurrentItem( firstChild() );
        setSelected( firstChild(), true );
        action( "edit_project" ) ->setEnabled( true );
        action( "create_datatable" ) ->setEnabled( true );
        action( "save_project" ) ->setEnabled( true );
        action( "save_project_as" ) ->setEnabled( true );
        action( "close_project" ) ->setEnabled( true );
        action( "print" ) ->setEnabled( false );
    }
    else
    {
        action( "edit_project" ) ->setEnabled( false );
        action( "create_datatable" ) ->setEnabled( false );
        action( "save_project" ) ->setEnabled( false );
        action( "save_project_as" ) ->setEnabled( false );
        action( "close_project" ) ->setEnabled( false );
        action( "print" ) ->setEnabled( false );
        emit selectionChanged();
    }

    if ( !fileName().isEmpty() )
        splash->finish( this );

    m_clipboard = new DataClip( KApplication::desktop() );
    connect( widget, SIGNAL( showing() ),
             this, SLOT( clipOnWindowShown() ) );
    connect( widget, SIGNAL( hiding() ),
             m_clipboard, SLOT( hide() ) );
    connect( widget, SIGNAL( raising() ),
             m_clipboard, SLOT( raise() ) );
    connect( widget, SIGNAL( lowering() ),
             m_clipboard, SLOT( lower() ) );

    delete splash;
}

//UGLY HACK
void DataTableBox::uglyInitialize()
{
    QTimer::singleShot( 0, this, SLOT( slotInitialize() ) );
}

void DataTableBox::newProject()
{
    createDatabaseDataTable();
    currentItem() ->setSelected( true );
    currentItem() ->repaint();
    emit selectionChanged();
    action( "edit_project" ) ->setEnabled( true );
    action( "create_datatable" ) ->setEnabled( true );
    action( "save_project" ) ->setEnabled( true );
    action( "save_project_as" ) ->setEnabled( true );
    action( "close_project" ) ->setEnabled( true );
    action( "print" ) ->setEnabled( false );
}

void DataTableBox::closeProject()
{
    Project::closeProject();

    selectAll( true );
    removeTable();
}

void DataTableBox::clipboard()
{
    Project::clipboard();

    if ( m_clipboard->isVisible() )
        m_clipboard->hide();
    else
        m_clipboard->show();

    m_clipShown = m_clipboard->isVisible();
}

void DataTableBox::clipOnWindowShown()
{
    if ( m_clipShown )
        m_clipboard->show();
}

DataTable *DataTableBox::currentDataTable() const
{
    Item * item = dynamic_cast<Item *>( currentItem() );

    if ( item )
        if ( DataTable * dataTable = dynamic_cast<DataTable*>( item->itemWidget() ) )
        {
            if ( item && dataTable )
                return dataTable;
            else
                return dynamic_cast<DataTable *>( dataTableStack() ->visibleWidget() );
        }
    return 0;
}

DataReport *DataTableBox::currentDataReport() const
{
    Item * item = dynamic_cast<Item *>( currentItem() );

    if ( item )
        if ( DataReport * dataReport = dynamic_cast<DataReport*>( item->itemWidget() ) )
        {
            if ( item && dataReport )
                return dataReport;
            else
                return dynamic_cast<DataReport *>( dataTableStack() ->visibleWidget() );
        }
    return 0;
}

DataTable *DataTableBox::rootOfCurrentDataTable()
{
    ItemList items = selectedItems();
    if ( items.count() > 1 )
        return 0;

    Item* root = items[ 0 ];
    while ( root->QListViewItem::parent() )
    {
        root = static_cast<Item *>( root->QListViewItem::parent() );
    }

    return dynamic_cast<DataTable*>( root->itemWidget() );
}

DataTableList DataTableBox::dataTablesInCurrentTree()
{
    DataTableList dataTables;

    ItemList items = selectedItems();
    if ( items.count() > 1 )
        return dataTables;

    Item* root = items[ 0 ];
    while ( root->QListViewItem::parent() )
    {
        root = static_cast<Item *>( root->QListViewItem::parent() );
    }

    QListViewItemIterator it( root );
    while ( it.current() )
    {
        if ( it.current() == root->nextSibling() )
            break;

        Item *item = static_cast<Item *>( ( *it ) );
        if ( DataTable * dataTable = dynamic_cast<DataTable*>( item ->itemWidget() ) )
            dataTables.append( dataTable );
        ++it;
    }

    return dataTables;
}

DataTableList DataTableBox::dataTablesInDataTableTree( DataTable *dataTable )
{
    DataTableList dataTables;

    Item* root = 0;
    for ( QListViewItemIterator it( this ); it.current(); ++it )
    {
        Item * item = dynamic_cast<Item *>( *it );

        if ( DataTable * d = dynamic_cast<DataTable*>( item->itemWidget() ) )
        {
            if ( d == dataTable )
            {
                root = item;
                break;
            }
        }
    }

    if ( !root )
        return dataTables;

    while ( root->QListViewItem::parent() )
    {
        root = static_cast<Item *>( root->QListViewItem::parent() );
    }

    QListViewItemIterator it( root );
    while ( it.current() )
    {
        if ( it.current() == root->nextSibling() )
            break;

        Item *item = static_cast<Item *>( ( *it ) );
        if ( DataTable * dataTable = dynamic_cast<DataTable*>( item ->itemWidget() ) )
            dataTables.append( dataTable );
        ++it;
    }

    return dataTables;
}

DataTableList DataTableBox::childrenOfDataTable( DataTable *dataTable )
{
    DataTableList dataTables;

    Item* root = 0;
    for ( QListViewItemIterator it( this ); it.current(); ++it )
    {
        Item * item = dynamic_cast<Item *>( *it );

        if ( DataTable * d = dynamic_cast<DataTable*>( item->itemWidget() ) )
        {
            if ( d == dataTable )
            {
                root = item;
                break;
            }
        }
    }

    if ( !root )
        return dataTables;

    if ( !root->firstChild() )
        return dataTables;

    QListViewItemIterator it( root->firstChild() );
    while ( it.current() )
    {
        if ( it.current()->depth() <= root->depth() )
            break;

        Item *item = static_cast<Item *>( ( *it ) );

        if ( DataTable * dataTable = dynamic_cast<DataTable*>( item ->itemWidget() ) )
            dataTables.append( dataTable );
        ++it;
    }

    return dataTables;
}

void DataTableBox::scrollTableUp()
{
    QListViewItem* nextItem = currentItem()->itemAbove();

    if ( !nextItem )
        nextItem = lastItem();

    if ( nextItem )
    {
        currentItem()->setSelected( false );
        nextItem->setSelected( true );
        setCurrentItem( nextItem );
    }

    if ( dynamic_cast<DataReport*>( dynamic_cast<Item*>( nextItem )->itemWidget() ) )
        scrollTableUp();
    else
        slotWidgetChanged();
}

void DataTableBox::scrollTableDown()
{
    QListViewItem* nextItem = currentItem()->itemBelow();

    if ( !nextItem )
        nextItem = firstChild();

    if ( nextItem )
    {
        currentItem()->setSelected( false );
        nextItem->setSelected( true );
        setCurrentItem( nextItem );
    }

    if ( dynamic_cast<DataReport*>( dynamic_cast<Item*>( nextItem )->itemWidget() ) )
        scrollTableDown();
    else
        slotWidgetChanged();
}

void DataTableBox::setSplashMessage( const QString &txt )
{
    emit splashMessage( txt, Qt::AlignRight, QColor( 0, 84, 255 ) );
}

void DataTableBox::setupDataTable( DataTable *dataTable, const QString &iconName )
{
    Project::setupDataTable( dataTable, iconName );
    Item *item = new Item( this, iconName, dataTable->name(), dataTable );
    //     connect( item, SIGNAL( itemDestroyed( Item* ) ),
    //              this, SLOT( slotItemDestroyed( Item* ) ) );
    connect( dataTable, SIGNAL( tableRefreshed() ),
             item, SLOT( repaintRequested() ) );
    connect( dataTable, SIGNAL( signalNameChanged( const QString & ) ),
             item, SLOT( slotSetName( const QString & ) ) );
}

void DataTableBox::setupDataTable( DataTable *dataTable, const QString &iconName, Item *parentItem )
{
    Project::setupDataTable( dataTable, iconName );
    Item *item = new Item( parentItem, iconName, dataTable->name(), dataTable );
    //     connect( item, SIGNAL( itemDestroyed( Item* ) ),
    //              this, SLOT( slotItemDestroyed( Item* ) ) );
    connect( dataTable, SIGNAL( tableRefreshed() ),
             item, SLOT( repaintRequested() ) );
    connect( dataTable, SIGNAL( signalNameChanged( const QString & ) ),
             item, SLOT( slotSetName( const QString & ) ) );
}

void DataTableBox::setupDataTable( DataTable *dataTable, const QString &iconName, DataTable *parentTable )
{
    Project::setupDataTable( dataTable, iconName );
    Item *item = new Item( m_dataTableDict.find( parentTable ), iconName, dataTable->name(), dataTable );
    //     connect( item, SIGNAL( itemDestroyed( Item* ) ),
    //              this, SLOT( slotItemDestroyed( Item* ) ) );
    connect( dataTable, SIGNAL( tableRefreshed() ),
             item, SLOT( repaintRequested() ) );
    connect( dataTable, SIGNAL( signalNameChanged( const QString & ) ),
             item, SLOT( slotSetName( const QString & ) ) );
}

void DataTableBox::setIconName( DataTable* dataTable, const QString &iconName )
{
    for ( QListViewItemIterator it( this ); it.current(); ++it )
    {
        Item *item = static_cast<Item *>( *it );
        if ( DataTable * d = dynamic_cast<DataTable*>( item->itemWidget() ) )
        {
            if ( d ->name() == dataTable->name() )
            {
                int iconSize = viewModeIndex() == 0 ? 32 : 16;
                static_cast<Item *>( *it ) ->setIconName( iconName );
                static_cast<Item *>( *it ) ->setPixmap( 0, SmallIcon( iconName, iconSize ) );
            }
        }
    }
}

void DataTableBox::setupDataReport( DataReport *dataReport, const QString &iconName, Item *parentItem )
{
    Project::setupDataReport( dataReport, iconName );
    Item *item = new Item( parentItem, iconName, dataReport->name(), dataReport );
    //     connect( item, SIGNAL( itemDestroyed( Item* ) ),
    //              this, SLOT( slotItemDestroyed( Item* ) ) );
    connect( dataReport, SIGNAL( reportRefreshed() ),
             item, SLOT( repaintRequested() ) );
    connect( dataReport, SIGNAL( signalNameChanged( const QString & ) ),
             item, SLOT( slotSetName( const QString & ) ) );
}

void DataTableBox::setupDataReport( DataReport *dataReport, const QString &iconName, DataTable *parentTable )
{
    Project::setupDataReport( dataReport, iconName );
    Item *item = new Item( m_dataTableDict.find( parentTable ), iconName, dataReport->name(), dataReport );
    //     connect( item, SIGNAL( itemDestroyed( Item* ) ),
    //              this, SLOT( slotItemDestroyed( Item* ) ) );
    connect( parentTable, SIGNAL( tableRefreshed() ),
             dataReport, SLOT( queueRefresh() ) );
    connect( dataReport, SIGNAL( reportRefreshed() ),
             item, SLOT( repaintRequested() ) );
    connect( dataReport, SIGNAL( signalNameChanged( const QString & ) ),
             item, SLOT( slotSetName( const QString & ) ) );
}

void DataTableBox::setIconName( DataReport *dataReport, const QString &iconName )
{
    for ( QListViewItemIterator it( this ); it.current(); ++it )
    {
        Item *item = static_cast<Item *>( *it );
        if ( DataReport * d = dynamic_cast<DataReport*>( item->itemWidget() ) )
        {
            if ( d ->name() == dataReport->name() )
            {
                int iconSize = viewModeIndex() == 0 ? 32 : 16;
                static_cast<Item *>( *it ) ->setIconName( iconName );
                static_cast<Item *>( *it ) ->setPixmap( 0, SmallIcon( iconName, iconSize ) );
            }
        }
    }
}

void DataTableBox::readConfig()
{
    KConfigGroup config( KGlobal::config(), "DataTableBox" );
    m_viewModeIndex = config.readNumEntry( "ViewMode", 0 );
}

void DataTableBox::saveConfig()
{
    KConfigGroup config( KGlobal::config(), "DataTableBox" );
    config.writeEntry( "ViewMode", action<KSelectAction>( "view_mode_menu" ) ->currentItem() );
    KGlobal::config() ->sync();
}

void DataTableBox::slotItemDestroyed( Item * )
{
    //     removeName( item ->text( 0 ) );
    //     m_dataTableDict.remove( item ->dataTable() );
}

void DataTableBox::editSearch()
{
    Project::editSearch();
    emit searchEdited();
}

void DataTableBox::removeSearch()
{
    Project::removeSearch();
    emit searchRemoved();
}

void DataTableBox::invokeSearch( const QString &search )
{
    Project::invokeSearch( search );
    emit searchInvoked( search );
}

void DataTableBox::advancedSearchRequested()
{
    emit advancedSearchClicked();
}

void DataTableBox::removeTable()
{
    ItemList items = selectedItems();

    if ( items.isEmpty() )
        return ;

    DataTableList removeQueue;

    for ( ItemList::ConstIterator it = items.begin(); it != items.end(); ++it )
    {
        if ( DataTable * dataTable = dynamic_cast<DataTable*>( ( *it ) ->itemWidget() ) )
        {
            removeName( ( *it ) ->text( 0 ) );
            m_dataTableDict.remove( dataTable );
            removeQueue.append( dataTable );
            removeDataTable( ( *it ) ->text( 0 ) );
        }
    }

    Item *x = static_cast<Item *>( items.back() ->nextSibling() );
    if ( x && dynamic_cast<DataTable*>( x->itemWidget() ) )
        setSingleItem( x );
    else
    {
        Item *y = static_cast<Item *>( items.front() ->itemAbove() );
        while ( y && !dynamic_cast<DataTable*>( y->itemWidget() ) )
            y = static_cast<Item *>( y->itemAbove() );

        setSingleItem( y );
    }

    for ( DataTableList::Iterator it = removeQueue.begin(); it != removeQueue.end(); ++it )
    {
        DataTableList dataTables = dataTablesInDataTablePeerTree( *it );
        DataTableList::Iterator it2 = dataTables.begin();
        for ( ; it2 != dataTables.end(); ++it2 )
        {
            if ( ( *it2 ) == ( *it ) )
                continue;
            else if ( ( *it2 )->number() > ( *it )->number() )
                ( *it2 )->setNumber( ( *it2 )->number() - 1 );
        }

        delete * it;
    }
}

void DataTableBox::sortTables()
{
    if ( childCount() > 1 )
        sort();
    else
        firstChild()->sort();
}

void DataTableBox::removeReport()
{
    ItemList items = selectedItems();

    if ( items.isEmpty() )
        return ;

    DataReportList removeQueue;

    for ( ItemList::ConstIterator it = items.begin(); it != items.end(); ++it )
    {
        if ( DataReport * dataReport = dynamic_cast<DataReport*>( ( *it ) ->itemWidget() ) )
        {
            removeName( ( *it ) ->text( 0 ) );
            m_dataTableDict.remove( dataReport );
            removeQueue.append( dataReport );
            removeDataReport( ( *it ) ->text( 0 ) );
        }
    }

    Item *x = static_cast<Item *>( items.back() ->nextSibling() );
    if ( x && dynamic_cast<DataTable*>( x->itemWidget() ) )
        setSingleItem( x );
    else
    {
        Item *y = static_cast<Item *>( items.front() ->itemAbove() );
        while ( y && !dynamic_cast<DataTable*>( y->itemWidget() ) )
            y = static_cast<Item *>( y->itemAbove() );

        setSingleItem( y );
    }

    for ( DataReportList::Iterator it = removeQueue.begin(); it != removeQueue.end(); ++it )
    {
        delete * it;
    }
}

void DataTableBox::contentsMousePressEvent( QMouseEvent *e )
{
    if ( e->button() == LeftButton )
        m_doingMultiSelect = true;
    KListView::contentsMousePressEvent( e );
}

void DataTableBox::contentsMouseReleaseEvent( QMouseEvent *e )
{
    if ( e->button() == LeftButton )
    {
        m_doingMultiSelect = false;
        slotWidgetChanged();
    }
    KListView::contentsMouseReleaseEvent( e );
}

void DataTableBox::keyPressEvent( QKeyEvent *e )
{
    if ( ( e->key() == Key_Up || e->key() == Key_Down ) && e->state() == ShiftButton )
        m_doingMultiSelect = true;
    KListView::keyPressEvent( e );
}

void DataTableBox::keyReleaseEvent( QKeyEvent *e )
{
    if ( m_doingMultiSelect && e->key() == Key_Shift )
    {
        m_doingMultiSelect = false;
        slotWidgetChanged();
    }
    KListView::keyReleaseEvent( e );
}

bool DataTableBox::hasSelection()
{
    QListViewItemIterator it( this, QListViewItemIterator::Selected );
    return it.current();
}

DataTableBox::ItemList DataTableBox::selectedItems()
{
    ItemList l;

    for ( QListViewItemIterator it( this, QListViewItemIterator::Selected ); it.current(); ++it )
        l.append( static_cast<Item *>( *it ) );

    return l;
}

void DataTableBox::setSingleItem( QListViewItem *item )
{
    KListView::setCurrentItem( item );
}

void DataTableBox::slotWidgetChanged()
{
    // Don't update while the mouse is pressed down.

    if ( m_doingMultiSelect )
        return ;

    ItemList items = selectedItems();
    m_hasSelection = !items.isEmpty();

    DataTableList dataTables;
    DataReportList dataReports;
    for ( ItemList::ConstIterator it = items.begin(); it != items.end(); ++it )
    {
        if ( DataTable * dataTable = dynamic_cast<DataTable*>( ( *it ) ->itemWidget() ) )
            dataTables.append( dataTable );
        else if ( DataReport * dataReport = dynamic_cast<DataReport*>( ( *it ) ->itemWidget() ) )
            dataReports.append( dataReport );
    }

    bool singleDataTable = dataTables.count() == 1;
    bool singleDataReport = dataReports.count() == 1;

    if ( ( dataTables.isEmpty() || !singleDataTable )
            && ( dataReports.isEmpty() || !singleDataReport ) )
    {
        static_cast<DataKiosk*>( widget ) ->partManager() ->setActivePart( 0L ); // sets as the active part
        dataTableStack() ->raiseWidget( 0 );
        action( "edit_table" ) ->setEnabled( false );
        action( "add_virtual" ) ->setEnabled( false );
        action( "table_up" ) ->setEnabled( false );
        action( "table_down" ) ->setEnabled( false );
        action( "remove_table" ) ->setEnabled( false );
        action( "create_childtable" ) ->setEnabled( false );
        action( "create_search" ) ->setEnabled( false );
        action( "create_report" ) ->setEnabled( false );
        action( "edit_search" ) ->setEnabled( false );
        action( "remove_search" ) ->setEnabled( false );
        action( "edit_report" ) ->setEnabled( false );
        action( "refresh_report" ) ->setEnabled( false );
        action( "remove_report" ) ->setEnabled( false );
        action( "print" ) ->setEnabled( false );
    }
    else if ( singleDataTable )
    {
        static_cast<DataKiosk*>( widget ) ->partManager() ->setActivePart( 0L );
        dataTableStack() ->raiseWidget( dataTables.front() );
        action( "edit_table" ) ->setEnabled( true );
        action( "add_virtual" ) ->setEnabled( true );
        action( "table_up" ) ->setEnabled( true );
        action( "table_down" ) ->setEnabled( true );
        action( "remove_table" ) ->setEnabled( true );
        action( "create_childtable" ) ->setEnabled( true );
        action( "create_search" ) ->setEnabled( true );
        action( "create_report" ) ->setEnabled( true );
        action( "edit_report" ) ->setEnabled( false );
        action( "refresh_report" ) ->setEnabled( false );
        action( "remove_report" ) ->setEnabled( false );
        action( "print" ) ->setEnabled( false );
    }
    else if ( singleDataReport )
    {
        for ( DataReport * report = m_dataReports.first(); report; report = m_dataReports.next() )
            report ->setAsActiveReport( false );

        DataReport * report = dataReports.front();
            report ->setAsActiveReport( true );

        static_cast<DataKiosk*>( widget ) ->partManager() ->setActivePart( report->kugarPart() );
        dataTableStack() ->raiseWidget( report );
//         if ( report->needsRefresh() )
//             report->refreshReport();
        action( "edit_table" ) ->setEnabled( false );
        action( "add_virtual" ) ->setEnabled( false );
        action( "table_up" ) ->setEnabled( false );
        action( "table_down" ) ->setEnabled( false );
        action( "remove_table" ) ->setEnabled( false );
        action( "create_childtable" ) ->setEnabled( false );
        action( "create_search" ) ->setEnabled( false );
        action( "create_report" ) ->setEnabled( false );
        action( "edit_search" ) ->setEnabled( false );
        action( "remove_search" ) ->setEnabled( false );
        action( "edit_report" ) ->setEnabled( true );
        action( "refresh_report" ) ->setEnabled( true );
        action( "remove_report" ) ->setEnabled( true );
        action( "print" ) ->setEnabled( report->isInitialized() );
    }
}

void DataTableBox::slotShowContextMenu( QListViewItem *item, const QPoint &point, int )
{
    Item * i = static_cast<Item*>( item );

    if ( i )
        if ( dynamic_cast<DataTable*>( i->itemWidget() ) )
            m_listMenu->popup( point );
        else if ( dynamic_cast<DataReport*>( i->itemWidget() ) )
            m_reportMenu->popup( point );
}

void DataTableBox::slotSetViewMode( int index )
{
    if ( index == m_viewModeIndex )
        return ;

    viewMode() ->setShown( false );
    m_viewModeIndex = index;
    viewMode() ->setShown( true );

    // The following call only does anything if the setup
    // hasn't already been performed.

    performTreeViewSetup();
}

void DataTableBox::setupItem( Item *item )
{
    m_dataTableDict.insert( item->itemWidget(), item );
    viewMode() ->queueRefresh();
}

void DataTableBox::performTreeViewSetup()
{
    if ( m_treeViewSetup || m_viewModeIndex != 2 )
        return ;

    m_treeViewSetup = true;
}

DataTableBox::Item *DataTableBox::Item::m_collectionItem = 0;

DataTableBox::Item::Item( DataTableBox *listBox, const QString &icon, const QString &text, QWidget *l )
        : QObject( listBox, text.ascii() ), KListViewItem( listBox, text ),
        m_widget( l ), m_text( text ), m_iconName( icon )
{
    init();
}

DataTableBox::Item::Item( Item *parent, const QString &icon, const QString &text, QWidget *l )
        : QObject( parent->listView(), text.ascii() ), KListViewItem( parent, text ),
        m_widget( l ), m_text( text ), m_iconName( icon )
{
    init();
}

DataTableBox::Item::~Item()
{
    emit itemDestroyed( this );
}

int DataTableBox::Item::compare( QListViewItem *i, int col, bool ) const
{
    DataTable* thisIsDataTable = dynamic_cast<DataTable*>( itemWidget() );
    DataTable* thatIsDataTable = dynamic_cast<DataTable*>( static_cast<Item *>( i ) ->itemWidget() );

    if ( thisIsDataTable && !thatIsDataTable )
        return -1;
    else if ( !thisIsDataTable && thatIsDataTable )
        return 1;
    else if ( thisIsDataTable && thatIsDataTable )
    {
        if ( thisIsDataTable->number() < thatIsDataTable->number() )
            return -1;
        else if ( thisIsDataTable->number() > thatIsDataTable->number() )
            return 1;
    }

    return text( col ).lower().localeAwareCompare( i->text( col ).lower() );
}

void DataTableBox::Item::paintCell( QPainter *painter, const QColorGroup &colorGroup, int column, int width, int align )
{
    DataTableBox * dataTableBox = static_cast<DataTableBox *>( listView() );
    dataTableBox->viewMode() ->paintCell( this, painter, colorGroup, column, width, align );
}

void DataTableBox::Item::setText( int column, const QString &text )
{
    m_text = text;
    KListViewItem::setText( column, text );
}

void DataTableBox::Item::setup()
{
    listView() ->viewMode() ->setupItem( this );
}

void DataTableBox::Item::repaintRequested()
{
    /*    kdDebug() << "repaintRequested" << dataTable()->numRows() << endl;*/
    listView() ->viewMode() ->queueRefresh();
    repaint();
}

void DataTableBox::Item::slotSetName( const QString &name )
{
    if ( listView() )
    {
        setText( 0, name );
        setSelected( true );

        listView() ->sort();
        listView() ->ensureItemVisible( listView() ->currentItem() );
        listView() ->viewMode() ->queueRefresh();
    }
}

void DataTableBox::Item::init()
{
    DataTableBox * list = listView();

    list->setupItem( this );

    setDragEnabled( false );
    setDropEnabled( false );

    int iconSize = list->viewModeIndex() == 0 ? 32 : 16;
    setPixmap( 0, SmallIcon( m_iconName, iconSize ) );
    list->addName( m_text );

    if ( m_widget )
    {
        connect( m_widget, SIGNAL( destroyed() ), this, SLOT( deleteLater() ) );
    }
}

#ifdef widget
#undef widget
#endif

#include "datatablebox.moc"
