/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

/***************************************************************************
*   This file adapted from libkdepim ktimeedit.cpp                        *
*                                                                         *
*   Copyright (c) 1999 Preston Brown, Ian Dawes                           *
*   Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>   *
*                                                                         *
***************************************************************************/

/*
    If you have any questions about how this class works, please see the
    version in libkdepim.  Hopefully these classes will be cleaned up and
    put in kdelibs for KDE4.
*/

#include "timeedit.h"

#include <qkeycode.h>
#include <qcombobox.h>
#include <qdatetime.h>
#include <qlineedit.h>

#include <kmessagebox.h>
#include <kglobal.h>
#include <kdebug.h>
#include <klocale.h>

#include <qvalidator.h>

class TimeValidator : public QValidator
{
public:
    TimeValidator( QWidget* parent, const char* name = 0 )
            : QValidator( parent, name )
    {}

    virtual State validate( QString& str, int& ) const
    {
        int length = str.length();
        if ( length <= 0 )
            return Intermediate;

        bool ok = false;
        KGlobal::locale() ->readTime( str, KLocale::WithoutSeconds, &ok );
        if ( ok )
            return Acceptable;
        int tm = str.toInt( &ok );
        if ( ok && ( 0 <= tm ) )
        {
            if ( ( tm < 2400 ) && ( tm % 100 < 60 ) )
                return Acceptable;
            else
                return Intermediate;
        }

        QChar sep = ':';
        if ( str[ 0 ] == sep )
        {
            if ( length == 1 )
                return Intermediate;
            QString minutes = str.mid( 1 );
            int m = minutes.toInt( &ok );
            if ( ok && m >= 0 && m < 60 )
                return Intermediate;
        }
        else if ( str[ str.length() - 1 ] == sep )
        {
            QString hours = str.left( length - 1 );
            int h = hours.toInt( &ok );
            if ( ok && h >= 0 && h < 24 )
                return Intermediate;
        }
        return Intermediate;
    }
    virtual void fixup ( QString & input ) const
    {
        bool ok = false;
        KGlobal::locale() ->readTime( input, KLocale::WithoutSeconds, &ok );
        if ( !ok )
        {
            int tm = input.toInt( &ok );
            if ( ( 0 <= tm ) && ( tm < 2400 ) && ( tm % 100 < 60 ) && ok )
            {
                input = KGlobal::locale() ->formatTime(
                            QTime( tm / 100, tm % 100, 0 ) );
            }
        }
    }
};

TimeEdit::TimeEdit( DataField* field, DataTable *dataTable,
                    QWidget *parent, QTime qt, const char *name )
        : DataComboBox( field, dataTable, true, parent, name )
{
    setInsertionPolicy( NoInsertion );
    setValidator( new TimeValidator( this ) );

    mTime = qt;

    QTime timeEntry( 0, 0, 0 );
    do
    {
        insertItem( KGlobal::locale() ->formatTime( timeEntry ) );
        timeEntry = timeEntry.addSecs( 60 * 15 );
    }
    while ( !timeEntry.isNull() );
    insertItem( KGlobal::locale() ->formatTime( QTime( 23, 59, 59 ) ) );

    updateText();
    setFocusPolicy( QWidget::StrongFocus );

    connect( this, SIGNAL( activated( int ) ), this, SLOT( active( int ) ) );
    connect( this, SIGNAL( textChanged( const QString& ) ),
             this, SLOT( changedText() ) );
}

TimeEdit::~TimeEdit()
{}

bool TimeEdit::hasTime() const
{
    if ( currentText().isEmpty() )
        return false;

    return true;
}

QTime TimeEdit::getTime() const
{
    bool ok = false;
    QTime time = KGlobal::locale() ->readTime( currentText(),
                 KLocale::WithoutSeconds, &ok );
    if ( !ok )
    {
        int tm = currentText().toInt( &ok );
        if ( ( 0 <= tm ) && ( tm < 2400 ) && ( tm % 100 < 60 ) && ok )
        {
            time.setHMS( tm / 100, tm % 100, 0 );
        }
        else
        {
            ok = false;
        }
    }
    return time;
}

void TimeEdit::setTime( QTime newTime )
{
    if ( mTime != newTime )
    {
        mTime = newTime;
        updateText();
    }
}

void TimeEdit::active( int i )
{
    if ( i == count() - 1 )
        mTime = QTime( 23, 59, 0 );
    else
        mTime = QTime( 0, 0, 0 ).addSecs( i * 15 * 60 );
    emit timeChanged( mTime );
}

void TimeEdit::addTime( QTime qt )
{
    mTime = qt.addSecs( mTime.minute() * 60 + mTime.hour() * 3600 );
    updateText();
    emit timeChanged( mTime );
}

void TimeEdit::subTime( QTime qt )
{
    int h, m;

    h = mTime.hour() - qt.hour();
    m = mTime.minute() - qt.minute();

    if ( m < 0 )
    {
        m += 60;
        h -= 1;
    }

    if ( h < 0 )
    {
        h += 24;
    }

    mTime.setHMS( h, m, 0 );
    updateText();
    emit timeChanged( mTime );
}

void TimeEdit::keyPressEvent( QKeyEvent *qke )
{
    switch ( qke->key() )
    {
    case Key_Down:
        addTime( QTime( 0, 1, 0 ) );
        break;
    case Key_Up:
        subTime( QTime( 0, 1, 0 ) );
        break;
    case Key_Prior:
        subTime( QTime( 1, 0, 0 ) );
        break;
    case Key_Next:
        addTime( QTime( 1, 0, 0 ) );
        break;
    default:
        QComboBox::keyPressEvent( qke );
        break;
    }
}

void TimeEdit::updateText()
{
    QString s = KGlobal::locale() ->formatTime( mTime );
    QLineEdit *line = lineEdit();
    line->blockSignals( true );
    int pos = line->cursorPosition();

    setCurrentItem( ( mTime.hour() * 4 ) + ( ( mTime.minute() + 7 ) / 15 ) );

    line->setText( s );
    line->setCursorPosition( pos );
    line->blockSignals( false );
}

bool TimeEdit::inputIsValid() const
{
    int cursorPos = lineEdit() ->cursorPosition();
    QString str = currentText();
    return validator() ->validate( str, cursorPos ) == QValidator::Acceptable;
}

void TimeEdit::changedText()
{
    if ( inputIsValid() )
    {
        mTime = getTime();
        emit timeChanged( mTime );
    }
}

#include "timeedit.moc"
