/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * action.h: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: action.h,v 1.2 2003/12/28 08:12:38 uid68112 Exp $
*/

#ifndef __ACTION_H__
#define __ACTION_H__

#include <glib.h>
#include <pthread.h>

G_LOCK_EXTERN(waiting_action);		/* this mutex protects waiting_action array */
extern GPtrArray *waiting_action;	/* array of pointers on WAIT_ACT structure (running connection) */

typedef enum
{
	CNX_ON_CONNECT,			/* created after a connect (in reply to a $ConnectToMe) */
	CNX_ON_ACCEPT,				/* created after accepting an incoming connection */
} CNX_TYPE;

typedef struct
{
	int running;					/* 0=not running, 1=waiting remote connection, 2=running, 3=killed */
	CNX_TYPE cnx_type;			/* thread created on accept or connect ? */
	time_t last_touch;			/* date of the last change of this task */
	pthread_t thread_id;			/* !=0, this is the thread id */
	int sock_fd;					/* socket descriptor used by the thread */
	GString *remote_nick;		/* nick of the user on the remote side */
	GString *remote_addr;		/* addr:port of the remote side */
	struct wait_revcon *run_task;	/* task associated to this thread */
	GString *disp_info;			/* info displayed when the xfer list is requested */

	GStringChunk *cap_str;		/* used to store remote client capabilities */
	GPtrArray *cap_ptr;			/* array of pointers on remote client capabilities */
} WAIT_ACT;

/* list of all RevConnectToMe sent */
G_LOCK_EXTERN(waiting_revcon);		/* this mutex protects waiting_action array */
extern GPtrArray *waiting_revcon;	/* queued task */

typedef struct wait_revcon
{
	time_t last_touch;		/* date of the last change of this task */
	GString *remote_nick;	/* the nickname of the people we want to reached */
	GString *action_to_do;	/* the action to do */
} WAIT_REVCON;

/* list of all commands waiting to be sent to the keyboard */
/* it acts like a loopback. When DCTC wants to send itself command like if a user has entered them */
G_LOCK_EXTERN(sim_input);
extern GArray *sim_input;

typedef struct
{
	unsigned long id;
	time_t min_start_time;	/* before this date, the task won't be started */
	GString *keyb_string;	/* string to enter */
} SIM_INPUT;

/**************************/
/* create an action to do */
/*********************************************************************************/
/* nick is copied but action is used as is and should not be freed by the caller */
/*********************************************************************************/
WAIT_REVCON *create_action_to_do(char *nick,GString *action);

/*********************************************************************/
/* create an action to do and append it to the list of actions to do */
/*********************************************************************************/
/* nick is copied but action is used as is and should not be freed by the caller */
/*********************************************************************************/
void create_and_add_action_to_do(char *nick,GString *action);

/*************************************************************************/
/* remote the given WAIT_REVCON from the waiting_revcon list and free it */
/*************************************************************************/
/* if with_remove==0, the ptr is only freed, else, it is also removed */
/* from waiting_revcon                                                */
/**********************************************************************/
void free_action_to_do(WAIT_REVCON *ptr,int with_remove);

/******************************************************************************************/
/* find the first wait_revcon for the given nick. The wait_revcon is removed of the array */
/******************************************************************************************/
WAIT_REVCON *get_action_to_do(GString *wanted_nick);

/******************************************************************************************/
/* find the first wait_revcon for the given nick. The wait_revcon is removed of the array */
/* the wait_rev_con is a download action (/LS or /DL).                                    */
/******************************************************************************************/
WAIT_REVCON *get_download_action_to_do(GString *wanted_nick);

/*************************************************************************/
/* add the given wait_revcon at the beginning of the waiting_revcon list */
/*************************************************************************/
void prepend_action_to_do(WAIT_REVCON *wrc);

/***********************************************/
/* destruction of the given WAIT_ACT structure */
/***********************************************/
void free_wait_act(WAIT_ACT *act);

/**************************/
/* create a new sim input */
/*****************************************************************************/
/* input: wait_duration: number of seconds to wait before sending the string */
/*        string: string to enter on the keyboard                            */
/*****************************************************************************/
void add_new_sim_input(time_t wait_duration, const char *string);

/****************************************************************************/
/* search the given string in the sim input array and return its start time */
/****************************************************************************/
/* output: 1=found, (*start_time) contains the sim_input start time */
/*         0=not found                                              */
/********************************************************************/
int find_sim_input_delay(const char *string,time_t *start_time);

/**********************************************************************************************/
/* check if the given user already has something in progress in wait_action or in wait_revcon */
/**********************************************************************************************/
/* output: 1=yes, 0=no */
/***********************/
int act_with_user_in_progress(const char *nickname);

/********************************************************************************************/
/* scan running transfer list and shutdown connection of all uploads for the given nickname */
/********************************************************************************************/
void terminate_upload_of_user(char *nickname);


#endif
