/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DECIBEL_DAEMON_ACCOUNTMANAGER_H_
#define _DECIBEL_DAEMON_ACCOUNTMANAGER_H_

#include <QtCore/QObject>

#include <QtDBus/QDBusContext>

#include <Decibel/Types>

namespace QtTapioca
{
class Connection;
} // namespace QtTapioca

class ConnectionFacade;
class AccountManagerPrivate;

/**
 * @brief The AccountManager persistently stores account data of the user.
 *
 * The AccountManager persistently stores data of the accounts of the user.
 * This data can then be used to create connections.
 *
 * Since this class is accessable via DBus it associates a handle with each
 * account on registration. This handle is garanteed to be unique for the
 * runtime of the Decibel daemon and may change on restarts.
 *
 * The 0 handle is reserved and returned in error conditions.
 *
 * Data on contacts of the user is accessed through the ContactManager and
 * not relevant in this class.
 *
 * This class is accessible over DBus so that account management components
 * can access it.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class AccountManager : public QObject, protected QDBusContext
{
    Q_OBJECT

public:
    /**
     * @brief Constructor.
     */
    explicit AccountManager(ConnectionFacade *, QObject * parent = 0);
    /**
     * @brief Destructor.
     */
    ~AccountManager();

    /**
     * @brief Bring up all accounts that are not offline.
     *
     * This method brings up all accounts that are not marked to be offline.
     *
     * This is a method used internally and is not exported to D-Bus.
     */
    void bringUpAccounts();

    /**
     * @brief Returns the protocol of an account.
     * @param account_handle A handle to the account to examine.
     * @returns The protocol of the account.
     *
     * Get the protocol information of an account. Returns an empty string if
     * the account is undefined.
     *
     * This is a method used internally and is not exported to D-Bus.
     */
    QString protocol(const uint account_handle) const;

    /**
     * @brief Finds out whether an account exists.
     * @param account_handle A handle to the account to examine.
     * @returns true if the handle points to an valid account and false otherwise.
     *
     * A simple way to find out whether an account exists.
     *
     * This is a method used internally and is not exported to D-Bus.
     */
    bool gotAccount(const uint account_handle) const;

    /**
     * @brief Find the connection associated with an account.
     * @param account_handle A handle to the account to examine.
     * @returns true if the handle points to an valid account and false otherwise.
     *
     * A simple way to find out whether an account exists.
     *
     * This is a method used internally and is not exported to D-Bus.
     */
    QtTapioca::Connection * connectionOf(const uint account_handle) const;

public slots:
    /**
     * @brief List all known account handles.
     * @returns List of defined account handles.
     *
     * This function returns a list of all known account handles.
     **/
    QList<uint> listAccounts() const;

    /**
     * @brief Return account data for the given handle.
     * @param account_handle Account handle to examine.
     * @returns A list of name value pairs defining the account. This list is
     *          empty if the account handle is undefined.
     *
     * Get all account data for the given handle.
     **/
    QVariantMap queryAccount(const uint account_handle) const;

    /**
     * @brief Add a new account.
     * @param nv_pairs A list of name value pairs defining the account.
     * @returns A handle to the new account.
     * @throws data_incomplete Some required information was missing.
     * @throws internal_error Something unexpected happened trying to store
     *    the account information.
     *
     * Add a new set of data as an account. The QVariantMap must not be empty
     * and may not conflict with an existing account. The account is not
     * created on the service: All this command does is to store the
     * data associated with an account.
     *
     * Prerequisites:
     *  * One NVPair must have a name of "protocol" and a value-type of
     *    QString.
     */
    uint addAccount(const QVariantMap & nv_pairs);

    /**
     * @brief Update the data stored on one account.
     * @param account_handle A handle to the account to update.
     * @param nv_pairs N list of name value pairs to update.
     * @throws Decibel::ErrorNoSuchAccount Thrown when the given Id is invalid.
     * @throws Decibel::ErrorDataIncomplete Thrown when the data is incomplete.
     *
     * Update the account represented by the given handle with the
     * data from the QVariantMap. The new data replaces the existing data.
     * Fields from the old data that are no longer in the new data are
     * removed.
     *
     * Prerequisites and errorcodes are identical to those of addAccount(...).
     */
    void updateAccount(const uint account_handle, const QVariantMap & nv_pairs);

    /**
     * @brief Delete a set of account data.
     * @param account_handle A handle to the account to delete.
     *
     * Delete the account represented by the given handle.
     * This command is a NOP if the handle is invalid.
     *
     * The account is not deleted on the service, all this command does is to
     * remove the actual account data from storage.
     */
    void deleteAccount(const uint account_handle);

    /**
     * @brief Find accounts matching certain criteria.
     * @param nv_pairs A list of name value pairs that need to be matched.
     * @returns A list of account handles matching the criteria.
     *
     * Find accounts that match the given name/value pairs.
     * All of name, value and the type of the value must match for a
     * account to be found (query by example).
     *
     * A empty list matches everything.
     */
    QList<uint> findAccounts(const QVariantMap & nv_pairs) const;

    /**
     * @brief Set the presence information on an account (and change online
     *        state accordingly!).
     * @param account_handle A handle to the account to update.
     * @param presence_state The new presence state.
     * @returns A positive value or a negative error code.
     * @throws no_such_account if the account is invalid.
     * @throws invalid_value if the presence is invalid.
     *
     * Set the presence information on an account and update the online
     * state accordingly.
     */
    int setPresence(const uint account_handle, const int presence_state);

    /**
     * @brief Get the intended presence state of a account.
     * @param account_handle A handle to the account to examine.
     * @returns The current presence state.
     * @throws no_such_account if the account is invalid.
     *
     * Get the presence state of an account.
     */
    int presence(const uint account_handle);

    /**
     * @brief Get the current presence state of a account.
     * @param account_handle A handle to the account to examine.
     * @returns The current presence state.
     * @throws no_such_account if the account is invalid.
     *
     * Get the presence state of an account.
     */
    int currentPresence(const uint account_handle);

    /**
     * @brief Set the presence message of the given account.
     * @param account_handle A handle to the account to update.
     * @param message The new presence message.
     * @returns A positive value on success.
     * @throws no_such_account if the account is invalid.
     *
     * Set the presence message of the given account.
     */
    int setPresenceMessage(const uint account_handle, const QString & message);

    /**
     * @brief Get the presence message of the given account.
     * @param account_handle A handle to the account to examine.
     * @returns The current presence message.
     * @throws no_such_account if the account is invalid.
     *
     * Get the presence message of the given account.
     */
    QString presenceMessage(const uint account_handle) const;

    /**
     * @brief Set the presence state and the presence message at the same time.
     * @param account_handle A handle to the account to update.
     * @param presence_state The new presence state to set.
     * @param message The new presence message to set.
     * @returns A positive value on success or a negative error code.
     * @throws no_such_account if the account is invalid.
     * @throws invalid_value if the presence is invalid.
     *
     * Set the presence state and the presence message at the same time.
     *
     * Return value is > 0 if successful or a negative error code otherwise.
     */
    int setPresenceAndMessage(const uint account_handle,
                              const int presence_state,
                              const QString & message);

    /**
     * @brief Get the DBus service name of the account.
     * @param account_handle A handle to the account to examine.
     * @returns The service name of the connection associated with the account.
     *
     * Get the DBus service name of the connection associated with the given
     * account.
     *
     * It is empty if the account is offline.
     *
     * This is a method used internally and is not exported to D-Bus.
     */
    QString serviceName(const uint account_handle) const;

    /**
     * @brief Get the DBus object path of the account.
     * @param account_handle A handle to the account to examine.
     * @returns The object path of the connection associated with the account.
     *
     * Get the DBus object path of the connection associated with the given
     * account.
     *
     * It is empty if the account is offline.
     *
     * This is a method used internally and is not exported to D-Bus.
     */
    QString objectPath(const uint account_handle) const;

signals:
    /**
     * @brief A account was made known to decibel.
     * @param account_handle Handle of the newly created account.
     *
     * This signal is raised as soon as a new account is ready for use. The
     * account was either newly created or was newly registered with Decibel.
     */

    void accountCreated(const uint account_handle);
    /**
     * @brief A account was updated.
     * @param account_handle Handle of the updated account.
     *
     * This signal is raised as soon as the updated account is ready for use
     * again.
     */
    void accountUpdated(const uint account_handle);

    /**
     * @brief A account was deleted.
     * @param account_handle Handle of the deleted account.
     *
     * This signal is raised just after a account was deleted.
     */
    void accountDeleted(const uint account_handle);

private slots:
    void onOwnPresenceUpdated(QtTapioca::Connection *, const int, const QString &);

    void onConnectionClosed(QtTapioca::Connection *);

    void doBlockReconnects();

private:
    AccountManagerPrivate * const d;

    friend class ConnectionFacade;
};

#endif
