/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DECIBEL_DAEMON_CONTACTMANAGER_H_
#define _DECIBEL_DAEMON_CONTACTMANAGER_H_

#include <QtCore/QObject>

#include <QtDBus/QDBusContext>

#include <Decibel/Types>

#include <QtTapioca/ContactBase>

namespace QtTapioca
{
class Connection;
}

class AccountManager;
class ContactManagerPrivate;

/**
 * @brief The ContactManager is a class interfacing to the desktop environment's
 *        contact data.
 *
 * The ContactManager is used by Decibel to query and update contact information
 * from the desktop environment's contact database. It does not expose an
 * interface to the contact database itself but makes it possible for an
 * application developer to interface with the ContactManager using the IDs
 * defined in the desktop environment's PIM database.
 *
 * The ContactManager leaves all the signaling of changes to the desktop
 * environment's PIM database.
 */
class ContactManager : public QObject, protected QDBusContext
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     * @param account_mgr Pointer to the AccountManager.
     * @param parent Parent Object.
     */
    explicit ContactManager(AccountManager * account_mgr, QObject * parent = 0);
    /** @brief Destructor */
    ~ContactManager();

signals:
    /**
     * @brief Emitted whenever a new channel is opened.
     *
     * This is necessary as Tapioca will only send signals for channels that
     * are created by outside entities.
     *
     * @param connection A pointer to the connection the channel belongs to.
     * @param channel A pointer to the channel.
     * @param incoming true if this channel is incoming.
     */
    void channelOpened(QtTapioca::Connection * connection,
                       QtTapioca::Channel * channel, const bool incoming);

public slots:
    /**
     * @brief Connect to a contact using a given account.
     * @param contact_id The Id of the contact to connect to.
     * @param account_handle A handle to the account to use.
     * @param type The type of connection requested.
     * @param suppress_handler Do not start any additional channel handlers for
     *           this channel if set to true.
     * @returns A channel info object describing the channel set up.
     *
     * This is a basic connection method: It sets up a communication channel
     * between the account and the contact if possible and returns it to the
     * caller.
     */
    Decibel::ChannelInfo
    contactContactUsingAccount(const uint contact_id, const int account_handle,
                               const int type, const bool suppress_handler);

    /**
     * @brief Connect to a somebody using a given url.
     * @param contact_url The URL of the contact to connect to.
     * @param account_handle A handle to the account to use.
     * @param type The type of connection requested.
     * @param suppress_handler Do not start any additional channel handlers for
     *           this channel if set to true.
     * @returns A channel info object describing the channel set up.
     *
     * This is a basic connection method: It sets up a communication channel
     * between the account and a url if possible and returns this channel to the
     * caller.
     */
    Decibel::ChannelInfo
    contactUrlUsingAccount(const QString & contact_url, const int account_handle,
                           const int type, const bool suppress_handler);

    /**
     * @brief Connect to a contact URL using any account available.
     * @param contact_url The URL of the contact to connect to.
     * @param type The type of connection requested.
     * @param suppress_handler Do not start any additional channel handlers for
     *           this channel if set to true.
     * @return A channel info object describing the channel set up.
     *
     * This connection method lets Decibel decide which of the accounts to use
     * to connect to the given URL. It sets up a communication channel between
     * an account it selected and the url if possible and returns this channel
     * to the caller.
     */
    Decibel::ChannelInfo
    contactUrl(const QString & contact_url, const int type,
               const bool suppress_handler);


private slots:
    void onConnectionOpened(QtTapioca::Connection *);
    void onConnectionClosed(QtTapioca::Connection *);
    void onPresenceUpdated(QtTapioca::ContactBase *,
                           QtTapioca::ContactBase::Presence,
                           const QString &);

    void onNewChannelCreated();

private:
    ContactManagerPrivate * const d;
};

#endif
