/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/file_test_data_source.hpp
 *
 * @brief [LEVEL: beta] @ref diagnostics::unittest::File_Test_Data_Source class
 *
 * $Id: file_test_data_source.hpp,v 1.7 2005/06/23 09:54:25 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/test_system/file_test_data_source.t.cpp
 */

#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__FILE_TEST_DATA_SOURCE_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__FILE_TEST_DATA_SOURCE_HPP__INCLUDE_GUARD

// used as base class
#include <diagnostics/unittest/test_system/test_data_source.hpp>

// used in the implementation by value
#include <map>

// used in the implementation by value
#include <string>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

/**
 * @class File_Test_Data_Source diagnostics/unittest/test_system/file_test_data_source.hpp
 *
 * @brief A @ref Test_Data_Source implementation which loads from and
 * stores to a file.
 *
 * @nosubgrouping
 */
class File_Test_Data_Source :
    public Test_Data_Source
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef ::std::map< ::std::string, ::std::string> Data_t;
    typedef File_Test_Data_Source Self;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Disabled Creation
     * @{
     */
private:
    File_Test_Data_Source(Self const &);
    Self & operator=(Self const &);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief noop
     *
     * @throw never
     */
    virtual ~File_Test_Data_Source();

    /**
     * @brief sets is_initialized()==false
     *
     * @throw never
     */
    File_Test_Data_Source();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Configuration
     * @{
     */
public:
    /**
     * @brief Initilizes with a file_name but without any entries. Useful for 
     * starting a new file. Throws aways all old information!
     *
     * @post POST.is_initialized()==true
     * @post POST.is_modified()==false
     * @post POST.file_name()==@a file_name
     * @post forall A POST.exists_entry(A)==false
     *
     * @throw never
     */
    void init(::std::string const & file_name);

    /**
     * @brief The File_Test_Data_Source is initialized to @a
     * file_name and with an EMPTY set of test data entries.
     *
     * @post POST.is_initialized()==true
     * @post POST.is_modified()==false
     * @post POST.file_name()==@a file_name
     * @post POST contains no entries
     *
     * @throw never
     */
    void load(::std::string const & file_name);

    /**
     * @brief Tries to open @a file_name and to save its current
     * entries to it. No backup is made before the writing is started.
     *
     * @pre is_initialized()==true
     * @pre (is_modified()==true || @a file_name!=file_name()) --> @a file_name can be opened, enough space exists
     *
     * @post POST.is_modified()==false
     * @post POST.file_name()==@a file_name
     * @post PRE.is_modified()==true || PRE.file_name()!=@ a file_name
     * --> a file @a file_name is created/overwritten with the entries
     * of this
     *
     * @throw Test_System_Exception if the preconditions are not met
     * (nothing changed -- however, the file @a file_name is an
     * undefined state)
     */
    void save(::std::string const & file_name);

    /**
     * @brief Tries to open @a file_name and to save its current
     * entries to it. No backup is made before the writing is started.
     *
     * @pre is_initialized()==true
     * @pre (is_modified()==true || @a file_name!=file_name() || @a force) 
	 *       --> @a file_name can be opened, enough space exists
     *
     * @post POST.is_modified()==false
     * @post POST.file_name()==@a file_name
     * @post (PRE.is_modified()==true || PRE.file_name()!=@ a file_name || @a froce)
     * --> a file @a file_name is created/overwritten with the entries
     * of this
     *
     * @throw Test_System_Exception if the preconditions are not met
     * (nothing changed -- however, the file @a file_name is an
     * undefined state)
     */
    void save(::std::string const & file_name, bool const force);

    /**
     * @brief true if the set of entries has been modified by a call
     * to @ref set_entry
     *
     * @pre is_initialized()==true
     *
     * @throw if the precondition is not met.
     */
    bool is_modified() const;

    /**
     * @brief the currently associated file_name.
     *
     * @pre is_initialized()==true
     *
     * @throw if the precondition is not met.
     */
    ::std::string const & file_name() const;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Implementation of the @ref Test_Data_Source interface
     * @{
     */
public:
    /**
     * @brief returns true if the contained set of entries is in a
     * defined state. 
     *
     * @throw never
     */
    virtual bool is_initialized() const;

    /**
     * @brief true iff an entry with @a id exists.
     *
     * @pre is_initialized()==true
     *
     * @throw Test_System_Exception if the precondition is not met
     */
    virtual bool exists_entry(::std::string const & id) const;
    
    /**
     * @brief retuns the entry with @a id 
     *
     * @pre is_initialized()==true
     * @pre exists_entry(@a id)==true
     *
     * @throw Test_System_Exception if a precondition is not met
     */
    virtual ::std::string const & get_entry(::std::string const & id) const;
    
    /**
     * @brief sets the string with @a id to @a value
     *
     * @pre is_initialized()==true
     * @post POST.get_entry(@a id)==@a value
     * @post POST.is_modified()==true
     *
     * @throw Test_System_Exception if the precondition is not met.
     */
    virtual void set_entry(::std::string const & id, ::std::string const & value);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Encapsulated State
     * @{
     */
private:
    bool m_initialized;
    ::std::string m_file_name;
    bool m_modified;
    Data_t m_data;
    // @}
};



UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


#endif
// vim:ts=4:sw=4
