/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/util/diff_generator.cpp
 *
 * @brief [LEVEL: alpha] Implementation of @ref
 * diagnostics::internal::Diff_Generator
 *
 * $Id: diff_generator.cpp 786 2007-03-01 21:48:44Z tautschn $
 * 
 * @author Piotr Esden-Tempski
 */

#include <diagnostics/util/diff_generator.hpp>

/*
 * This is a hack: diagnostics is not allowed to log itself.
 * For debugging, we used the following macros, which worked fine --
 * but this construction is not guaranteed to work again. 
 *
 * DO NOT EVER USE THIS IN A PRODUCTION BUILD!
 */
#if 0
#	include <diagnostics/macros/front_end.hpp>
#	define DIFF_GENERATOR_AUDIT 1
#	define DIFF_GENERATOR_AUDIT_BLOCK_GUARD(TXT) DIAGNOSTICS_AUDIT_BLOCK_GUARD(TXT)
#	define DIFF_GENERATOR_AUDIT_TRACE(TXT) DIAGNOSTICS_AUDIT_TRACE(TXT)
#else
#	define DIFF_GERERATOR_AUDIT 0
#	define DIFF_GENERATOR_AUDIT_BLOCK_GUARD(TXT)
#	define DIFF_GENERATOR_AUDIT_TRACE(TXT)
#endif

DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;

Diff_Generator::Diff_Generator()
    : m_diff_done(false)
{
    DIFF_GENERATOR_AUDIT_BLOCK_GUARD("Diff_Generator");
}

Diff_Generator::~Diff_Generator()
{
    DIFF_GENERATOR_AUDIT_BLOCK_GUARD("~Diff_Generator");
}

/* Interface methods */
void Diff_Generator::texta(::std::string const & texta)
{
    DIFF_GENERATOR_AUDIT_BLOCK_GUARD("texta");

    m_diff_done=false;

    /* Preparing texta */
    m_vtexta.clear();
    m_hasha.clear();
    p_split_hash(m_vtexta, m_hasha, texta);
}

void Diff_Generator::textb(::std::string const & textb)
{
    DIFF_GENERATOR_AUDIT_BLOCK_GUARD("textb");

    m_diff_done=false;

    /* Preparing textb */
    m_vtextb.clear();
    m_hashb.clear();
    p_split_hash(m_vtextb, m_hashb, textb);
}

/* Private methods */
/* Splitting the input text around '\n' and
 * generating hashes for every line using
 * hashpjw algorithm by P. J. Weinberg
 */
void Diff_Generator::p_split_hash(VText_t & vtext, Hash_t & hashmap, ::std::string const &text)
{
    DIFF_GENERATOR_AUDIT_BLOCK_GUARD("p_split_hash");
    unsigned long int h(0), g;
    unsigned long int const hash_prime(211);
    ::std::string::const_iterator begin(text.begin());
    ::std::string::const_iterator const end(text.end());
    ::std::string line;

    // If text is empty leave vtext and hashmap empty
    DIFF_GENERATOR_AUDIT_TRACE("check if input empty");
    if(text.size()==0){
	return;
    }

    // @todo slow loop -- improve
    DIFF_GENERATOR_AUDIT_TRACE("main loop");
    for(;begin!=end;++begin) {
	if(*begin=='\n'){
	    DIFF_GENERATOR_AUDIT_TRACE("adding line: \"" + line + "\"");
	    // insert a new string
	    vtext.push_back(line);
	    line.clear();
	    // insert string hashvalue and reset hash algorithm
	    hashmap.push_back(h%hash_prime);
	    h = 0;
	}else{
	    line += *begin;

	    // hash calculation
	    h = (h << 4) + *begin;
	    if((g=h&0xF0000000)){
		h ^= g>>24;
		h ^= g;
	    }
	}
    }

    vtext.push_back(line);
    hashmap.push_back(h%hash_prime);
}

/* Use Largest Common Subsequence algorithm to create a diff
 */
void Diff_Generator::p_do_diff() const 
{
    DIFF_GENERATOR_AUDIT_BLOCK_GUARD("p_do_diff");

    if(m_diff_done) return;

    // reset data immediately
    m_diff_tupel.clear();

    // at the end of this procedure, the diff will be done
    m_diff_done=true;
	
    // in either case, m_diff_tupel must be empty:
    // m_diff_tupel contains matches and in these two cases,
    // there are no matches.
    if(m_vtexta.size() == 0) return;
    if(m_vtextb.size() == 0) return;

    // local variables
    int i, j;
    int const m(static_cast<int>(m_hasha.size()));
    int const n(static_cast<int>(m_hashb.size()));
    int const size((m+1)*(n+1));
    bool * const bit_matrix(new bool[size]);
	
    // Prepare the bit_matrix
    bool * bmi_begin = bit_matrix;
    bool * const bmi_end = &bit_matrix[size];
    for(; bmi_begin!=bmi_end; ++bmi_begin) *bmi_begin = true;


    ::std::vector<unsigned int> ci(n+1,0);
    ::std::vector<unsigned int> ci1(n+1,0);

    // Create ci and ci1 iterators
    ::std::vector<unsigned int>::const_iterator const cie_const(ci.end());

    // Create hash map iterators
    Hash_t::const_iterator hasha_e(m_hasha.end());
    Hash_t::const_iterator hashb_e;
    Hash_t::const_iterator hasha_b(m_hasha.begin());
    bool * bmi_e_i = &bit_matrix[size-1];

    // Prepare LCS matrix
    DIFF_GENERATOR_AUDIT_TRACE("main loop");
    for(; hasha_e >= hasha_b; --hasha_e, bmi_e_i-=(n+1)){
	// Move one row down in the virtual matrix
	{
	    ::std::vector<unsigned int>::iterator cib(ci.begin());
	    ::std::vector<unsigned int>::iterator ci1b(ci1.begin());
	    for(; cib!=cie_const; ++cib, ++ci1b){
		*ci1b = *cib;
		*cib = 0;
	    }
	}
		

	// Populate the bit_matrix
	{
	    Hash_t::const_iterator const hashb_b(m_hashb.begin());
	    hashb_e = m_hashb.end();
	    bool * bmi_e_j = bmi_e_i;
	    ::std::vector<unsigned int>::iterator cie(ci.end() - 1);
	    ::std::vector<unsigned int>::iterator ci1e(ci1.end() - 1);
	    for(; hashb_e >= hashb_b; --hashb_e, --cie, --ci1e, --bmi_e_j) {
		if(*hasha_e == *hashb_e) 
		    *cie = 1 + *(ci1e+1);
		else {
		    if(*ci1e > *(cie+1)) {
			*bmi_e_j = true;
			*cie = *ci1e;
		    } else {
			*bmi_e_j = false;
			*cie = *(ci1e+1);
		    }
		}
	    }
	}
    }

    // Create diff tupel
    i=0;
    j=0;
    hasha_b = m_hasha.begin();
    hasha_e = m_hasha.end();
    Hash_t::const_iterator hashb_b(m_hashb.begin());
    hashb_e = m_hashb.end();
	
    DIFF_GENERATOR_AUDIT_TRACE("tupel loop");
    bmi_begin = bit_matrix;
    while(hasha_b != hasha_e && hashb_b != hashb_e) {
#		if DIFF_GENERATOR_AUDIT
	::std::stringstream diagnostics_audit_trace_output;
	diagnostics_audit_trace_output << "processing hashes: " << "[" << *hasha_b << "," << *hashb_b << "]";
#		endif
	DIFF_GENERATOR_AUDIT_TRACE(diagnostics_audit_trace_output.str());
	if(*hasha_b == *hashb_b) {
	    DIFF_GENERATOR_AUDIT_TRACE("match");
	    m_diff_tupel.push_back(Match_t(i, j));
	    bmi_begin+=(n+1); ++hasha_b; ++i;
	    ++bmi_begin; ++hashb_b; ++j;
	} else {
	    DIFF_GENERATOR_AUDIT_TRACE("no match");
	    if(*bmi_begin) {
		bmi_begin+=(n+1); ++hasha_b; ++i;
	    } else {
		++bmi_begin; ++hashb_b; ++j;
	    }
	}
    }

    // Cleanup
    delete[] bit_matrix;
}

void Diff_Generator::diff_text(::std::string & diff_text, bool const use_color) const
{
    DIFF_GENERATOR_AUDIT_BLOCK_GUARD("diff_text");

    p_do_diff();
	
    unsigned int first, second;
    VMatch_t::const_iterator dt_b(m_diff_tupel.begin());
    VMatch_t::const_iterator const dt_e(m_diff_tupel.end());
    VText_t::const_iterator vta_b(m_vtexta.begin());
    VText_t::const_iterator const vta_e(m_vtexta.end());
    VText_t::const_iterator vtb_b(m_vtextb.begin());
    VText_t::const_iterator const vtb_e(m_vtextb.end());

    diff_text = "";

    /*
     * Main loop running through the matches and generating the appropriate
     * '+', '-' and ' ' lines.
     * 
     * It is possible that this loop is not entered when there is no match
     * between the two files.
     */
    for(;dt_b!=dt_e; ++dt_b){
	first = dt_b->first;
	second = dt_b->second;
#		if DIFF_GENERATOR_AUDIT
	::std::stringstream diff_tupel_s;
	diff_tupel_s << "[" << first << "," << second << "]";
#		endif
	DIFF_GENERATOR_AUDIT_TRACE("processing tupel: " + diff_tupel_s.str());
	VText_t::const_iterator vta_block(m_vtexta.begin()+first);
	VText_t::const_iterator vtb_block(m_vtextb.begin()+second);

	for(;vta_b!=vta_block; ++vta_b)
	    p_append_composed_line(diff_text, '-', *vta_b, use_color);

	for(;vtb_b!=vtb_block; ++vtb_b)
	    p_append_composed_line(diff_text, '+', *vtb_b, use_color);

	p_append_composed_line(diff_text, ' ', *vta_b, use_color);
	++vta_b;
	++vtb_b;
    }

    /*
     * Finishing up. If there are still lines that we have not printed in one
     * of the files we do so.
     */
    DIFF_GENERATOR_AUDIT_TRACE("finishing");
    for(;vta_b!=vta_e; ++vta_b)
	p_append_composed_line(diff_text, '-', *vta_b, use_color);

    for(;vtb_b!=vtb_e; ++vtb_b)
	p_append_composed_line(diff_text, '+', *vtb_b, use_color);

    // Remove the last newline in the text. It is one too much
    if(!diff_text.empty()) diff_text.erase(diff_text.size()-1,1);
}

void Diff_Generator::p_append_composed_line(::std::string &target, char prefix, ::std::string const &line, bool const use_color)
{
    char const * const ansi_red = "\033[31m";
    char const * const ansi_green = "\033[32m";
    char const * const ansi_reset = "\033[0m";

    char prefixs[2];

    prefixs[0] = prefix;
    prefixs[1] = '\0';
	
    DIFF_GENERATOR_AUDIT_TRACE(::std::string("p_append_composed_line: \"") + prefixs + line + "\"");

    if(use_color){
	switch(prefix){
	    case '+':
		target += ansi_green;
		target += prefixs + line + ansi_reset + "\n";
		break;
	    case '-':
		target += ansi_red;
		target += prefixs + line + ansi_reset + "\n";
		break;
	    case ' ':
		target += prefixs + line + "\n";
		break;
	}
    }else{
	target += prefixs + line + "\n";
    }
}


INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

// vim:ts=4:sw=4
