#
# Client.py - DITrack Client interface module
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: Client.py 1943 2007-08-24 23:16:27Z vss $
# $HeadURL: https://127.0.0.1/ditrack/src/tags/0.7/DITrack/Client.py $
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

import DITrack.DB.Exceptions

#
# Generic client exception class visible to the application.
#
class Error(Exception):
    """
    The following data members are available:

    message
        One-line description of a problem. No new lines, not terminated with
        a period. Suitable to be displayed in a pop-up window whatsoever.
        Always available.

    details
        A [possibly empty] list of strings describing the problem in a greater
        detail. Strings are not terminated with newlines.
    """

    def __init__(self, message, details):
        self.message = message
        self.details = details


class Client:
    """
    A DITrack client class. Provides high-level API.
    """

    def __init__(self, db):
        """
        Initialize the client. DB is the DITrack.DB.Common.Database object.
        """
        self.db = db

    def issues(self, filters=None):
        """
        Returns a list of tuples (ID, ISSUE) for each issue in the database,
        filtering them with FILTERS (objects of DITrack.DB.Common.Filter). If any of
        the filters match, the issue matches.
        """

        res = []

        for id, issue in self.db.issues():
            matches = None

            if filters:
                for f in filters:
                    if f.matches(issue):
                        matches = True

                if not matches: continue

            res.append((id, issue))

        return res

    def update(self, maxage=None):
        """
        Update the working copy (sync from the repository).
        
        If MAXAGE is None, the update takes place unconditionally. Otherwise
        if the database was updated less than MAXAGE seconds back, the update
        is skipped.

        On error raises DITrack.Client.Error().

        On success may return:

            None
                No update took place.

            not None
                An instance of DITrack.Client.UpdateStatus containing the
                details of the update.
        """

        try:
            us = self.db.update(maxage)
        except DITrack.DB.Exceptions.BackendError, e:
            raise Error(
                "Database update failed: %s" % e.backend_e.message,
                e.backend_e.details
            )


        if us:
            return UpdateStatus(us)
        else:
            return None

class UpdateStatus:
    """
    Class representing the result of an update.

    Not to be instantiated by an application.
    """

    def __init__(self, backend_us):
        """
        Given backend-specific update status BACKEND_US create an object
        describing the update in a backend-agnostic way.
        """

        assert backend_us is not None

        # XXX: we should ensure that backend_us is a descendant of 
        # DITrack.Backend.Common.UpdateStatus.

        # For now we just store the backend-specific object. This will change
        # later when multiple backends come into play.
        self._status = backend_us

    def __str__(self):
        """
        Return the summary of the update in a free form.

        XXX: To be changed later, the format of the output should not be relied
        upon.
        """

        return str(self._status)
