#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_date.h"
#include "edv_obj.h"
#include "edv_status_bar.h"
#include "archiver.h"
#include "archiver_contents_list.h"
#include "browser.h"
#include "browser_dir_tree.h"
#include "browser_contents_list.h"
#include "recbin_contents_list.h"
#include "imbr.h"
#include "imbr_tlist.h"
#include "find_win.h"
#include "find_win_cb.h"
#include "endeavour2.h"
#include "edv_find.h"
#include "edv_open.h"
#include "edv_op.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"


void EDVFindWinListItemDestroyCB(gpointer data);

gint EDVFindWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

gint EDVFindWinCrossingCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint EDVFindWinButtonCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

void EDVFindWinDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void EDVFindWinDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);

void EDVFindWinComboActivateCB(GtkWidget *widget, gpointer data);

void EDVFindWinBrowseLocationCB(GtkWidget *widget, gpointer data);

void EDVFindWinFindByChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
);

gint EDVFindWinFindProgressCB(
	const gchar *path, gfloat progress, gpointer data
);

void EDVFindWinFindMatchExcerptCB(
	const gchar *path, struct stat *lstat_buf,
	const gchar *excerpt, gint line_index,
	gpointer data
);
void EDVFindWinFindMatchCB(
	const gchar *path, struct stat *lstat_buf,
	gpointer data
);

void EDVFindWinClickColumnCB(
	GtkCList *clist, gint column, gpointer data
);
void EDVFindWinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void EDVFindWinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

void EDVFindWinSearchCB(GtkWidget *widget, gpointer data);
void EDVFindWinStopCB(GtkWidget *widget, gpointer data);
void EDVFindWinClearCB(GtkWidget *widget, gpointer data);
void EDVFindWinCloseCB(GtkWidget *widget, gpointer data);
void EDVFindWinOpenCB(GtkWidget *widget, gpointer data);
void EDVFindWinOpenWithCB(GtkWidget *widget, gpointer data);
void EDVFindWinGotoCB(GtkWidget *widget, gpointer data);

void EDVFindWinWriteProtectChangedCB(
	edv_find_win_struct *fw, gboolean state
);
void EDVFindWinReconfiguredNotifyCB(edv_find_win_struct *fw);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Results GtkCList item "destroy" signal callback.
 */
void EDVFindWinListItemDestroyCB(gpointer data)
{
	EDVObjectDelete(EDV_OBJECT(data));
}

/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
gint EDVFindWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	EDVFindWinCloseCB(widget, data);
	return(TRUE);
}

/*
 *	Any GtkWidget "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
gint EDVFindWinCrossingCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	const gchar *msg = NULL;
	gint etype;
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if((widget == NULL) || (crossing == NULL) || (fw == NULL))
	    return(FALSE);

	etype = (gint)crossing->type;

	if(etype == GDK_ENTER_NOTIFY)
	{
	    if(widget == fw->case_sensitive_check)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Especifica si los iguales de cuerda no deben ser el casos sensibles ni"
#elif defined(PROG_LANGUAGE_FRENCH)
"Spcifie si les allumettes de ficelle doivent tre des cas sensibles ou pas"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gibt an, wenn schnur gegenstcke fall empfindlich oder nicht sein sollen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Non specifica se i fiammiferi di cordicella dovrebbero essere dei casi sensibili o"
#elif defined(PROG_LANGUAGE_DUTCH)
"Specificeert indien koord stellen geval gevoelig of niet zouden moeten zijn"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Especifica se partidas de barbante devem ser caso sensvel ou no"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Spesifiserer om snorkamper er tilfelle sensitiv eller ikke"
#else
"Specifies if string matches should be case sensitive or not"
#endif
		;
	    else if(widget == fw->recursive_check)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Especifica si el proceso de la bsqueda debe recurse en subdirectorios"
#elif defined(PROG_LANGUAGE_FRENCH)
"Spcifie si le procd de recherche devrait recurse dans subdirectories"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gibt an, wenn das suche verfahren recurse in unterverzeichnisse soll"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Specifica se il processo di ricerca dovrebbe il recurse nelle directory secondarie"
#elif defined(PROG_LANGUAGE_DUTCH)
"Specificeert indien het zoektocht proces recurse in subdirectories zal"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Especifica se o processo de busca devia recurse em subdirectories"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Spesifiserer om letingssom prosessen skal recurse inn i subdirectories"
#else
"Specifies if the search process should recurse into subdirectories"
#endif
		;
	    else if(widget == fw->search_btn)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Empiece la bsqueda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commencer la recherche"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fangen sie die suche an"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cominciare la ricerca"
#elif defined(PROG_LANGUAGE_DUTCH)
"Begin de zoektocht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comece a busca"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Start letingen"
#else
"Start the search"
#endif
		;
	    else if(widget == fw->stop_btn)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Pare el procedimiento actual del hallazgo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Arrter la procdure actuelle de dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Halten sie das jetzige fund verfahren auf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Fermare la corrente trova la procedura"
#elif defined(PROG_LANGUAGE_DUTCH)
"Stop de huidig vondst procedure"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pare a corrente achar procedimento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stans den nvrendee funnprosedyre"
#else
"Stop the current find procedure"
#endif
		;
	    else if(widget == fw->clear_btn)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"La cuerda clara de la bsqueda y la lista de resultados"
#elif defined(PROG_LANGUAGE_FRENCH)
"La ficelle clairs de recherche et la liste de rsultats"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klare suche schnur und ergebnisse fhren auf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'elenco di cordicella di ricerca e risultati chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder zoektocht koord en resultaten sommen op"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Barbante clara de busca e lista de resultados"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar letingssnor og resultater lister opp"
#else
"Clear search string and results list"
#endif
		;
	    else if(widget == fw->close_btn)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre esta ventana"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fermer cette fentre"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schlieen sie dieses fenster"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino questa finestra"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sluit deze raam"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo esta janela"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stenge dette vinduet"
#else
"Close this window"
#endif
		;
	    else if(widget == fw->open_btn)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir archivo escogido"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir l'objet choisi"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen ausgewhlten objekt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto scelto oggetto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open geselecteerd voorwerp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Objeto selecionado aberto"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn valgt ut objekt"
#else
"Open selected object"
#endif
		;
	    else if(widget == fw->open_with_btn)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir archivo escogido utilizando un mtodo especfico"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir l'objet choisi utilisant une mthode spcifique"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen ausgewhlten objekt, eine spezifische methode zu benutzen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto scelto oggetto usando uno specifico metodo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open geselecteerd voorwerp een specifieke methode te gebruiken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Objeto selecionado aberto usando um mtodo especfico"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn valgt ut objekt bruke en spesifikk metode"
#else
"Open selected object using a specific method"
#endif
		;
	    else if(widget == fw->goto_btn)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya a escogido se opone"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller  l'objet choisi"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen sie zu ausgewhltem objekt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare all'oggetto scelto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga te geselecteerd voorwerp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V a objeto selecionado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra til valgt ut objekt"
#else
"Go to selected object"
#endif
		;
	}
	EDVStatusBarMessage(fw->status_bar, msg, FALSE);

	return(FALSE);
}

/*
 *	Any GtkWidget "button_press_event" or "button_release_event"
 *	signal callback.
 */
gint EDVFindWinButtonCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	gboolean is_press;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if((widget == NULL) || (button == NULL) || (fw == NULL))
	    return(status);

	if(fw->processing)
	    return(status);

	core = fw->core;
	cfg_list = core->cfg_list;

	etype = (gint)button->type;
	is_press = (etype == GDK_BUTTON_PRESS) ? TRUE : FALSE;

	/* Check which ctree this signal is for */
	if(widget == fw->results_clist)
	{
	    gint row, column;
	    gint rows_selected = 0, selected_row = -1;
	    GList *glist;
	    GtkCList *clist = GTK_CLIST(widget);


	    /* Find row and column based on given coordinates */
	    if(!gtk_clist_get_selection_info(
		clist, (gint)button->x, (gint)button->y, &row, &column
	    ))
	    {
		row = -1;
		column = 0;
	    }

	    /* Get number of selected rows and highest selected row */
	    glist = clist->selection;
	    while(glist != NULL)
	    {
		rows_selected++;
		selected_row = (gint)glist->data;
		glist = g_list_next(glist);
	    }

	    /* If button press then grab focus as needed */
	    if(is_press && !GTK_WIDGET_HAS_FOCUS(widget))
		gtk_widget_grab_focus(widget);

	    /* Handle by button number */
	    switch(button->button)
	    {
	      case 3:
		if(is_press)
		{
		    GtkMenu *menu = (GtkMenu *)fw->results_clist_menu;

		    /* Select item before mapping menu? */
		    if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
		       (row >= 0) && (row < clist->rows)
		    )
		    {
			/* Select the row that the button was pressed
			 * over
			 *
			 * if no key modifiers are held then this will
			 * also unselect all previously selected rows
			 */
			gtk_clist_freeze(clist);
			if(!(button->state & GDK_CONTROL_MASK) &&
			   !(button->state & GDK_SHIFT_MASK)
			)
			    gtk_clist_unselect_all(clist);
			clist->focus_row = row;
			gtk_clist_select_row(clist, row, 0);
			gtk_clist_thaw(clist);
		    }

		    /* Update all menus and map right click menu */
		    EDVFindWinUpdateMenus(fw);
		    if(menu != NULL)
			gtk_menu_popup(
			    menu, NULL, NULL,
			    NULL, NULL,
			    button->button, button->time
			);
		}
		status = TRUE;
		break;

	      case 2:
		if(is_press)
		{
#if 0
		    if((row >= 0) && (row < clist->rows))
			EDVFindWinContentsDoFPromptRename(
			    fw, row, column
			);
#endif
		}
		status = TRUE;
		break;

	      case 1:
		/* Double click? */
		if(etype == GDK_2BUTTON_PRESS)
		{
		    if((row >= 0) && (row < clist->rows))
		    {
			EDVFindWinOpenCB(NULL, fw);
			status = TRUE;
		    }
		}
		break;
	    }
	}

	return(status);
}


/*
 *	GtkCList DND "drag_data_get" signal callback.
 */
void EDVFindWinDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint row;
	GList *glist, *url_list;
	GtkCList *clist;
	url_struct *url;
	edv_location_type location_type;
	edv_object_struct *obj;
	edv_core_struct *core;
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if((dc == NULL) || (fw == NULL))
	    return;

	if(fw->freeze_count > 0)
	    return;

	location_type = fw->location_type;
	clist = GTK_CLIST(fw->results_clist);
	core = fw->core;

	/* Generate a list of URLs from the selected rows */
	url_list = NULL;
	switch(fw->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    for(glist = clist->selection;
	        glist != NULL;
	        glist = g_list_next(glist)
	    )
	    {
		row = (gint)glist->data;
		obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
		if(obj == NULL)
		    continue;

		url = URLNew();
		url->path = STRDUP(obj->full_path);
		url_list = g_list_append(url_list, url);
	    }
	    break;

	  case EDV_LOCATION_TYPE_RECBIN:
	    for(glist = clist->selection;
	        glist != NULL;
	        glist = g_list_next(glist)
	    )
	    {
	        row = (gint)glist->data;
		obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
	        if(obj == NULL)
		    continue;

		url = URLNew();

		/* Set the URL's path as the recycled object's
		 * index, the index will be obtained from
		 * obj->index
		 */
		url->path = g_strdup_printf(
		    "%u", (guint)obj->index
		);

		/* For recycled objects, the path argument must be
		 * set as the recycled object's original name
		 */
		url->path_arg = STRDUP(obj->name);

		url_list = g_list_append(url_list, url);
	    }
	    break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
	    url = URLNew();
	    url->path = STRDUP(EDVFindWinCurrentLocation(fw));
	    url_list = g_list_append(url_list, url);

	    for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		row = (gint)glist->data;
		obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
		if(obj == NULL)
		    continue;

		url = URLNew();
		url->path = STRDUP(obj->full_path);
		url_list = g_list_append(url_list, url);
	    }
	    break;
	}

	/* Encode the DDE buffer from the URL list */
	if(url_list != NULL)
	{
	    gint buf_len;
	    guint8 *buf = URLEncode(url_list, &buf_len);
	    if(buf != NULL)
	    {
		/* Send out the buffer */
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,			/* Bits Per Character */
		    buf,		/* Data */
		    buf_len		/* Length */
		);
		data_sent = TRUE;
		g_free(buf);
	    }
	}

	/* Delete the URL list */
	if(url_list != NULL)
	{
	    g_list_foreach(url_list, (GFunc)URLDelete, NULL);
	    g_list_free(url_list);
	}

	/* If failed to send out data then respond with error */
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		s,			/* Data */
		STRLEN(s)		/* Length */
	    );
	    data_sent = TRUE;
	}
}

/*
 *	GtkCList DND "drag_data_delete" signal callback.
 */
void EDVFindWinDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{

}


/*
 *	Search or Location GtkCombo "activate" signal callback.
 */
void EDVFindWinComboActivateCB(GtkWidget *widget, gpointer data)
{
	EDVFindWinSearchCB(widget, data);
}


/*
 *	Browse location callback.
 */
void EDVFindWinBrowseLocationCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar *cur_location, **paths_list;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if((fw == NULL) || FileBrowserIsQuery())
	    return;

	if(fw->freeze_count > 0)
	    return;

	toplevel = fw->toplevel;
	cur_location = STRDUP(EDVFindWinCurrentLocation(fw));

	EDVFindWinSetBusy(fw, TRUE);

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for a location */
	fw->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Location",
	    "Select", "Cancel",
	    cur_location,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	fw->freeze_count--;

	/* Got response? */
	if(response)
	{
	    const gchar *new_location = (npaths > 0) ?
		paths_list[npaths - 1] : NULL;
	    EDVFindWinSetLocation(fw, new_location, TRUE);
	}

	FileBrowserDeleteTypeList(ftypes_list, nftypes);
	g_free(cur_location);

	EDVFindWinSetBusy(fw, FALSE);
}


/*
 *	Find By changed callback.
 */
void EDVFindWinFindByChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return;

	if(fw->freeze_count > 0)
	    return;

	fw->freeze_count++;

	EDVFindWinUpdateMenus(fw);

	fw->freeze_count--;
}


/*
 *	Find progress callback.
 *
 *	Returns the number of stop counts.
 */
gint EDVFindWinFindProgressCB(
	const gchar *path, gfloat progress, gpointer data
)
{
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return(0);

	if(fw->stop_count > 0)
	    return(fw->stop_count);

	EDVStatusBarMessage(fw->status_bar, path, FALSE);
	EDVStatusBarProgress(fw->status_bar, progress, TRUE);

	return(0);
}

/*
 *	Find procedure matched excerpt callback.
 */
void EDVFindWinFindMatchExcerptCB(
	const gchar *path, struct stat *lstat_buf,
	const gchar *excerpt, gint line_index,
	gpointer data
)
{
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return;

	EDVFindWinListAppend(fw, path, lstat_buf, excerpt, line_index);
}

/*
 *	Find procedure matched callback.
 */
void EDVFindWinFindMatchCB(
	const gchar *path, struct stat *lstat_buf, gpointer data
)
{
	EDVFindWinFindMatchExcerptCB(path, lstat_buf, NULL, -1, data);
}


/*
 *	GtkCList "click_column" signal callback.
 */
void EDVFindWinClickColumnCB(
	GtkCList *clist, gint column, gpointer data
)
{
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if((clist == NULL) || (fw == NULL))
	    return;

	if(fw->processing || (fw->freeze_count > 0))
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_set_sort_type(
	    clist,
	    (clist->sort_type == GTK_SORT_ASCENDING) ?
		GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
	);
	gtk_clist_set_sort_column(clist, column);
	gtk_clist_sort(clist);
	gtk_clist_thaw(clist);
}

/*
 *	GtkCList "select_row" signal callback.
 */
void EDVFindWinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if((clist == NULL) || (fw == NULL))
	    return;

	if(fw->freeze_count > 0)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == fw->results_clist)
	{
	    const gint nselected = g_list_length(clist->selection);
	    const edv_object_struct *obj = EDV_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );

#if 0
	    /* Update the selected row */
	    fw->results_clist_selected_row = row;
#endif

	    if(obj != NULL)
	    {
		/* Update the DND icon for results clist */
		gint i;
		gchar *text = NULL;
		guint8 spacing = 0;
		GdkPixmap *pixmap = NULL;
		GdkBitmap *mask = NULL;

		/* Iterate through each cell of this row, looking for
		 * a useable pixmap
		 */
		for(i = 0; i < clist->columns; i++)
		{
		    switch(gtk_clist_get_cell_type(clist, row, i))
		    {
		      case GTK_CELL_PIXMAP:
			gtk_clist_get_pixmap(
			    clist, row, i,
			    &pixmap, &mask
			);
			break;
		      case GTK_CELL_PIXTEXT:
			gtk_clist_get_pixtext(
			    clist, row, i,
			    &text, &spacing, &pixmap, &mask
			);
			break;
		      case GTK_CELL_TEXT:
		      case GTK_CELL_WIDGET:
		      case GTK_CELL_EMPTY:
			break;
		    }
		    if(pixmap != NULL)
			break;
		}
		/* Got icon pixmap? */
		if(pixmap != NULL)
		{
		    gint w = 15, h = 15;

		    /* Get size of icon pixmap */
		    gdk_window_get_size(pixmap, &w, &h);

		    /* Set DND icon */
		    GUIDNDSetDragIcon(
			pixmap, mask,
			w / 2, h / 2
		    );
		}

		/* Update the status bar message */
		if(!STRISEMPTY(obj->name))
		{
		    gchar *s, *size_str = NULL;
		    const gchar *type_str = NULL;

		    /* Get object type and size strings */
		    switch(obj->type)
		    {
		      case EDV_OBJECT_TYPE_UNKNOWN:
			type_str = "Object";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_FILE:
			type_str = "File";
			size_str = g_strdup_printf(
			    " (%s %s)",
			    EDVSizeStrDelim(obj->size),
			    (obj->size == 1l) ? "byte" : "bytes"
			);
			break;
		      case EDV_OBJECT_TYPE_DIRECTORY:
			type_str = "Directory";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_LINK:
			type_str = "Link";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
			type_str = "Block device";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
			type_str = "Character device";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_FIFO:
			type_str = "FIFO pipe";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_SOCKET:
			type_str = "Socket";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_ERROR:
			type_str = "Error";
			size_str = NULL;
			break;
		    }
		    /* Set the status bar message */
		    if(nselected > 1)
		    {
			gulong total_size = 0l;
			edv_object_struct *obj;
			GList *glist = clist->selection;
			while(glist != NULL)
			{
			    obj = EDV_OBJECT(gtk_clist_get_row_data(
				clist, (gint)glist->data
			    ));
			    if(obj != NULL)
				total_size += obj->size;
			    glist = g_list_next(glist);
			}
			g_free(size_str);
			size_str = g_strdup_printf(
			    "%s %s",
			    EDVSizeStrDelim(total_size),
			    (total_size == 1l) ? "byte" : "bytes"
			);
			s = g_strdup_printf(
			    "%s objects selected (totaling %s)",
			    EDVSizeStrDelim(nselected),
			    size_str
			);
		    }
		    else if(!strcmp((const char *)obj->name, ".."))
			s = g_strdup_printf(
			    "Parent directory selected"
			);
		    else
			s = g_strdup_printf(
			    "%s \"%s\" selected%s",
			    type_str, obj->name,
			    (size_str != NULL) ? size_str : ""
			);
		    EDVStatusBarMessage(
			fw->status_bar, s, FALSE
		    );
		    g_free(s);
		    g_free(size_str);
		}
		else
		{
		    EDVStatusBarMessage(
			fw->status_bar,
			"Object with no name selected",
			FALSE
		    );
		}
	    }

	    /* If the selected row is not fully visible then scroll
	     * to position the selected row in the center of the list
	     */
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    row, -1,			/* Row, column */
		    0.5f, 0.0f			/* Row, column */
		);

/*	    EDVFindWinSetTitle(fw); */
	    EDVFindWinUpdateMenus(fw);
	}
}

/*
 *	GtkCList "unselect_row" signal callback.
 */
void EDVFindWinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if((clist == NULL) || (fw == NULL))
	    return;

	if(fw->freeze_count > 0)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == fw->results_clist)
	{
	    const gint nselected = g_list_length(clist->selection);
	    gchar *s;

	    /* Update the status bar message */
	    if(nselected > 0)
	    {
		gchar *size_str;
		gulong total_size = 0l;
		edv_object_struct *obj;
		GList *glist = clist->selection;
		while(glist != NULL)
		{
		    obj = EDV_OBJECT(gtk_clist_get_row_data(
			clist, (gint)glist->data
		    ));
		    if(obj != NULL)
			total_size += obj->size;
		    glist = g_list_next(glist);
		}
		size_str = g_strdup_printf(
		    "%s %s",
		    EDVSizeStrDelim(total_size),
		    (total_size == 1l) ? "byte" : "bytes"
		);
		s = g_strdup_printf(
		    "%s %s selected (totaling %s)",
		    EDVSizeStrDelim(nselected),
		    (nselected == 1) ? "object" : "objects",
		    size_str
		);
		g_free(size_str);
	    }
	    else
		s = STRDUP("No objects selected");
	    EDVStatusBarMessage(
		fw->status_bar, s, FALSE
	    );
	    g_free(s);

/*	    EDVFindWinSetTitle(fw); */
	    EDVFindWinUpdateMenus(fw);
	}
}


/*
 *	Search callback.
 *
 *	Starts the find procedure.
 */
void EDVFindWinSearchCB(GtkWidget *widget, gpointer data)
{
	gboolean	case_sensitive = FALSE,
			recursive = FALSE;
	gchar		*location,
			*parent_path,
			*search_string;
	gint nmatches;
	edv_find_win_find_by find_by;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return;

	if(fw->processing || (fw->freeze_count > 0))
	    return;

	core = fw->core;
	cfg_list = core->cfg_list;

	fw->processing = TRUE;

	/* Get the search parameters */
	find_by = EDVFindWinCurrentFindBy(fw);
	search_string = STRDUP(EDVFindWinCurrentSearch(fw));

	if(search_string == NULL)
	{
	    fw->processing = FALSE;
	    return;
	}

	/* No wild cards in search string? */
	if((strchr((char *)search_string, '*') == NULL) &&
	   (strchr((char *)search_string, '?') == NULL)
	)
	{
	    /* Add wildcards to the search string for certain find by
	     * methods that need them
	     */
	    if(find_by == EDV_FIND_WIN_FIND_BY_NAME)
	    {
		/* Add wildcards to the search string */
		gchar *s = g_strdup_printf("*%s*", search_string);
		g_free(search_string);
		search_string = s;
	    }
	}

	/* Record the search string on to the search history list */
	EDVFindWinSetSearch(fw, search_string, TRUE);

	/* Get the starting location based on the location type */
	switch(fw->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    /* Get current location as the starting location */
	    location = EDVEvaluatePath(
		NULL, EDVFindWinCurrentLocation(fw)
	    );
	    /* Record starting location history */
	    EDVFindWinSetLocation(fw, location, TRUE);
	    break;

	  case EDV_LOCATION_TYPE_RECBIN:
	    /* Get recycled objects index file as the starting location */
	    location = EDVEvaluatePath(
		NULL, EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX)
	    );
	    /* Record starting location history */
	    parent_path = g_dirname(location);
	    EDVFindWinSetLocation(fw, parent_path, TRUE);
	    g_free(parent_path);
	    break;

	  default:
	    location = NULL;
	    break;
	}

	/* Case sensitive */
	case_sensitive = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    fw->case_sensitive_check
	);

	/* Recursive */
	recursive = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    fw->recursive_check
	);


	/* Begin search */

	fw->stop_count = 0;
	EDVFindWinSetBusy(fw, TRUE);
	EDVFindWinListClear(fw);
	EDVFindWinListResetColumns(fw, find_by);
	EDVFindWinUpdateMenus(fw);

	nmatches = 0;

	/* Search */
	switch(find_by)
	{
	  case EDV_FIND_WIN_FIND_BY_NAME:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByName(
		    location, search_string,
		    recursive,
		    FALSE,			/* Do not follow links */
		    case_sensitive,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByName(
		    location,			/* Recycled objects index file */
		    search_string,
		    case_sensitive,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByName(
		    core,
		    location,			/* Archive */
		    search_string,
		    case_sensitive,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	    }
	    break;

	  case EDV_FIND_WIN_FIND_BY_CONTENT:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByContent(
		    location,
		    search_string,
		    recursive,
		    FALSE,			/* Do not follow links */
		    case_sensitive,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchExcerptCB, fw
		);
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByContent(
		    location,			/* Recycled objects index file */
		    search_string,
		    case_sensitive,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchExcerptCB, fw
		);
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Not supported */
		break;
	    }
	    break;

	  case EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectBySize(
		    location,
		    ATOL(search_string),	/* Size */
		    recursive,
		    TRUE,			/* Follow links */
		    EDV_FIND_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectBySize(
		    location,			/* Recycled objects index file */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectBySize(
		    core,
		    location,			/* Archive */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	    }
	    break;

	  case EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectBySize(
		    location,
		    ATOL(search_string),	/* Size */
		    recursive,
		    TRUE,			/* Follow links */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectBySize(
		    location,			/* Recycled objects index file */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectBySize(
		    core,
		    location,			/* Archive */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	    }
	    break;

	  case EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectBySize(
		    location,
		    ATOL(search_string),	/* Size */
		    recursive,
		    TRUE,			/* Follow links */
		    EDV_FIND_LESS_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectBySize(
		    location,			/* Recycled objects index file */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_LESS_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectBySize(
		    core,
		    location,			/* Archive */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_LESS_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	    }
	    break;

	  case EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectBySize(
		    location,
		    ATOL(search_string),	/* Size */
		    recursive,
		    TRUE,			/* Follow links */
		    EDV_FIND_GREATER_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectBySize(
		    location,			/* Recycled objects index file */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_GREATER_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectBySize(
		    core,
		    location,			/* Archive */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_GREATER_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	    }
	    break;

	  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByModifyTime(
		    location,
		    EDVDateParseEPOCH(search_string),	/* Time */
		    recursive,
		    TRUE,			/* Follow links */
		    EDV_FIND_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByModifyTime(
		    location,			/* Recycled objects index file */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByModifyTime(
		    core,
		    location,			/* Archive */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	    }
	    break;

	  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByModifyTime(
		    location,
		    EDVDateParseEPOCH(search_string),	/* Time */
		    recursive,
		    TRUE,			/* Follow links */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByModifyTime(
		    location,			/* Recycled objects index file */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByModifyTime(
		    core,
		    location,			/* Archive */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	    }
	    break;

	  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByModifyTime(
		    location,
		    EDVDateParseEPOCH(search_string),	/* Time */
		    recursive,
		    TRUE,			/* Follow links */
		    EDV_FIND_LESS_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByModifyTime(
		    location,			/* Recycled objects index file */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_LESS_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByModifyTime(
		    core,
		    location,			/* Archive */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_LESS_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	    }
	    break;

	  case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN:
	    switch(fw->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByModifyTime(
		    location,
		    EDVDateParseEPOCH(search_string),	/* Time */
		    recursive,
		    TRUE,			/* Follow links */
		    EDV_FIND_GREATER_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByModifyTime(
		    location,			/* Recycled objects index file */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_GREATER_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByModifyTime(
		    core,
		    location,			/* Archive */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_GREATER_THAN,
		    EDVFindWinFindProgressCB, fw,
		    EDVFindWinFindMatchCB, fw
		);
		break;
	    }
	    break;
	}



	/* Begin tallying results */
	EDVStatusBarProgress(fw->status_bar, 0.0f, FALSE);
	if(nmatches > 0)
	{
	    gchar *msg = g_strdup_printf(
		"Matched %i object%s%s",
		nmatches,
		(nmatches == 1) ? "" : "s",
		(fw->stop_count > 0) ?
		    " (search interrupted)" : ""
	    );
	    EDVStatusBarMessage(fw->status_bar, msg, FALSE);
	    g_free(msg);
	}
	else
	{
	    EDVStatusBarMessage(
		fw->status_bar,
		(fw->stop_count > 0) ?
"No objects found (search interrupted)" :
"No objects found",
		FALSE
	    );
	}


	/* Set values on find window reflecting end of search */
	fw->stop_count = 0;
	EDVFindWinSetBusy(fw, FALSE);
	fw->processing = FALSE;
	EDVFindWinUpdateMenus(fw);


	g_free(location);
	g_free(search_string);
}

/*
 *	Stop callback.
 */
void EDVFindWinStopCB(GtkWidget *widget, gpointer data)
{
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return;

	/* Skip processing check, this callback is allowed to continue
	 * during processing
	 */

	fw->stop_count++;
}

/*
 *	Clear callback.
 */
void EDVFindWinClearCB(GtkWidget *widget, gpointer data)
{
	GtkCombo *combo;
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return;

	if(fw->processing || (fw->freeze_count > 0))
	    return;

	combo = GTK_COMBO(fw->search_combo);
	gtk_entry_set_text(GTK_ENTRY(combo->entry), "");

	EDVFindWinListClear(fw);
	EDVFindWinUpdateMenus(fw);
}

/*
 *	Close callback.
 */
void EDVFindWinCloseCB(GtkWidget *widget, gpointer data)
{
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return;

	if(fw->processing || (fw->freeze_count > 0))
	    return;

	EDVFindWinSyncConfiguration(fw);
	EDVFindWinUnmap(fw);
}

/*
 *	Open callback.
 */
void EDVFindWinOpenCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GtkWidget *toplevel;
	GtkCList *clist;
	const edv_object_struct *obj;
	edv_core_struct *core;
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return;

	if(fw->processing || (fw->freeze_count > 0))
	    return;

	toplevel = fw->toplevel;
	clist = GTK_CLIST(fw->results_clist);
	core = fw->core;

	/* Unsupported location type for opening object? */
	if(fw->location_type != EDV_LOCATION_TYPE_VFS)
	{
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
#if defined(PROG_LANGUAGE_SPANISH)
"Apertura Fallida",
"El objeto no es un objeto normal de disco, slo objetos normales\n\
de disco se pueden abrir.",
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir Echou",
"L'objet n'est pas un objet de disque normal, les objets de\n\
disque seulement normaux peuvent tre ouverts.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Versagt",
"Das objekt ist kein normales scheibe objekt, nur normale\n\
scheibe, die objekte geffnet werden knnen",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto Fallito",
"L'oggetto non  un oggetto di dischetto normale, gli oggetti\n\
di dischetto soltanto normali possono essere aperti.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Verzuimde",
"Het voorwerp kunnen geen normaal schijf voorwerp, enige\n\
normale schijf voorwerpen geopend worden is.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto Fracassado",
"O objeto no  um objeto normal de disco, objetos s normais de\n\
disco podem ser abertos.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn Sviktet",
"Objektet er ikke et normalt skiveobjekt, bare normal\n\
skiveobjekt pnet.",
#else
"Open Failed",
"The object is not a normal disk object, only normal disk objects\n\
can be opened.",
#endif
		NULL,
		toplevel
	    );
	    return;
	}

	/* Get the last selected row and object */
	row = EDVCListGetSelectedLast(clist, NULL);
	obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));

	if((obj != NULL) ? !STRISEMPTY(obj->full_path) : FALSE)
	{
	    const gchar *full_path = obj->full_path;

	    /* Is it a directory? */
	    if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
	    {
/* TODO  */
	    }
	    else
	    {
		gchar	*stdout_path_rtn = NULL,
			*stderr_path_rtn = NULL;
		GList *paths_list = NULL;

		paths_list = g_list_append(
		    paths_list,
		    STRDUP(full_path)
		);

		EDVFindWinSetBusy(fw, TRUE);
		EDVOpen(
		    core,
		    paths_list,			/* Paths List */
		    NULL,			/* Command Name */
		    toplevel,			/* Toplevel */
		    TRUE,			/* Verbose */
		    &stdout_path_rtn,
		    &stderr_path_rtn
		);
		EDVFindWinSetBusy(fw, FALSE);

		g_list_foreach(paths_list, (GFunc)g_free, NULL);
		g_list_free(paths_list);

		g_free(stdout_path_rtn);
		g_free(stderr_path_rtn);
	    }
	}
}

/*
 *	Open with callback.
 */
void EDVFindWinOpenWithCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GtkWidget *toplevel;
	GtkCList *clist;
	const edv_object_struct *obj;
	edv_core_struct *core;
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return;

	if(fw->processing || (fw->freeze_count > 0))
	    return;

	toplevel = fw->toplevel;
	clist = GTK_CLIST(fw->results_clist);
	core = fw->core;

	/* Unsupported location type for opening object? */
	if(fw->location_type != EDV_LOCATION_TYPE_VFS)
	{
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
#if defined(PROG_LANGUAGE_SPANISH)
"Apertura Fallida",
"El objeto no es un objeto normal de disco, slo objetos normales\n\
de disco se pueden abrir.",
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir Echou",
"L'objet n'est pas un objet de disque normal, les objets de\n\
disque seulement normaux peuvent tre ouverts.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Versagt",
"Das objekt ist kein normales scheibe objekt, nur normale\n\
scheibe, die objekte geffnet werden knnen",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto Fallito",
"L'oggetto non  un oggetto di dischetto normale, gli oggetti\n\
di dischetto soltanto normali possono essere aperti.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Verzuimde",
"Het voorwerp kunnen geen normaal schijf voorwerp, enige\n\
normale schijf voorwerpen geopend worden is.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto Fracassado",
"O objeto no  um objeto normal de disco, objetos s normais de\n\
disco podem ser abertos.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn Sviktet",
"Objektet er ikke et normalt skiveobjekt, bare normal\n\
skiveobjekt pnet.",
#else
"Open Failed",
"The object is not a normal disk object, only normal disk objects\n\
can be opened.",
#endif
		NULL,
		toplevel
	    );
	    return;
	}

	/* Get the last selected row and object */
	row = EDVCListGetSelectedLast(clist, NULL);
	obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));

	if((obj != NULL) ? !STRISEMPTY(obj->full_path) : FALSE)
	{
	    gchar       *stdout_path_rtn = NULL,
			*stderr_path_rtn = NULL;
	    GList *paths_list = NULL;

	    paths_list = g_list_append(
		paths_list,
		STRDUP(obj->full_path)
	    );

	    EDVOpenWith(
		core,
		paths_list,		/* Paths List */
		NULL,			/* Command Name */
		toplevel,		/* Toplevel */
		TRUE,			/* Verbose */
		&stdout_path_rtn,
		&stderr_path_rtn
	    );

	    g_list_foreach(paths_list, (GFunc)g_free, NULL);
	    g_list_free(paths_list);

	    g_free(stdout_path_rtn);
	    g_free(stderr_path_rtn);
	}
}


/*
 *	Goto callback.
 */
void EDVFindWinGotoCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GtkCList *clist;
	const edv_object_struct *obj;
	edv_core_struct *core;
	gchar *parent_dir, *full_path;
	edv_find_win_struct *fw = EDV_FIND_WIN(data);
	if(fw == NULL)
	    return;

	if(fw->processing || (fw->freeze_count > 0))
	    return;

	clist = GTK_CLIST(fw->results_clist);
	core = fw->core;

	/* Get the last selected row and object */
	row = EDVCListGetSelectedLast(clist, NULL);
	obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
	if((obj != NULL) ? STRISEMPTY(obj->full_path) : TRUE)
	    return;

	/* Copy/evaluate the full path of the object */
	full_path = EDVEvaluatePath(NULL, obj->full_path);
	parent_dir = g_dirname(full_path);


	/* Check which window to goto on
	 *
	 * File browser
	 */
	if(fw->browser_num > -1)
	{
	    const gint browser_num = fw->browser_num;
	    edv_browser_struct *browser;

	    if((browser_num >= 0) && (browser_num < core->total_browsers))
		browser = core->browser[browser_num];
	    else
		browser = NULL;

	    if(browser != NULL)
	    {
		const gchar *cur_path = EDVBrowserCurrentLocation(browser);
		GtkWidget *toplevel = browser->toplevel;
		GtkCList *clist = GTK_CLIST(browser->contents_clist);

		EDVBrowserSetBusy(browser, TRUE);
		GUIBlockInput(toplevel, TRUE);

		/* Is the parent directory within the Directory Tree? */
		if(EDVBrowserIsPathFromDirTreeOrigin(browser, parent_dir))
		{
		    /* Select the parent directory only if its not the
		     * current directory
		     */
		    if((cur_path != NULL) ?
			strcmp((const char *)cur_path, (const char *)parent_dir) : TRUE
		    )
			EDVBrowserDirTreeSelectPath(browser, parent_dir);
		}
		else
		{
		    /* Set the Directory Tree origin as the parent
		     * directory and select the parent directory
		     */
		    EDVBrowserDirTreeSetOriginPath(browser, parent_dir);
		}


		/* The contents listing on the browser should be updated
		 * if the parent directory was selected successfully
		 *
		 * Now find the object on the content's clist who's path
		 * matches the path of the object
		 */
		if(clist != NULL)
		{
		    const gint row = EDVBrowserContentsFindRowByPath(
			browser, full_path
		    );
		    if(row > -1)
		    {
			gtk_clist_freeze(clist);
			gtk_clist_unselect_all(clist);
			gtk_clist_select_row(clist, row, 0);
			gtk_clist_thaw(clist);
		    }
		}

		GUIBlockInput(toplevel, FALSE);
		EDVBrowserSetBusy(browser, FALSE);
	    }
	}
	/* Image browser */
	else if(fw->imbr_num > -1)
	{
	    const gint imbr_num = fw->imbr_num;
	    edv_imbr_struct *imbr;

	    if((imbr_num >= 0) && (imbr_num < core->total_imbrs))
		imbr = core->imbr[imbr_num];
	    else
		imbr = NULL;

	    if(imbr != NULL)
	    {
		const gchar *cur_path = EDVImbrCurrentLocation(imbr);
		tlist_struct *tlist = imbr->tlist;

		/* Select the parent directory only if its different
		 * than the current directory
		 */
		if((cur_path != NULL) ?
		    strcmp((const char *)cur_path, (const char *)parent_dir) : TRUE
		)
		    EDVImbrSelectPath(imbr, parent_dir);

		/* The thumbs list should now contain a list of the
		 * contents of the parent_dir
		 *
		 * Now we look for a particular thumb that matches the
		 * path of the object
		 */
		if(tlist != NULL)
		{
		    const gint thumb_num = EDVImbrTListFindThumbByPath(
			imbr, full_path
		    );
		    TListFreeze(tlist);
		    TListUnselectAll(tlist);
		    TListSelectThumb(tlist, thumb_num);
		    TListThaw(tlist);
		}
	    }
	}
	/* Recycle Bin */
	else if(fw->recbin_num > -1)
	{
	    edv_recbin_struct *recbin = core->recbin;
	    if((recbin != NULL) && (obj->name != NULL))
	    {
		GtkCList *clist = GTK_CLIST(recbin->contents_clist);
		if(clist != NULL)
		{
		    const gint row = EDVRecBinContentsFindRowByIndex(
			recbin,
			obj->index
		    );
		    if(row > -1)
		    {
			gtk_clist_freeze(clist);
			gtk_clist_unselect_all(clist);
			gtk_clist_select_row(clist, row, 0);
			gtk_clist_thaw(clist);
		    }
		}
	    }
	}
	/* Archiver */
	else if(fw->archiver_num > -1)
	{
	    const gint archiver_num = fw->archiver_num;
	    edv_archiver_struct *archiver;

	    if((archiver_num >= 0) && (archiver_num < core->total_archivers))
		archiver = core->archiver[archiver_num];
	    else
		archiver = NULL;

	    if(archiver != NULL)
	    {
		GtkCList *clist = GTK_CLIST(archiver->contents_clist);
		if(clist != NULL)
		{
		    const gint row = EDVArchiverContentsFindRowByPath(
			archiver,
			obj->full_path
		    );
		    if(row > -1)
		    {
			gtk_clist_freeze(clist);
			gtk_clist_unselect_all(clist);
			gtk_clist_select_row(clist, row, 0);
			gtk_clist_thaw(clist);
		    }
		}
	    }
	}
	/* All else create a new file browser */
	else
	{
	    const gint browser_num = EDVNewBrowser(core, parent_dir);
	    edv_browser_struct *browser = (browser_num > -1) ?
		core->browser[browser_num] : NULL;

	    fw->browser_num = browser_num;

	    if(browser != NULL)
	    {
		/* The contents listing on the browser should be updated
		 * if the parent directory was selected successfully
		 *
		 * Now find the object on the content's clist who's path
		 * matches the path of the object
		 */
		GtkCList *clist = GTK_CLIST(browser->contents_clist);
		if(clist != NULL)
		{
		    const gint row = EDVBrowserContentsFindRowByPath(
			browser,
			full_path
		    );
		    if(row > -1)
		    {
			gtk_clist_freeze(clist);
			gtk_clist_unselect_all(clist);
			gtk_clist_select_row(clist, row, 0);
			gtk_clist_thaw(clist);
		    }
		}
	    }
	}

	g_free(parent_dir);
	g_free(full_path);
}


/*
 *	Find Window write protect changed signal callback.
 */
void EDVFindWinWriteProtectChangedCB(
	edv_find_win_struct *fw, gboolean state
)
{
	if(fw == NULL)
	    return;

	EDVFindWinUpdateMenus(fw);	
}

/*
 *	Find Window reconfigured notify signal callback.
 */
void EDVFindWinReconfiguredNotifyCB(edv_find_win_struct *fw)
{
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(fw == NULL)
	    return;

	core = fw->core;
	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;


	/* Begin updating */

	/* Update RC styles */
	w = fw->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = fw->results_clist;
	if((w != NULL) && (lists_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = fw->results_clist_menu;
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, standard_rcstyle);


	EDVFindWinUpdateMenus(fw);
}
