#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>

#include <gtk/gtk.h>

#include "../guiutils.h"
#include "../cdialog.h"

#include "../lib/endeavour2.h"
#include "../lib/edv_devices_list.h"

#include "formatmanager.h"
#include "formatcb.h"
#include "config.h"


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	UNIX signal callback.
 */
static void FormatManagerSignalCB(int s)
{
	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGSEGV:
	    exit(1);
	    break;
	}
}

/*
 *	Main timeout callback.
 */
static gint FormatManagerMainTOCB(gpointer data)
{
	format_manager_struct *fm = FORMAT_MANAGER(data);
	if((fm != NULL) ? !fm->map_state : TRUE)
	{
	    gtk_main_quit();
	    return(FALSE);
	}
	else
	{
	    return(TRUE);
	}
}

int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	gint i;
	format_manager_struct *fm;
	gboolean auto_start = FALSE;
	const gchar *arg, *startup_device = NULL;
	gchar *devices_ini_file;
	edv_device_struct **device = NULL;
	gint total_devices = 0;
	edv_context_struct *ctx;

	/* Set up signal callbacks */
	signal(SIGINT, FormatManagerSignalCB);
	signal(SIGTERM, FormatManagerSignalCB);
	signal(SIGKILL, FormatManagerSignalCB);
	signal(SIGSEGV, FormatManagerSignalCB);
	signal(SIGSTOP, FormatManagerSignalCB);
	signal(SIGCONT, FormatManagerSignalCB);
	signal(SIGPIPE, FormatManagerSignalCB);

#define CLEANUP_RETURN(_v_)	{	\
					\
 return(_v_);				\
}

	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    /* Help */
	    if(!g_strcasecmp(arg, "--help") ||
	       !g_strcasecmp(arg, "-help") ||
	       !g_strcasecmp(arg, "--h") ||
	       !g_strcasecmp(arg, "-h") ||
	       !g_strcasecmp(arg, "-?")
	    )
	    {
		g_print("%s", PROG_HELP_MESG);
		CLEANUP_RETURN(0);
	    }
	    /* Version */
	    else if(!g_strcasecmp(arg, "--version") ||
		    !g_strcasecmp(arg, "-version")
	    )
	    {
		g_print("%s %s\n%s", PROG_NAME, PROG_VERSION, PROG_COPYRIGHT);
		CLEANUP_RETURN(0);
	    }
	    /* Auto start */
	    else if(!g_strcasecmp(arg, "--start") ||
		    !g_strcasecmp(arg, "-start") ||
		    !strcmp(arg, "--s") ||
		    !strcmp(arg, "-s")
	    )
	    {
		auto_start = TRUE;
	    }
	    /* Skip these arguments so that gtk_window_apply_args()
	     * handles them
	     */
	    else if(gtk_is_window_arg(arg))
	    {
		i++;
	    }
	    /* Single character argument? */
	    else if((*arg == '-') ? (arg[1] != '-') : FALSE)
	    {
		const gchar *v = arg + 1;
		gchar c;

		while(*v != '\0')
		{
		    c = *v;
		    if(c == 's')
		    {
			auto_start = TRUE;
		    }
		    else
		    {
			g_printerr(
"-%c: Unsupported argument.\n",
			    c  
			);
			CLEANUP_RETURN(2);
		    }
		    v++;
		}
	    }
	    /* Non-option argument? */
	    else if((*arg != '-') && (*arg != '+'))
	    {
		startup_device = arg;
	    }
	    else
	    {
		g_printerr(
"%s: Unsupported argument.\n",
		    arg
		);
		CLEANUP_RETURN(2);
	    }
	}



	/* Initialize GTK+ as needed */
	if(!initialized_gtk)
	{
	    if(!gtk_init_check(&argc, &argv))
	    {
		g_printerr("Unable to initialize GTK.\n");
		CLEANUP_RETURN(1);
	    }
	    initialized_gtk = TRUE;
	}


	/* Initialize GDK RGB buffers system */
	gdk_rgb_init();


	/* Initialize dialogs */
	CDialogInit();

	/* Initialize Endeavour context */
	ctx = EDVContextNew();
	EDVContextLoadConfigurationFile(ctx, NULL);

	/* Load devices */
	devices_ini_file = STRDUP(EDVGetS(ctx, EDV_CFG_PARM_FILE_DEVICES));
	device = EDVDevicesListFileOpen(devices_ini_file, &total_devices);

	/* Update device mount states and stats */
	EDVDevicesListUpdateMountStates(device, total_devices);
	EDVDevicesListUpdateStats(device, total_devices);

	/* Check if no devices have been loaded, suggesting that
	 * either no device references exist in the devices.ini or
	 * that the file does not exist
	 */
	if(total_devices == 0)
	{
	    gchar *buf = g_strdup_printf(
"No devices were found in the file:\n\
\n\
    %s\n\
\n\
It is possible that no device references were defined\n\
or that the file does not exist.",
		devices_ini_file
	    );
	    EDVPlaySoundWarning(ctx);
	    CDialogSetTransientFor(NULL);
	    CDialogGetResponse(
		"No Devices Found",
		buf,
"If you do have devices then you should run\n\
Endeavour Mark II to configure the device\n\
references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	}


	/* Create first and only format manager window */
	fm = FormatManagerNew(ctx);
	if(fm != NULL)
	{
	    gtk_window_apply_args(GTK_WINDOW(fm->toplevel), argc, argv);
	    FormatManagerMap(fm);

	    /* Transfer loaded devices to format manager */
	    FormatManagerSetBusy(fm, TRUE);
	    FormatManagerSetDevices(fm, device, total_devices);
	    device = NULL;
	    total_devices = 0;
	    FormatManagerSetBusy(fm, FALSE);

	    /* Select startup device? */
	    if(startup_device != NULL)
	    {
		gboolean device_selected = FALSE;
		gint row, column;
		gchar *cell_text;
		guint8 spacing;
		GdkPixmap *pixmap;
		GdkBitmap *mask;
		GtkCList *clist = (GtkCList *)fm->devices_clist;
		if(clist != NULL)
		{
		    /* Search column 1, the device path column, for
		     * a cell who's value matches startup_device
		     */
		    column = 1;
		    for(row = 0; row < clist->rows; row++)
		    {
			cell_text = NULL;
			switch((gint)gtk_clist_get_cell_type(clist, row, column))
			{
			  case GTK_CELL_TEXT:
			    gtk_clist_get_text(clist, row, column, &cell_text);
			    break;
			  case GTK_CELL_PIXTEXT:
			    gtk_clist_get_pixtext(
				clist, row, column, &cell_text,
				&spacing, &pixmap, &mask
			    );
			    break;
			}
			/* Got match? */
			if((cell_text != NULL) ?
			    !strcmp(startup_device, cell_text) : FALSE
			)
			{
			    gtk_clist_select_row(clist, row, column);
			    device_selected = TRUE;
			    break;
			}
		    }
		}

		/* Auto start and a device was selected? */
		if(auto_start && device_selected)
		{
		    FormatManagerStartCB(fm->start_btn, fm);
		}
	    }


	    gtk_timeout_add(
		1000l,
		FormatManagerMainTOCB, fm
	    );
	    gtk_main();

	    FormatManagerDelete(fm);
	}

	/* Delete list of devices (just in case a format manager was
	 * not created)
	 */
	for(i = 0; i < total_devices; i++)
	    EDVDeviceDelete(device[i]);
	g_free(device);
	device = NULL;
	total_devices = 0;

	g_free(devices_ini_file);
	devices_ini_file = NULL;

	EDVContextSync(ctx);
	EDVContextDelete(ctx);
	ctx = NULL;

	/* Shutdown dialogs */
	CDialogShutdown();

	/* Reset the DND Icon */
	GUIDNDSetDragIcon(NULL, NULL, 0, 0);

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}
