#include <string.h>
#include <time.h>
#include <glib.h>

#include "edv_context.h"
#include "edv_get.h"
#include "edv_date.h"
#include "edv_cfg_list.h"


const gchar *EDVDateString(
	edv_context_struct *ctx,
	const gulong t
);
const gchar *EDVDateStringDuration(
	edv_context_struct *ctx,
	const gulong dt
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns a statically allocated string describing the date &
 *	time.
 *
 *	The format and relativity of the date & time is specified by
 *	the configuration on the context.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The t specifies the time in seconds since EPOCH.
 */
const gchar *EDVDateString(
	edv_context_struct *ctx,
	const gulong t
)
{
	const gchar *format;
	edv_date_relativity relativity;
	static gchar buf[80];

	if(ctx == NULL)
	    return(NULL);

	format = EDVGetS(ctx, EDV_CFG_PARM_DATE_FORMAT);
	relativity = EDVGetI(ctx, EDV_CFG_PARM_DATE_RELATIVITY);

	/* Handle by relativity */
	switch(relativity)
	{
	  case EDV_DATE_RELATIVITY_CURRENT:
	    if(t > 0l)
	    {
		const gulong dt = (gulong)time(NULL) - t;
		gulong ct;

		/* Less than one second? */
		if(dt < 1l)
		{
		    strncpy(
			(char *)buf,
			"less than a second ago",
			sizeof(buf)
		    );
		    buf[sizeof(buf) - 1] = '\0';
		}
		/* Less than one minute? */
		else if(dt < (1l * 60l))
		{
		    ct = MAX((dt / 1l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "second" : "seconds")
		    );
		}
		/* Less than one hour? */
		else if(dt < (60l * 60l))
		{
		    ct = MAX((dt / 60l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "minute" : "minutes")
		    );
		}
		/* Less than one day? */
		else if(dt < (24l * 60l * 60l))
		{
		    ct = MAX((dt / 60l / 60l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "hour" : "hours")
		    );
		}
		/* Less than one week? */
		else if(dt < (7l * 24l * 60l * 60l))
		{
		    ct = MAX((dt / 60l / 60l / 24l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "day" : "days")
		    );
		}
		/* Less than one month (30 days)? */
		else if(dt < (30l * 24l * 60l * 60l))
		{
		    ct = MAX((dt / 60l / 60l / 24l / 7l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "week" : "weeks")
		    );
		}
		/* Less than 6 months ago? */
		else if(dt < (6l * 30l * 24l * 60l * 60l))
		{
		    ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "month" : "months")
		    );
		}
		/* 6 months or longer ago */
		else
		{
		    /* All else revert to using ctime() */
		    time_t tv = (time_t)t;
		    gchar *s = ctime(&tv);

		    strncpy(
			(char *)buf,
			(const char *)((s != NULL) ? s : ""),
			sizeof(buf)
		    );
		    buf[sizeof(buf) - 1] = '\0';

		    s = (gchar *)strchr((char *)buf, '\n');
		    if(s != NULL)
			*s = '\0';
		}

		return(buf);
	    }
	    break;

	  case EDV_DATE_RELATIVITY_ABSOLUTE:
	    /* Date and time format string specified? */
	    if(format != NULL)
	    {
		time_t tv = (time_t)t;
		const struct tm *tm_ptr;

		*buf = '\0';
		tm_ptr = localtime(&tv);

		if(tm_ptr != NULL)
		    strftime(
			(char *)buf, sizeof(buf),
			(const char *)format, tm_ptr
		    );
		buf[sizeof(buf) - 1] = '\0';

		return(buf);
	    }
	    else
	    {
		/* No format string specified so use ctime() */
		time_t t2 = (time_t)t;
		return((const gchar *)ctime(&t2));
	    }
	    break;
	}

	return("");
}

/*
 *      Formats a date and time string describing the time lapsed in
 *      the format of "<n1> <units1> <n2> <units2>".
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *      The dt specifies the time lapsed in seconds. If dt is 0 then
 *      "less than a second" will be returned.
 *
 *      Returns a statically allocated string describing the
 *      time lapsed.
 */
const gchar *EDVDateStringDuration(
	edv_context_struct *ctx,
	const gulong dt
)
{
	gulong ct, ct2;
	static gchar buf[80];

	/* Less than one second? */
	if(dt < 1l)
	{
	    g_snprintf(
		buf, sizeof(buf),
		"less than a second"
	    );
	}
	/* Less than one minute? */
	else if(dt < (1l * 60l))
	{
	    ct = MAX((dt / 1l), 1l);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld %s",
		ct, ((ct == 1l) ? "second" : "seconds")
	    );
	}
	/* Less than one hour? */
	else if(dt < (60l * 60l))
	{
	    ct = MAX((dt / 60l), 1l);
	    ct2 = MAX((dt / 1l), 1l) % 60l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "minute" : "minutes"),
		    ct2, ((ct2 == 1l) ? "second" : "seconds")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "minute" : "minutes")
		);
	}
	/* Less than one day? */
	else if(dt < (24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l), 1l);
	    ct2 = MAX((dt / 60l), 1l) % 60l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "hour" : "hours"),
		    ct2, ((ct2 == 1l) ? "minute" : "minutes")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "hour" : "hours")
		);
	}
	/* Less than one week? */
	else if(dt < (7l * 24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l / 24l), 1l);
	    ct2 = MAX((dt / 60l / 60l), 1l) % 24l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "day" : "days"),
		    ct2, ((ct2 == 1l) ? "hour" : "hours")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "day" : "days")
		);
	}
	/* Less than one month (30 days)? */
	else if(dt < (30l * 24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l / 24l / 7l), 1l);
	    ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 7l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "week" : "weeks"),
		    ct2, ((ct2 == 1l) ? "day" : "days")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "week" : "weeks") 
		);
	}
#if 0
	/* Less than 6 months ago? */
	else if(dt < (6l * 30l * 24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
	    ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 30l;
	    g_snprintf(
		buf, sizeof(buf),
		"%ld %s %ld %s",
		ct, ((ct == 1l) ? "month" : "months"),
		ct2, ((ct2 == 1l) ? "day" : "days")
	    );
	}
#endif
	/* Less than a year (365 days)? */
	else if(dt < (12l * 30l * 24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
	    ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 30l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "month" : "months"),
		    ct2, ((ct2 == 1l) ? "day" : "days")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "month" : "months")
		);
	}
	/* A year or longer */
	else
	{
	    ct = MAX((dt / 60l / 60l / 24l / 30l / 12l), 1l);
	    ct2 = MAX((dt / 60l / 60l / 24l / 30l), 1l) % 12l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "year" : "years"),
		    ct2, ((ct2 == 1l) ? "month" : "months")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "year" : "years")
		);
	}

	return(buf);
}
