#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
# 
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
# 
# Author: Enthought, Inc.
# Description: <Enthought pyface package component>
#------------------------------------------------------------------------------
""" A simple progress bar intended to run in the UI thread """

import wx

# Enthought library imports
from enthought.traits.api import Instance, Enum, Str, Int, Bool

# Local imports
from widget import Widget
from window import Window
from dialog import Dialog

class ProgressBar(Widget):
    """ A simple progress bar dialog intended to run in the UI thread """
    
    parent = Instance(wx.Frame)
    direction = Enum('horizontal', 'horizontal', 'vertical')
    
    def __init__(self, parent, minimum=0, maximum=100, direction='horizontal',
                 size=(150, -1)):
        """
        Constructs a progress bar which can be put into a panel, or optionaly,
        its own window
        
        """
        self._max = max        
        self.parent = parent
        
        style = wx.GA_HORIZONTAL
        if direction == "vertical":
            style = wx.GA_VERTICAL
        
        print wx.DefaultSize
        print wx.DefaultPosition
        self.control = wx.Gauge(parent, -1, maximum, style=style, size=size)
        

    def update(self, value):
        """ 
        Updates the progress bar to the desired value.
        
        """
        
        self.control.SetValue(value)
        self.control.Update()        
                
    def _create_parent(self):
        parent = wx.Frame(None, -1, "progress")
        parent.SetExtraStyle(parent.GetExtraStyle() | wx.WS_EX_TRANSIENT)        
        
        return parent            

    def _show(self):
        # Show the parent
        self.parent.Show()
        
        # Show the toolkit-specific control in the parent
        self.control.Show()            

class ProgressDialog(Window):
    """ A simple progress dialog window which allows itself to be updated """
    
    progress_bar = Instance(ProgressBar)
    title = Str
    message = Str
    min = Int
    max = Int
    margin = Int(5)
    can_cancel = Bool(False)

    dialog_size = Instance(wx.Size)

    # Label for the 'cancel' button
    cancel_button_label = Str('Cancel')
    
    def open(self):
        super(ProgressDialog, self).open()
        wx.Yield()

    
    def update(self, value):
        """ 
        updates the progress bar to the desired value. If the value is >= 
        the maximum and the progress bar is not contained in another panel
        the parent window will be closed
        
        """
        
        if self.progress_bar is not None:
            self.progress_bar.update(value)
            
        if value >= self.max:
            self.close()


    def _create_buttons(self, dialog, parent_sizer):
        """ Creates the buttons. """

        sizer = wx.BoxSizer(wx.HORIZONTAL)
        self._cancel = None
        
        if self.can_cancel == True:    
    
            # 'Cancel' button.
            self._cancel = cancel = wx.Button(dialog, wx.ID_CANCEL,
                                              self.cancel_button_label)
            wx.EVT_BUTTON(dialog, wx.ID_CANCEL, self._on_cancel)
            sizer.Add(cancel, 0, wx.LEFT, 10)
            
            button_size = cancel.GetSize()
            self.dialog_size.x = max(self.dialog_size.x, button_size.x + 2*self.margin)
            self.dialog_size.y += button_size.y + 2*self.margin
    
            parent_sizer.Add(sizer, 0, wx.ALIGN_RIGHT | wx.ALL, 5)
        return
    
    def _create_gauge(self, dialog, parent_sizer):
        self.progress_bar = ProgressBar(dialog, self.min, self.max)
        parent_sizer.Add(self.progress_bar.control, 0, wx.CENTER | wx.ALL, 5)

        progress_bar_size = self.progress_bar.control.GetSize()
        self.dialog_size.x = max(self.dialog_size.x, progress_bar_size.x + 2*self.margin)
        self.dialog_size.y += progress_bar_size.y + 2*self.margin

        return
        
    def _create_label(self, dialog, parent_sizer):
        msg_control = wx.StaticText(dialog, -1, self.message)
        parent_sizer.Add(msg_control, 0, wx.LEFT | wx.TOP, 5)

        msg_control_size = msg_control.GetSize()
        self.dialog_size.x = max(self.dialog_size.x, msg_control_size.x + 2*self.margin)
        self.dialog_size.y += msg_control_size.y + 2*self.margin

        return
        
            
    def _create_contents(self, parent):
        """ Creates the window contents.

        This method is intended to be overridden if necessary.  By default we
        just create an empty panel.

        """
        dialog = parent
        
        sizer = wx.BoxSizer(wx.VERTICAL)
        dialog.SetSizer(sizer)
        dialog.SetAutoLayout(True)
        dialog.SetBackgroundColour(wx.NullColor)
        
        self.dialog_size = wx.Size()
        
        # The 'guts' of the dialog.
        self._create_label(dialog, sizer)
        self._create_gauge(dialog, sizer)
        self._create_buttons(dialog, sizer)

        dialog.SetClientSize(self.dialog_size)


#        if self.size != (-1,-1):
#            dialog.SetSize(self.size)
#
#        else:
#            sizer.Fit(dialog)

        dialog.CentreOnParent()
        
        return dialog
            
if __name__ == "__main__":
    #
    # simple example of its use
    #
    import time
    from enthought.pyface.api import GUI, ApplicationWindow
    from enthought.pyface.action.api import Action, MenuManager, MenuBarManager

    def task_func(t):
        progress = ProgressDialog(title="progress", message="counting to %d"%t, max=t)
        progress.open()
        
        for i in range(0,t+1):
            time.sleep(1)
            print i
            progress.update(i)
            
        progress.update(4)

    def _main():
        task_func(4)
    
    class MainWindow(ApplicationWindow):
        """ The main application window. """

        ######################################################################
        # 'object' interface.
        ######################################################################

        def __init__(self, **traits):
            """ Creates a new application window. """

            # Base class constructor.
            super(MainWindow, self).__init__(**traits)

            # Add a menu bar.
            self.menu_bar_manager = MenuBarManager(
                MenuManager(
                    Action(name='E&xit', on_perform=self.close),
                    Action(name='DoIt', on_perform=_main),
                    name = '&File',
                )
            )

            return


    gui = GUI(redirect=False)

    # Create and open the main window.
    window = MainWindow()
    window.open()

    wx.CallAfter(_main)

    # Start the GUI event loop!
    gui.event_loop()
