#include <gtk/gtk.h>
#include <stdlib.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"


/*--element

Element Name: <entry>

This creates a small, one line high text area allowing you to enter text.

%widget%

*/

/*--attr

Attribute: "onchange"

If this attribute is set, then whenever a change is made (a key is
pressed) in the entry, the perl function specified as it's value will
be called.  The first argument is the full path to the entry that
generated the event.

Default: none (no action)

*/

static void
rendgtk_entry_onchange_callback (GtkWidget * entry, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;
    gchar *text;

    text = gtk_entry_get_text (GTK_ENTRY (entry));

    enode_attrib_quiet (node, "text", ebuf_new_with_str (text));

    function = enode_attrib_str (node, "onchange", NULL);

    if (function) {
	enode_call_ignore_return (node, function, "");
    }
}

/*--attr

Attribute: "onenter"

When the "Enter" key is pressed, this attribute will be checked.  If
it is set, the perl function specified as it's value will be called.
The argument is the full path to the entry that generated the event.

Default: none (no action)

*/

static void
rendgtk_entry_onenter_callback (GtkWidget * widget, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;

    function = enode_attrib_str (node, "onenter", NULL);

    enode_call_ignore_return (node, function, "");
}

static gint
rendgtk_entry_text_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *entry;

    entry = enode_get_kv (node, "top-widget");
    if (!entry)
	return (TRUE);

    if (ebuf_not_empty (value))
	gtk_entry_set_text (GTK_ENTRY (entry), value->str);
    else
	gtk_entry_set_text (GTK_ENTRY (entry), "");


    return TRUE;
}

static gint
rendgtk_widget_entry_hidden_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *entry;

    entry = enode_get_kv (node, "top-widget");
    if (!entry)
	return (TRUE);

    gtk_entry_set_visibility (GTK_ENTRY (entry), !erend_value_is_true (value));
    return (TRUE);
}


static void
rendgtk_entry_render (ENode * node)
{
    GtkWidget *entry;

    entry = gtk_entry_new ();

    enode_set_kv (node, "top-widget", entry);
    enode_set_kv (node, "bottom-widget", entry);

    enode_attribs_sync (node);

    gtk_signal_connect (GTK_OBJECT (entry), "changed",
			rendgtk_entry_onchange_callback, node);

    gtk_signal_connect (GTK_OBJECT (entry), "activate",
			rendgtk_entry_onenter_callback, node);

    rendgtk_show_cond (node, entry);
}

void
entry_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

    /* clist */
    element = g_malloc0 (sizeof (Element));
    element->render_func = rendgtk_entry_render;
    element->destroy_func = rendgtk_element_destroy;
    element->parent_func = rendgtk_box_pack;
    element->tag = "entry";
    element->description = "Create a single line text entry.";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "hidden";
    e_attr->description = "Set if the widget should echo the text.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "false,true";
    e_attr->set_attr_func = rendgtk_widget_entry_hidden_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "text";
    e_attr->description = "Text displayed in the widget.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_entry_text_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onenter";
    e_attr->description =
	"Sets the function to call when the enter is pressed.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(text_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    rendgtk_widget_attr_register (element, GTK_TYPE_ENTRY);
}
