/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-key.c,v 1.5 2003/12/10 23:16:45 hoa Exp $
 */

#include "etpan-cfg-key.h"

#include "etpan-errors.h"
#include "etpan-db-read.h"
#include <ctype.h>
#include <ncurses.h>
#include <string.h>
#include <stdlib.h>

static const struct key_value {
  char * name;
  int value;
} key_value_tab[] = {
  { "BACKSPACE",   KEY_BACKSPACE},
  { "BREAK",       KEY_BREAK},
  { "DELETE",      KEY_DC},
  { "DOWN",        KEY_DOWN},
  { "END",         KEY_END},
  { "ENTER",       KEY_ENTER},
  { "F1",          KEY_F(1)},
  { "F2",          KEY_F(2)},
  { "F3",          KEY_F(3)},
  { "F4",          KEY_F(4)},
  { "F5",          KEY_F(5)},
  { "F6",          KEY_F(6)},
  { "F7",          KEY_F(7)},
  { "F8",          KEY_F(8)},
  { "F9",          KEY_F(9)},
  { "F10",         KEY_F(10)},
  { "F11",         KEY_F(11)},
  { "F12",         KEY_F(12)},
  { "HOME",        KEY_HOME},
  { "INSERT",      KEY_IC},
  { "LEFT",        KEY_LEFT},
  { "PAGE_DOWN",   KEY_NPAGE},
  { "PAGE_UP",     KEY_PPAGE},
  { "RETURN",      '\n'},
  { "RIGHT",       KEY_RIGHT},
  { "SPACE",       ' '},
  { "SUSPEND",     KEY_SUSPEND},
  { "TAB",         '\t'},
  { "UP",          KEY_UP},
};

static int key_cmp(char * name1, struct key_value * key2) {
  return (strcasecmp(name1, key2->name));
}

static int get_main_key_value(char * name)
{
  struct key_value * key;
  
  if (name[0] == '\0')
    return 0;
  
  if (name[1] == '\0')
    return tolower(name[0]);
  
  key = bsearch(name, key_value_tab,
      sizeof(key_value_tab) / sizeof(struct key_value),
      sizeof(struct key_value),
      (int (*)(const void *, const void *)) key_cmp);
  
  if (key == NULL)
    return 0;

  return key->value;
}

#define MODIFIER_SHIFT 1
#define MODIFIER_CTRL  2
#define MODIFIER_ALT   4

#define ALT_MASK     0x80
#define KEY_ALT(x)  ((x) | ALT_MASK)
#define KEY_CTRL(x) ((x) & 0x1f)

static int get_key_value(char * name)
{
  char * p;
  int val;
  int modifier;
  
  modifier = 0;
  
  p = name;
  
  val = get_main_key_value(p);
  if (val != 0)
    return val;
  
  while (p[0] != '\0') {
    if (p[1] == '-') {
      char mod;
      
      mod = toupper(p[0]);
      switch (mod) {
      case 'S':
        modifier |= MODIFIER_SHIFT;
        break;
      case 'C':
        modifier |= MODIFIER_CTRL;
        break;
      case 'A':
        modifier |= MODIFIER_ALT;
        break;
      }
    }
    else if (p[1] == '\0') {
      val = tolower(p[0]);
      if ((modifier & MODIFIER_SHIFT) != 0)
        val = toupper(val);
      if ((modifier & MODIFIER_CTRL) != 0)
        val = KEY_CTRL(val);
      if ((modifier & MODIFIER_ALT) != 0)
        val = KEY_ALT(val);
      
      return val;
    }
    else
      return 0;
  }
  
  return 0;
}

int etpan_key_config_read(char * filename,
    struct etpan_key_config ** result)
{
  struct etpan_db * db;
  int r;
  int res;
  unsigned int i;
  chash * key_hash;
  struct etpan_key_config * config;
  
  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  key_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (key_hash == NULL) {
    res = ERROR_MEMORY;
    goto free_db;
  }

  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    chash * entry;
    char * key_name;
    char * action;
    chashdatum key;
    chashdatum value;
    int key_value;

    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    key_name = etpan_db_entry_get_value(entry, "key");
    if (key_name == NULL)
      continue;
    
    key_value = get_key_value(key_name);

    action = etpan_db_entry_get_value(entry, "action");
    
    key.data = &key_value;
    key.len = sizeof(key_value);
    value.data = action;
    value.len = strlen(action) + 1;
    
    r = chash_set(key_hash, &key, &value, NULL);
    if (r < 0) {
      res = ERROR_MEMORY;
      goto free_key_hash;
    }
  }

  config = etpan_key_config_new(key_hash);
  if (config == NULL) {
    res = ERROR_MEMORY;
    goto free_key_hash;
  }
  
  etpan_db_free(db);
  
  * result = config;
  
  return NO_ERROR;

 free_key_hash:
  chash_free(key_hash);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}

struct etpan_key_config * etpan_key_config_new(chash * key_hash)
{
  struct etpan_key_config * config;
  
  config = malloc(sizeof(* config));
  if (config == NULL)
    return NULL;
  
  config->key_hash = key_hash;
  
  return config;
}

void etpan_key_config_free(struct etpan_key_config * config)
{
  chash_free(config->key_hash);
  free(config);
}

