/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: object.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "fullengine.h"
#include "object.h"
#include "lists.h"
#include "engine.h"
#include "handlemgr.h"
#include "discover.h"
#include "common.h"
#include "volume.h"
#include "internalAPI.h"
#include "message.h"
#include "memman.h"
#include "common.h"
#include "remote.h"


int isa_valid_input_object(storage_object_t    * obj,
			   storage_container_t * disk_group) {

	LOG_PROC_ENTRY();

	if ((obj->object_type != DISK) &&
	    (obj->object_type != SEGMENT) &&
	    (obj->object_type != REGION) &&
	    (obj->object_type != EVMS_OBJECT)) {
		LOG_ERROR("This is not a storage object.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The object must not be corrupt. */
	if (obj->flags & SOFLAG_CORRUPT) {
		LOG_ERROR("Object %s is not a valid input object.  It is corrupt.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The object must be a top level object. */
	if (!is_top_object(obj)) {
		LOG_ERROR("Object %s is not a valid input object.  It is not a top level object.\n", obj->name);
		return EINVAL;
	}

	/* The object must not insist on being the top object. */
	if (obj->flags & SOFLAG_MUST_BE_TOP) {
		LOG_ERROR("Object %s is not a valid input object.  It insists it must be a top level object.\n", obj->name);
		return EINVAL;
	}

	/* The object must be in the specified disk group. */
	if (obj->disk_group != disk_group) {
		LOG_ERROR("Object %s in disk group %s is not in disk group %s.\n", obj->name, (obj->disk_group != NULL) ? obj->disk_group->name : "(local)", (disk_group != NULL) ? disk_group->name : "(local)");
		return EINVAL;
	}

	LOG_DEBUG("Object %s is a valid input object.\n", obj->name);
	LOG_PROC_EXIT_INT(0);
	return 0;
}


static int validate_create_parameters(plugin_handle_t     plugin_handle,
				      handle_array_t    * objects,
				      plugin_record_t * * ppPlugRec,
				      list_anchor_t     * pObjectList) {
	int rc = 0;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	*ppPlugRec = NULL;
	*pObjectList = NULL;

	rc = translate_handle(plugin_handle,
			      &object,
			      &type);

	if (rc == HANDLE_MANAGER_NO_ERROR) {
		if (type == PLUGIN) {
			plugin_record_t * pPlugRec = (plugin_record_t *) object;
			plugin_type_t plugin_type = GetPluginType(pPlugRec->id);

			*ppPlugRec = pPlugRec;

			if ((plugin_type == EVMS_DEVICE_MANAGER) ||
			    (plugin_type == EVMS_SEGMENT_MANAGER) ||
			    (plugin_type == EVMS_REGION_MANAGER) ||
			    (plugin_type == EVMS_FEATURE) ||
			    (plugin_type == EVMS_ASSOCIATIVE_FEATURE)) {
				list_anchor_t object_list = allocate_list();

				if (object_list != NULL) {

					rc = make_list(objects, object_list);

					if (rc == 0) {
						list_element_t iter;
						storage_object_t * obj;
						storage_container_t * disk_group = NULL;

						obj = first_thing(object_list, NULL);
						if (obj != NULL) {
							disk_group = obj->disk_group;
						}

						/*
						 * Make sure that the input object list contains
						 * top level objects and none of them insist on
						 * being a top object.
						 */
						LIST_FOR_EACH(object_list, iter, obj) {
							rc = isa_valid_input_object(obj, disk_group);
							if (rc != 0) {
								break;
							}
						}
					}

					if (rc != 0) {
						destroy_list(object_list);
					}

					*pObjectList = object_list;

				} else {
					LOG_CRITICAL("Error allocating memory for an output object list.\n");
					rc = ENOMEM;
				}

			} else {
				LOG_ERROR("The plug-in %s is not a type that manages storage objects.\n", pPlugRec->short_name);
				rc = EINVAL;
			}

		} else {
			LOG_ERROR("The plugin_handle is not for a plug-in.\n");
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_create(plugin_handle_t    plugin_handle,
		handle_array_t   * objects,
		option_array_t   * options,
		handle_array_t * * output_objects) {

	int rc = 0;
	plugin_record_t * pPlugRec = NULL;
	list_anchor_t object_list = NULL;
	list_anchor_t input_objects;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_create(plugin_handle, objects, options, output_objects);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = validate_create_parameters(plugin_handle, objects,
					&pPlugRec,     &object_list);
	if (rc == 0) {
		/*
		 * Make a copy of the input objects list in case we have to
		 * clean up after an error.
		 */
		input_objects = copy_list(object_list);

		if (input_objects != NULL) {

			/*
			 * Make a list to receive the new objects from the
			 * create.
			 */
			STATIC_LIST_DECL(new_objects);

			/*
			 * We are finally setup to do the create.
			 */
			rc = pPlugRec->functions.plugin->create(object_list, options, &new_objects);

			if (rc == 0) {

				list_element_t iter;
				storage_object_t * new_obj;
				storage_object_t * child_obj;

				remove_corrupt_objects(&new_objects);

				LIST_FOR_EACH(&new_objects, iter, new_obj) {
					list_element_t tmp_iter;

					/*
					 * Now that the input objects are
					 * consumed, the parent can write
					 * on them anywhere, including over
					 * any stop data.  Forget that the
					 * child objects have stop data.
					 */
					LIST_FOR_EACH(new_obj->child_objects, tmp_iter, child_obj) {
						child_obj->flags &= ~SOFLAG_HAS_STOP_DATA;
					}

					/*
					 * Mark the new objects' feature
					 * headers dirty.
					 */
					set_feature_header_dirty(new_obj);

					/*
					 * Set the object's SOFLAG_NEEDS_ACTIVATE
					 * flag if all of its children are
					 * active or are scheduled to be
					 * activated..
					 */
					if (new_obj->data_type == DATA_TYPE) {
						new_obj->flags |= SOFLAG_NEEDS_ACTIVATE;
						if (new_obj->producing_container != NULL) {
							LIST_FOR_EACH(new_obj->producing_container->objects_consumed, tmp_iter, child_obj) {
								if (!(child_obj->flags & (SOFLAG_ACTIVE | SOFLAG_NEEDS_ACTIVATE))) {
									new_obj->flags &= ~SOFLAG_NEEDS_ACTIVATE;
									break;
								}
							}
						} else {
							LIST_FOR_EACH(new_obj->child_objects, tmp_iter, child_obj) {
								if (!(child_obj->flags & (SOFLAG_ACTIVE | SOFLAG_NEEDS_ACTIVATE))) {
									new_obj->flags &= ~SOFLAG_NEEDS_ACTIVATE;
									break;
								}
							}
						}
						if (new_obj->flags & SOFLAG_NEEDS_ACTIVATE) {
							LIST_FOR_EACH(new_obj->associated_children, tmp_iter, child_obj) {
								if (!(child_obj->flags & (SOFLAG_ACTIVE | SOFLAG_NEEDS_ACTIVATE))) {
									new_obj->flags &= ~SOFLAG_NEEDS_ACTIVATE;
									break;
								}
							}
						}
					}
				}

				propigate_cluster_info(&new_objects);

				/*
				 * Make sure the compatibility volume names
				 * are in sync.
				 */
				sync_volumes();

				/* Sort the list that was affected. */
				switch (GetPluginType(pPlugRec->id)) {
					case EVMS_DEVICE_MANAGER:
						sort_list(&disks_list, compare_objects, NULL);
						break;
					case EVMS_SEGMENT_MANAGER:
						sort_list(&segments_list, compare_objects, NULL);
						break;
					case EVMS_REGION_MANAGER:
						sort_list(&regions_list, compare_objects, NULL);
						break;
					case EVMS_FEATURE:
					case EVMS_ASSOCIATIVE_FEATURE:
						sort_list(&EVMS_objects_list, compare_objects, NULL);
						break;
					default:
						break;
				}

				/*
				 * If the user wants handles for the
				 * new objects, make the handles.
				 */
				if (output_objects != NULL) {
					/*
					 * type doesn't really
					 * matter as long as it's
					 * DISK, SEGMENT, or EVMS_OBJECT.
					 * make_handle_array() will figure out the
					 * correct type for each object in the list.
					 */
					rc = make_user_handle_array(&new_objects, output_objects);
				}

			} else {
				LOG_CRITICAL("Error returned from plugin's create routine.\n");
			}

			destroy_list(input_objects);

		} else {
			LOG_CRITICAL("Error getting memory for copying the input object list.\n");
			rc = ENOMEM;
		}

		destroy_list(object_list);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * evms_get_object_list returns a pointer to a handle_array_t with handles for
 * storage objects, optionally filtering on the object type, the data type of
 * the object, the plug-in that manages the object, the cluster node on which
 * the object resides, and some flags.  If any of the filtering parameters is 0,
 * that filter is not used.
 */
int evms_get_object_list(object_type_t         object_type,
			 data_type_t           data_type,
			 plugin_handle_t       plugin_handle,
			 object_handle_t       disk_group_handle,
			 object_search_flags_t flags,
			 handle_array_t    * * object_handle_list) {
	int rc = 0;
	void * object = NULL;
	object_type_t type;
	plugin_record_t * plugin = NULL;
	storage_container_t * disk_group = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_object_list(object_type,
					    data_type,
					    plugin_handle,
					    disk_group_handle,
					    flags,
					    object_handle_list);

		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (plugin_handle != 0) {
		/* Translate the handle for the feature to make sure it is valid */
		/* and to get the plugin_record_t for the feature. */
		rc = translate_handle(plugin_handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {
			if (type == PLUGIN) {
				plugin = (plugin_record_t *) object;
			} else {
				rc = EINVAL;
			}
		}
	}

	if (rc == 0) {
		if (disk_group_handle != 0) {
			/*
			 * Translate the handle for the disk group
			 * to make sure it is valid and to get the
			 * disk group storage_container_t.
			 */
			rc = translate_handle(disk_group_handle,
					      &object,
					      &type);

			if (rc == HANDLE_MANAGER_NO_ERROR) {
				if (type == CONTAINER) {
					disk_group = (storage_container_t *) object;
				} else {
					rc = EINVAL;
				}
			}
		}
	}

	if (rc == 0) {
		list_anchor_t object_list;

		/*
		 * Call the internal version of engine_get_object_list.  "plugin"
		 * will be NULL if the caller did not specify a plug-in, else it
		 * will be a pointer to the plug-in's plugin_record_t.
		 */
		rc = engine_get_object_list(object_type, data_type, plugin, disk_group, flags, &object_list);

		if (rc == 0) {
			/*
			 * type doesn't really matter as long as it's
			 * DISK, SEGMENT, or EVMS_OBJECT.
			 * make_handle_array() will figure out the
			 * correct type for each object in the list.
			 */
			rc = make_user_handle_array(object_list, object_handle_list);

			/* We are finished with the list that was returned by */
			/* engine_get_object_list. */
			destroy_list(object_list);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * evms_assign
 */
int evms_assign(object_handle_t  object_handle,
		plugin_handle_t  plugin_handle,
		option_array_t * options) {

	int rc = 0;
	void * object = NULL;
	object_type_t type;
	storage_object_t * obj = NULL;
	plugin_record_t * plugin = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return(rc);
	}

	if (!local_focus) {
		rc = remote_assign(object_handle, plugin_handle, options);
		LOG_PROC_EXIT_INT(rc);
		return(rc);
	}

	if (object_handle != 0) {
		/*
		 * Translate the handle for the object to make sure it is valid
		 * and to get the storage_object_t for the object.
		 */
		rc = translate_handle(object_handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {
			if ((type == DISK) ||
			    (type == SEGMENT) ||
			    (type == REGION) ||
			    (type == EVMS_OBJECT)) {
				storage_object_t * parent;

				obj = (storage_object_t *) object;
				parent = first_thing(obj->parent_objects, NULL);

				if (parent != NULL) {
					LOG_ERROR("Object %s already has parent objects produced by plug-in %s.\n", obj->name, parent->plugin->short_name);
				}

			} else {
				LOG_ERROR("Handle %d is not a handle for a storage object.\n", object_handle);
				rc = EINVAL;
			}
		}

	} else {
		LOG_ERROR("An object handle must be given.\n");
		rc = EINVAL;
	}

	if (plugin_handle != 0) {
		/*
		 * Translate the handle for the plug-in to make sure it is valid
		 * and to get the plugin_record_t for the plug-in.
		 */
		rc = translate_handle(plugin_handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {
			if (type == PLUGIN) {
				plugin = (plugin_record_t *) object;
			} else {
				LOG_ERROR("Handle %d is not a handle for a plug-in.\n", plugin_handle);
				rc = EINVAL;
			}
		}

	} else {
		LOG_ERROR("A plug-in handle must be given.\n");
		rc = EINVAL;
	}

	if (rc == 0) {
		rc = plugin->functions.plugin->assign(obj, options);

		if (rc == 0) {
			/*
			 * Now that the object is consumed,
			 * any stop data on it can be
			 * overwritten.  Forget that it is
			 * there.
			 */
			obj->flags &= ~SOFLAG_HAS_STOP_DATA;

			propigate_cluster_info(obj->parent_objects);

			/*
			 * If the target object of the assign is active or
			 * scheduled to be activated, mark its new parent
			 * objects for activation.
			 */
			if (obj->flags & (SOFLAG_ACTIVE | SOFLAG_NEEDS_ACTIVATE)) {
				list_element_t iter;
				storage_object_t * parent;

				LIST_FOR_EACH(obj->parent_objects, iter, parent) {
					if (parent->data_type == DATA_TYPE) {
						parent->flags |= SOFLAG_NEEDS_ACTIVATE;
					}
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return(rc);
}


static int get_plugin(storage_object_t   * obj,
		      plugin_record_t  * * pp_plugin_record) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (*pp_plugin_record == NULL) {
		*pp_plugin_record = obj->plugin;

	} else {
		if (*pp_plugin_record != obj->plugin) {
			LOG_ERROR("Object %s does not have parent objects that are all managed by the same plug-in.\n", obj->name);
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int get_parent_plugin(storage_object_t * obj, plugin_record_t * * p_parent_plugin) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (!list_empty(obj->parent_objects)) {
		list_element_t iter;
		storage_object_t * parent;

		LIST_FOR_EACH(obj->parent_objects, iter, parent) {
			rc = get_plugin(parent, p_parent_plugin);
			if (rc !=0) {
				break;
			}
		}

	} else {
		LOG_ERROR("Object %s does not have any parent objects.\n", obj->name);
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int is_in_use(storage_object_t * obj) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (!list_empty(obj->parent_objects)) {
		LOG_ERROR("Object %s has parent objects.\n", obj->name);
		rc = EINVAL;
	}

	if (obj->consuming_container != NULL) {
		LOG_ERROR("Object %s is part of a container.\n", obj->name);
		rc = EINVAL;
	}

	if (obj->volume != NULL) {
		LOG_ERROR("Object %s is used as volume %s.\n", obj->name, obj->volume->name);
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * evms_can_unassign determines if a plug-in can be unassigned from consuming
 * an object.  The parent objects must themselves not have parent objects,
 * must not be consumed by a container, and if a volume is created from them,
 * the volume must not be mounted.  If that is true, then ask the plug-in
 * if it can unassign itself from the object.
 */
int evms_can_unassign(object_handle_t object_handle) {

	int rc = 0;
	void * object = NULL;
	object_type_t type;
	storage_object_t * obj = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return(rc);
	}

	if (!local_focus) {
		rc = remote_can_unassign(object_handle);
		LOG_PROC_EXIT_INT(rc);
		return(rc);
	}

	if (object_handle != 0) {
		/*
		 * Translate the handle for the object to make sure it is valid
		 * and to get the storage_object_t for the object.
		 */
		rc = translate_handle(object_handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {
			if ((type == DISK) ||
			    (type == SEGMENT) ||
			    (type == REGION) ||
			    (type == EVMS_OBJECT)) {
				obj = (storage_object_t *) object;
			} else {
				LOG_DETAILS("Handle %d is not a handle for a storage object.\n", object_handle);
				rc = EINVAL;
			}
		}

	} else {
		LOG_ERROR("An object handle must be given.\n");
		rc = EINVAL;
	}

	if (rc == 0) {
		list_element_t iter;
		storage_object_t * parent;

		/*
		 * Check to make sure the parent objects don't have parents
		 * themselves, that they are not part of a container, and
		 * that they are not part of a volume.
		 */
		LIST_FOR_EACH(obj->parent_objects, iter, parent) {
			rc = is_in_use(parent);
			if (rc != 0) {
				break;
			}
		}

		if (rc == 0) {
			plugin_record_t * parent_plugin = NULL;

			/*
			 * Get the plug-in that is managing the parents of this
			 * object.
			 */
			rc = get_parent_plugin(obj, &parent_plugin);

			if (rc == 0) {
				rc = parent_plugin->functions.plugin->can_unassign(obj);
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return(rc);
}


/*
 * evms_unassign unassigns a plug-in from consuming an object.  This is most
 * useful for segment managers that claim a disk or segment.
 */
int evms_unassign(object_handle_t object_handle) {

	int rc = 0;
	void * object = NULL;
	object_type_t type;
	storage_object_t * obj = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return(rc);
	}

	if (!local_focus) {
		rc = remote_unassign(object_handle);
		LOG_PROC_EXIT_INT(rc);
		return(rc);
	}

	if (object_handle != 0) {
		/*
		 * Translate the handle for the object to make sure it is valid
		 * and to get the storage_object_t for the object.
		 */
		rc = translate_handle(object_handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {
			if ((type == DISK) ||
			    (type == SEGMENT) ||
			    (type == REGION) ||
			    (type == EVMS_OBJECT)) {
				obj = (storage_object_t *) object;
			} else {
				LOG_ERROR("Handle %d is not a handle for a storage object.\n", object_handle);
				rc = EINVAL;
			}
		}

	} else {
		LOG_ERROR("An object handle must be given.\n");
		rc = EINVAL;
	}

	if (rc == 0) {
		list_element_t iter;
		storage_object_t * parent;

		/*
		 * Check to make sure the parent objects don't have parents
		 * themselves, that they are not part of a container, and
		 * that they are not part of a volume.
		 */
		LIST_FOR_EACH(obj->parent_objects, iter, parent) {
			rc = is_in_use(parent);
			if (rc != 0) {
				break;
			}
		}

		if (rc == 0) {
			plugin_record_t * parent_plugin = NULL;

			/*
			 * Get the plug-in that is managing the parents of this
			 * object.
			 */
			rc = get_parent_plugin(obj, &parent_plugin);

			if (rc == 0) {
				/*
				 * If this object has parent objects that are
				 * data objects, warn the user that the data
				 * objects will be destroyed.
				 */
				LIST_FOR_EACH(obj->parent_objects, iter, parent) {
					if (parent->data_type == DATA_TYPE) {
						rc = EEXIST;
						break;
					}
				}

				if (rc != 0) {
					char * choices[] = {"Continue", "Cancel", NULL};
					int answer = 0;	    /* Default is "Continue" */

					rc = engine_user_message(&answer, choices,
								 _("WARNING:  Plug-in %s is producing data objects from object %s.  "
								   "Unassigning plug-in %s from object %s will destroy the data objects.\n"),
								 parent_plugin->short_name, obj->name,
								 parent_plugin->short_name, obj->name);

					if (answer == 1) {
						/* User wants to cancel. */
						rc = E_CANCELED;
					}
				}

				if (rc == 0) {
					rc = parent_plugin->functions.plugin->unassign(obj);

					if (rc == 0) {
						/*
						 * The previous owner may
						 * have trashed stop data while the
						 * object was consumed.
						 */
						obj->flags &= ~SOFLAG_HAS_STOP_DATA;
					}
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return(rc);
}


void free_function_info_array_contents(void * object) {

	int i;
	function_info_array_t * function_array = (function_info_array_t *) object;

	for (i = 0; i < function_array->count; i++) {
		if (function_array->info[i].title != NULL) {
			engine_free(function_array->info[i].title);
		}
		if (function_array->info[i].verb != NULL) {
			engine_free(function_array->info[i].verb);
		}
		if (function_array->info[i].help != NULL) {
			engine_free(function_array->info[i].help);
		}
	}
}

/*
 * Get an array of private actions that are supported by the plug-in that is
 * managing "thing".
 */
int evms_get_plugin_functions(engine_handle_t           thing_handle,
			      function_info_array_t * * actions) {

	int rc = 0;
	void * object = NULL;
	object_type_t type;
	function_info_array_t * function_array = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_plugin_functions(thing_handle, actions);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (thing_handle == 0) {
		LOG_ERROR("A handle must be given.\n");
		rc = EINVAL;
	}

	rc = translate_handle(thing_handle,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}
	switch (type) {
		case PLUGIN:
			{
				plugin_record_t * plugin = (plugin_record_t *) object;

				switch (GetPluginType(plugin->id)) {
					case EVMS_DEVICE_MANAGER:
					case EVMS_SEGMENT_MANAGER:
					case EVMS_REGION_MANAGER:
					case EVMS_FEATURE:
					case EVMS_ASSOCIATIVE_FEATURE:
						rc = plugin->functions.plugin->get_plugin_functions(NULL, &function_array);
						break;

					case EVMS_FILESYSTEM_INTERFACE_MODULE:
						rc = plugin->functions.fsim->get_plugin_functions(NULL, &function_array);
						break;

					case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
						rc = plugin->functions.cluster->get_plugin_functions(NULL, &function_array);
						break;

					default:
						LOG_ERROR("Plug-in %s has an unknown type of %#x.  Can't get plug-in functions for that type of plug-in.\n", plugin->short_name, GetPluginType(plugin->id));
						rc = ENOSYS;
						break;
				}
			}
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			{
				storage_object_t * obj = (storage_object_t *) object;

				rc = obj->plugin->functions.plugin->get_plugin_functions(obj, &function_array);
			}
			break;

		case CONTAINER:
			{
				storage_container_t * con = (storage_container_t *) object;

				rc = con->plugin->container_functions->get_plugin_functions(con, &function_array);
			}
			break;

		case VOLUME:
			{
				logical_volume_t * vol = (logical_volume_t *) object;

				if (vol->file_system_manager != NULL) {
					rc = vol->file_system_manager->functions.fsim->get_plugin_functions(vol, &function_array);

				} else {
					rc = ENOSYS;
				}
			}
			break;

		default:
			LOG_ERROR("Handle %d is of type %d which is not valid for this function.\n", thing_handle, type);
			rc = EINVAL;
	}

	if (rc == 0) {
		/*
		 * Allocate a user memory block and copy the action info
		 * to the user memory.
		 */
		uint size = sizeof(function_info_array_t) + function_array->count * sizeof(function_info_t);

		*actions = alloc_app_struct(size, free_function_info_array_contents);

		if (*actions != NULL) {
			/* We do not copy the strings, just the pointers to the strings. */
			memcpy(*actions, function_array, size);
		} else {
			rc = ENOMEM;
		}

		/*
		 * Free the action array returned by the plug-in but don't
		 * free the strings.  The user copy now has the pointers to the strings.
		 * the strings will be freed when the user frees the user copy of
		 * the function_info_array_t.
		 */
		engine_free(function_array);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Execute a plug-in private action on a "thing".
 */
int evms_do_plugin_function(engine_handle_t  thing_handle,
			    task_action_t    action,
			    handle_array_t * objects,
			    option_array_t * options) {

	int rc = 0;
	void * object = NULL;
	object_type_t type;
	list_anchor_t object_list;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_do_plugin_function(thing_handle, action, object, options);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (thing_handle == 0) {
		LOG_ERROR("A handle must be given.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	rc = translate_handle(thing_handle,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}
	object_list = allocate_list();

	if (object_list != NULL) {

		rc = make_list(objects, object_list);

		if (rc == 0) {

			switch (type) {
				case PLUGIN:
					{
						plugin_record_t * plugin = (plugin_record_t *) object;

						if (GetPluginType(plugin->id) == EVMS_FILESYSTEM_INTERFACE_MODULE) {
							rc = plugin->functions.fsim->plugin_function(NULL, action, object_list, options);
						} else {
							rc = plugin->functions.plugin->plugin_function(NULL, action, object_list, options);
						}
					}
					break;

				case EVMS_OBJECT:
				case REGION:
				case SEGMENT:
				case DISK:
					{
						storage_object_t * obj = (storage_object_t *) object;

						rc = obj->plugin->functions.plugin->plugin_function(obj, action, object_list, options);
					}
					break;

				case CONTAINER:
					{
						storage_container_t * con = (storage_container_t *) object;

						rc = con->plugin->container_functions->plugin_function(con, action, object_list, options);
					}
					break;

				case VOLUME:
					{
						logical_volume_t * vol = (logical_volume_t *) object;

						if (vol->file_system_manager != NULL) {
							rc = vol->file_system_manager->functions.fsim->plugin_function(vol, action, object_list, options);

						} else {
							rc = ENOSYS;
						}
					}
					break;

				default:
					LOG_ERROR("Handle %d is of type %d which is not valid for this function.\n", thing_handle, type);
					rc = EINVAL;
			}
		}

		destroy_list(object_list);

	} else {
		LOG_CRITICAL("Error allocating memory for an output object list.\n");
		rc = ENOMEM;
	}

	if (rc == 0) {
		/*
		 * A plug-in function can add or remove child objects to/from
		 * the given object.  Since we don't know what went on inside
		 * the set_info() and how it affected other objects, update the
		 * stop data state on all the objects that can have stop data.
		 */
		update_all_stop_data_states();
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

