/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <ncurses.h>
#include <panel.h>
#include <frontend.h>
#include <glib.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "object.h"
#include "views.h"
#include "readable.h"
#include "logging.h"

/**
 *	get_object_handles - get an object's parent and container handles
 *	@handle: handle to the object
 *	@parent: address where to store the parent's object handle
 *	@producing_container: address where to store the producing container handle
 *	@consuming_container: address where to store the consuming container handle
 *
 *	This routine returns the handles for an object's parent, producing
 *	and consuming containers. If the object is topmost (no parent) and
 *	has a volume then the volume handle is returned as the parent handle. 
 **/
int get_object_handles(object_handle_t handle, object_handle_t *parent,
			object_handle_t *producing_container,
			object_handle_t *consuming_container)
{
	gint rc;
	handle_object_info_t *object;

	*parent = 0;
	*producing_container = 0;
	*consuming_container = 0;
    
	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		switch (object->type) {
		case DISK:
		case SEGMENT:
		case REGION:
		case EVMS_OBJECT:
			*producing_container = object->info.object.producing_container;
			*consuming_container = object->info.object.consuming_container;

			if (object->info.object.parent_objects) {
				if (object->info.object.parent_objects->count == 1) {    
					*parent = object->info.object.parent_objects->handle[0];
				} else if (object->info.object.parent_objects->count == 0 &&
						object->info.object.volume != 0) {
					*parent = object->info.object.volume;
				}
			} else if (object->info.object.volume != 0) {
				*parent = object->info.object.volume;
			}
			break;
		default:
			break;
		}
		evms_free(object);
	}
	return rc;
}

/**
 *	get_plugin_handle_for_object - gets the plugin handle of the plugin for the object
 *	@handle: the handle of a storage object
 *
 *	This routine returns the plugin handle of the plugin that
 *	manages the supplied storage object.
 */
inline plugin_handle_t get_plugin_handle_for_object(object_handle_t handle)
{
	plugin_handle_t plugin = 0;
	handle_object_info_t *info;

	if ((evms_get_info(handle, &info)) == 0) {
		plugin = info->info.object.plugin;                
		evms_free(info);
	}
	return plugin;
}

/**
 *	append_child_object - append information about a child object to an object hiearchy view
 *	@clist: the hiearchy list
 *	@object: the address of the object info
 *	@level: the current to use when indenting the child info
 *
 *	This routine appends a clist item to list for an object hiearchy view. We use
 *	the level parameter for proper indentation.
 */
void append_child_object(struct clist *clist, handle_object_info_t *object, int level)
{
	GPtrArray *text;
	gchar *child, *fill;

	fill = g_strnfill(level * 2, ' ');
	child = g_strconcat(fill, object->info.object.name, NULL);

	text = g_ptr_array_new();
	g_ptr_array_add(text, child);
	append_item(clist, text, GUINT_TO_POINTER(object->info.object.handle), NULL);
	g_ptr_array_free(text, TRUE);
	g_free(fill);
}

/**
 *	populate_object_tree - recursively populate an object tree
 *	@clist: the list to append the object info to
 *	@handle: the object handle
 *	@level: the level used in indenting
 *
 *	This routine appends the information for an object to a list displaying
 *	a hiearchy of storage objects. If this object in turn has children of
 *	its own, we recurse to append the children to the list as well.
 */
void populate_object_tree(struct clist *clist, object_handle_t handle, int level)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		append_child_object(clist, object, level);
		if (object->info.object.child_objects != NULL) {
			int i;

			for (i = 0; i < object->info.object.child_objects->count; i++) {
				populate_object_tree(clist,
						object->info.object.child_objects->handle[i],
						level + 1);
			}
		}
		evms_free(object);
	} else {
		log_error("%s: evms_get_info() returned error code %d.\n", __FUNCTION__, rc);
	}
}

/**
 *	jump_to_object_button_activated - invoked when "Jump to" button in Object Tree dialog activated
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the "Jump to" button is activated in the "Object Tree" 
 *	dialog. It switches to the view and focus to the currently focused object in the
 *	object tree clist.
 */
int jump_to_object_button_activated(struct menu_item *item)
{
	GList *element;
	struct clist_item *focus_item;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;
	
	element = get_clist_focus_item(selwin->clist);
	focus_item = element->data;
	jump_to_object(GPOINTER_TO_UINT(focus_item->user_data));
	dialog->status = DLG_STATUS_CLOSING;
	
	return 0;
}

/**
 *	show_object_tree - display a read-only dialog list that shows the object and its descendents
 *	@handle: the object to start with
 *
 *	This routine creats a dialog containing a list of the objects in the hiearchy for a given
 *	starting object.
 */
void show_object_tree(object_handle_t handle)
{
	char *title, *name;
	struct selwin *selwin;
	struct dialog_window *dialog;

	name = get_object_name(handle);
	title = g_strdup_printf(_("Object Tree for %s"), name);

	selwin = create_selection_window(title, NULL, "", _("_OK"),
				(menuitem_activate_cb)close_window_button_activated,
				_("_Jump to"),
				(menuitem_activate_cb)jump_to_object_button_activated,
				NULL);

	dialog = (struct dialog_window *)selwin;
	set_menu_item_sensitivity(dialog->next_button, TRUE);
	set_menu_item_visibility(dialog->cancel_button, FALSE);
	set_horizontal_menu_focus(dialog->menu, dialog->next_button);

	set_clist_select_item_cb(selwin->clist, (clist_select_item_cb)disallow_select_events);
	set_clist_unselect_item_cb(selwin->clist, (clist_unselect_item_cb)disallow_select_events);
	set_clist_column_count(selwin->clist, 1);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 1.00),
				0,
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, _("Storage Object"));

	populate_object_tree(selwin->clist, handle, 0);
	process_modal_dialog(dialog);	

	g_free(name);
	g_free(title);
}

/**
 *	display_object_tree_menuitem_activated - invoked when "Display Object Tree" menuitem activated
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the "Display Object Tree" menu item is activated in the
 *	popup context menu.
 */
int display_object_tree_menuitem_activated(struct menu_item *item)
{
	show_object_tree(GPOINTER_TO_UINT(item->user_data));
	return 0;
}
