/****************************************************************\
*                                                                *
*  A simple bitarray data structure                              *
*                                                                *
*  Guy St.C. Slater..   mailto:guy@ebi.ac.uk                     *
*  Copyright (C) 2000-2006.  All Rights Reserved.                *
*                                                                *
*  This source code is distributed under the terms of the        *
*  GNU Lesser General Public License. See the file COPYING       *
*  or http://www.fsf.org/copyleft/lesser.html for details        *
*                                                                *
*  If you use this code, please keep this notice intact.         *
*                                                                *
\****************************************************************/

#include "bitarray.h"

/**/

#ifndef GetBit
#define GetBit(i,n) (((i)>>(n))&1)
#endif /* GetBit */

#ifndef BitOn
#define BitOn(i,n) ((i)|1<<(n))
#endif /* BitOn */

#ifndef BitOff
#define BitOff(i,n) ((i)&~(1<<(n)))
#endif /* BitOff */


/**/

BitArray *BitArray_create(void){
    register BitArray *ba = g_new(BitArray, 1);
    ba->alloc = 16;
    ba->length = 0;
    ba->data = g_new0(guchar, ba->alloc);
    return ba;
    }

void BitArray_destroy(BitArray *ba){
    g_free(ba->data);
    g_free(ba);
    return;
    }

void BitArray_info(BitArray *ba){
    register gint64 i;
    g_print("BitArray length [%d] alloc [%d] data [",
            (gint)ba->length, (gint)ba->alloc);
    for(i = 0; i < ba->length; i++)
        g_print("%d", BitArray_get_bit(ba, i));
    g_print("]\n");
    return;
    }

void BitArray_empty(BitArray *ba){
    ba->length = 0;
    return;
    }

gsize BitArray_get_size(BitArray *ba){
    return (ba->length / (sizeof(guchar) * CHAR_BIT))
         + ((ba->length % (sizeof(guchar) * CHAR_BIT))?1:0);
    }

void BitArray_write(BitArray *ba, FILE *fp){
    fwrite(ba->data, sizeof(guchar), BitArray_get_size(ba), fp);
    fflush(fp); /* Keep valgrind quiet ?? */
    return;
    }

BitArray *BitArray_read(FILE *fp, gsize size){
    register BitArray *ba = g_new(BitArray, 1);
    ba->alloc = size;
    ba->length = size * CHAR_BIT;
    ba->data = g_new(guchar, ba->alloc);
    fread(ba->data, sizeof(guchar), size, fp);
    return ba;
    }

void BitArray_append_bit(BitArray *ba, gboolean bit){
    register gint64 word = ba->length / CHAR_BIT,
                    pos = ba->length & (CHAR_BIT-1);
    if(ba->length == (ba->alloc*CHAR_BIT)){
        ba->alloc <<= 1;
        ba->data = g_realloc(ba->data, ba->alloc);
        }
    ba->data[word] = bit?BitOn(ba->data[word], pos)
                        :BitOff(ba->data[word], pos);
    ba->length++;
    return;
    }

void BitArray_append(BitArray *ba, guint64 data, guchar width){
    register guchar i;
    g_assert(width <= (sizeof(guint64)*CHAR_BIT));
    for(i = 0; i < width; i++)
        BitArray_append_bit(ba, GetBit(data, i));
    return;
    }
/* FIXME: optimisation: replace by non-bit-by-bit version */

gboolean BitArray_get_bit(BitArray *ba, guint64 pos){
    register gint64 word = pos / CHAR_BIT,
                     bit = pos & (CHAR_BIT-1);
    return GetBit(ba->data[word], bit);
    }

guint64 BitArray_get(BitArray *ba, guint64 start, guchar width){
    register gint i;
    register guint64 data = 0;
    for(i = 0; i < width; i++)
        data |= (BitArray_get_bit(ba, start+i) << i);
    return data;
    }
/* FIXME: optimisation: replace by non-bit-by-bit version */


/**/


