/****************************************************************\
*                                                                *
*  Sparse Cache Object                                           *
*                                                                *
*  Guy St.C. Slater..   mailto:guy@ebi.ac.uk                     *
*  Copyright (C) 2000-2006.  All Rights Reserved.                *
*                                                                *
*  This source code is distributed under the terms of the        *
*  GNU Lesser General Public License. See the file COPYING       *
*  or http://www.fsf.org/copyleft/lesser.html for details        *
*                                                                *
*  If you use this code, please keep this notice intact.         *
*                                                                *
\****************************************************************/

#include "sparsecache.h"

/**/

static void SparseCache_Page_destroy(SparseCache_Page *page){
    g_free(page->data);
    g_free(page);
    return;
    }

SparseCache *SparseCache_create(gint length,
                                SparseCache_FillFunc fill_func,
                                SparseCache_EmptyFunc empty_func,
                                gpointer user_data){
    register SparseCache *sc = g_new(SparseCache, 1);
    g_assert(fill_func);
    sc->ref_count = 1;
    sc->page_total = (length >> SparseCache_PAGE_SIZE_BIT_WIDTH) + 1;
    sc->page_used = 0;
    sc->page_list = g_new0(SparseCache_Page*, sc->page_total);
    sc->length = length;
    sc->fill_func = fill_func;
    sc->empty_func = empty_func;
    sc->user_data = user_data;
    return sc;
    }

void SparseCache_destroy(SparseCache *sc){
    register SparseCache_Page *page;
    register gint i;
    if(--sc->ref_count)
        return;
    for(i = 0; i < sc->page_total; i++){
        page = sc->page_list[i];
        if(page){
            if(sc->empty_func)
                sc->empty_func(page, sc->user_data);
            else
                SparseCache_Page_destroy(page);
            }
        }
    g_free(sc->page_list);
    g_free(sc);
    return;
    }

SparseCache *SparseCache_share(SparseCache *sc){
    sc->ref_count++;
    return sc;
    }

gpointer SparseCache_get(SparseCache *sc, gint pos){
    register gint page_id = SparseCache_pos2page(pos);
    register SparseCache_Page *page = sc->page_list[page_id];
    g_assert(pos >= 0);
    g_assert(pos < sc->length);
    if(!page){
        page = sc->fill_func((page_id << SparseCache_PAGE_SIZE_BIT_WIDTH),
                             sc->user_data);
        sc->page_list[page_id] = page;
        sc->page_used++;
        }
    return page->get_func((pos & (SparseCache_PAGE_SIZE-1)), page->data,
                          sc->user_data);
    }

/**/

