/*  extipl.c for FreeBSD/Linux
 *	extipl is a simple boot-selector which makes coexistence of
 *	several operating systems on your machine.
 *				Auther: KIMURA Takamichi<takamiti@tsden.org>
 *				last update : 2001/07/01
 *
 *  $Id: extipl.c,v 1.4 2001/07/02 15:35:12 takamiti Exp $
 *
 * CAUTION:
 *   TAKE CARE!
 *   I(WE) HAVE NOT SHARE IN THE TROUBLES, RUNNING BY YOUR OWN RISKS.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <regex.h>
#ifdef __linux__
#include <linux/unistd.h>
#endif
#include "extipl.h"
#include "sysident.h"
#include "extndipl.src"

#define DEVDIR			"/dev"
#define FD_DEVICE		"/dev/.*fd.*"
#define FILE_CREATE		(O_CREAT | O_TRUNC | O_RDWR)
#define FILE_MODE		(S_IRUSR | S_IWUSR)
#define FILE_RDONLY		S_IRUSR
#define MODE_RDONLY		1
#define MODE_RDWR		0

#define OK			 1
#define OK_NOMSG		 0
#define ERR			-1
#define DEBUG_OUT		"_BootSec"

#define DEFAULT_INSTALL		"gemini"
#define AUTOSAVE_BASE		"fdiskIPL"
#define PROVSAVE_NAME		"master.ipl"

static int  help(char *, int, char **);
static char *takedevname(char *);
static int  rdipl(char *, char *, struct offset_s *, int);
static int  wripl(char *, char *, struct offset_s *, int);
static int  save(char *, char *, int, int);
static int  load(char *, char *, int);
static int  isfdready(char *, char *);
static int  chk_fddev(char *);
static int  save_prompt(char*, char *, char *, int, int);
static struct inneripl_s *choose_iplcode(char *);
static int  copy_innercode(char *, char *, int, char **);
static int  load_extcode(int, char **, char *, char **);
static int  install(char *, int, char **);
static int  fdtest(char *, int, char **);
static int  saveipl(char *, int, char **);
static int  restore(char *, int, char **);
static int  clrboot(char *, int, char **);
static struct sysident_s *get_sysidnt(int);
static void free_extinfo(struct extinfo_s *);
static int  rd_extpart(char *, struct extinfo_s *);
static int  wr_extpart(char *, struct extinfo_s *);
static int  show_extpart(struct extinfo_s *);
static int  show_mastertable(char *, struct partition_s *, int);
static int  showtbl(char *, int, char **);
static int  get_bootable(int, int);
static int  chk_systype(int, struct partition_s *);
static int  touch_extflag(char *, struct partition_s *, int);
static int  setboot(char *, int, char **);
static void tblsort(struct partition_s *);
static void tblpack(struct partition_s *);
static char *ask(char *);
static int  sure(char *);
#ifdef __linux__
static _syscall5(int, _llseek,  uint,  fd, ulong, hi, ulong, lo, loff_t *, res, uint, wh);
static int long_seek(int, off_t, int);
#endif
/*================ for DEBUG ======================== */
static void hexdump(char *, int);
static int  debug_out(char *, struct offset_s *, int);

typedef struct cmdtab_s {
	char *command;
	int (*func)(char *, int, char **);
	char *help;
} cmdtab_s;

static struct cmdtab_s sub_commands[] = {
	{"install", install,	"install boot-loader on specified device" },
	{"fdtest",  fdtest,	"testing boot-loader from diskette" },
	{"save",    saveipl,	"save current IPL code to file" },
	{"restore", restore,	"restore last IPL code from file"},
	{"show",    showtbl,	"show partition table"},
	{"chgboot", setboot,	"change bootable partition"},
	{"clrboot", clrboot,	"clear active flag"},
	{"help",    help,	"show this"},
	{NULL, NULL, NULL}
    };

static struct inneripl_s inner_ipl[] = {
	{"gemini", 2, 
	 "has friendly menu, works CHS/LBA(holds first 18 sectors max)",
	    { {polluxIPL, sizeof(polluxIPL)},   {castorIPL, sizeof(castorIPL)} },
	    { {polluxAIPL, sizeof(polluxAIPL)}, {castorIPL, sizeof(castorIPL)} }  },
	{"taurus", 1, 
	 "boot from only active part in the first HDD, works CHS/LBA",
	    { {aldebaranIPL, sizeof(aldebaranIPL)}, {NULL, 0} },
	    { {aldebaranIPL, sizeof(aldebaranIPL)}, {NULL, 0} }  },
	{"aquila", 1, 
	 "smallest selector, choose from only first HDD, works CHS/LBA",
	    { {altairIPL,  sizeof(altairIPL)},  {NULL, 0} },
	    { {altairAIPL, sizeof(altairAIPL)}, {NULL, 0} }  },
	{"scorpius", 1, 
	 "once called 'ExtendedIPL', works only CHS mode",
	    { {antaresIPL,  sizeof(antaresIPL)},  {NULL, 0} },
	    { {antaresAIPL, sizeof(antaresAIPL)}, {NULL, 0} }  },
    };

static offset_s mbr;
static int opt_sort = 0;
static int opt_pack = 0;
static int opt_debug = 0;
static int opt_force = 0;

main(argc, argv)
int argc;
char **argv;
{
    char *device;
    struct cmdtab_s *cmdtab;
    int n, r;

    while ((n = getopt(argc, argv, "dfFhsp?")) != EOF) {
	switch(n) {
	case 'h':
	case '?': exit(help(NULL, 0, NULL));
	case 's': opt_sort = 1; break;
	case 'p': opt_pack = 1; break;
	case 'd': opt_debug = 1; break;
	case 'f': opt_force = 1; break;
	case 'F': opt_force = 2; break;
	default:  fprintf(stderr, "unknown option '%c'\n", n); exit(1);
	}
    }
    argc -= optind;
    argv += optind;

    if (argc < 2) exit(help(NULL, 0, NULL));
    if (opt_sort && opt_pack) opt_sort = 0;

#ifdef DEBUG
    opt_debug = 1;
#endif

    for(cmdtab = sub_commands; cmdtab->command != NULL; cmdtab++) {
	if (strcmp(*argv, cmdtab->command) == 0) break;
    }
    if (cmdtab->command == NULL) {
	fprintf(stderr, "Unknown command \"%s\"\n", *argv);
	exit(ERR);
    }

    if (--argc <= 0)
	exit(help(NULL, 0, NULL));

    argv++;
    device = takedevname(*argv);

    mbr.chs.head = mbr.chs.cyl = 0;
    mbr.chs.sect = 1;
    mbr.lba = 0;

    r = (*(cmdtab->func))(device, --argc, ++argv);
    if (r != OK_NOMSG) {
	fprintf(stderr, "%s.\n", (r == ERR) ? "Aborted": "Ok");
    }
    exit(r == ERR ? r: 0);
}

static int help(arg1, n, arg2)
char *arg1, **arg2;
int n;
{
    struct cmdtab_s *cmdtab;
    struct inneripl_s *ipl;
    int i;

    fprintf(stderr, "|\n|\t*** Extended IPL ver %s %s ***\n", VERSION, DATE);
    fprintf(stderr, "|Usage: extipl command device-name [@inner-iplname or arg1..argN]\n");
    fprintf(stderr, "|\n| vaild commands are:\n");
    for(cmdtab = sub_commands; cmdtab->command != NULL; cmdtab++)
	if (*cmdtab->help)
	    fprintf(stderr,"|   %-9s- %s\n", cmdtab->command, cmdtab->help);
    fprintf(stderr, "|\n| inner-iplname are follows:\n");
    for(i = 0, ipl = inner_ipl; i < sizeof(inner_ipl)/sizeof(struct inneripl_s); i++, ipl++)
	fprintf(stderr, "|   %-9s: %s\n", ipl->iplname, ipl->note);
    return(OK_NOMSG);
}

static char *basename(path)
char *path;
{
    char *p;
    p = strrchr(path, '/');
    return(p == NULL ? path : p + 1);
}

static char *takedevname(argv)
char *argv;
{
    static char name[LBUF_SIZE];
    char *device, buf[SECTOR_SIZE];
    struct stat st;
    int fd;

    if (argv == NULL) {
	fprintf(stderr, "extipl : no device name\n");
	exit(ERR);
    }
    if (strncmp(argv, DEVDIR, strlen(DEVDIR)) == 0) {
	device = argv;
    } else {
	snprintf(name, LBUF_SIZE, "%s/%s", DEVDIR, argv);
	device = name;
    }

    if (stat(device, &st) == -1) {
	perror(device);
	exit(ERR);
    }
    if (!(st.st_mode & S_IFCHR)) {
	fprintf(stderr,"%s: not character special\n", device);
	exit(ERR);
    }
    return(device);
}

static int rdipl(device, buf, offset, len)
char *device, *buf;
struct offset_s *offset;
int len;
{
    int fd, n;

    if ((fd = open(device, O_RDONLY)) < 0) {
	perror(device);
	return(ERR);
    }
#ifdef __linux__
    long_seek(fd, (off_t)offset->lba, SEEK_SET);
#else
    lseek(fd, (off_t)offset->lba * SECTOR_SIZE, SEEK_SET);
#endif
    if ((n = read(fd, buf, len)) != len) {
	perror("rdipl");
    }
    close(fd);
    return(n);
}

static int wripl(device, buf, offset, len)
char *device, *buf;
struct offset_s *offset;
int len;
{
    int fd, r;

    if (offset->lba == 0) {
	if (opt_sort) {
	    tblsort((struct partition_s *)(buf + TABLE_OFFSET));
	} else if (opt_pack) {
	    tblpack((struct partition_s *)(buf + TABLE_OFFSET));
	}
    }

    if (opt_debug)
	return(debug_out(buf, offset, len));

    if ((fd = open(device, O_RDWR)) < 0) {
	perror(device);
	return(ERR);
    }
#ifdef __linux__
    long_seek(fd, (off_t)offset->lba, SEEK_SET);
#else
    lseek(fd, (off_t)offset->lba * (off_t)SECTOR_SIZE, SEEK_SET);
#endif
    if ((r = write(fd, buf, len)) != len) {
	perror("wripl");
    }
    close(fd);
    sync();
    return(r);
}

static int adj_sectorbound(len)
int len;
{
    len = (len + SECTOR_SIZE - 1) / SECTOR_SIZE;
    len *= SECTOR_SIZE;
    return(len);
}

static int save(file, buf, len, mode)
char *file, *buf;
int len, mode;
{
    int fd, i;

    if (*file == 0) {
	i = 1;
	do {
	    sprintf(file, "%s.%03d", AUTOSAVE_BASE, i++);
	    if (i > 999) return(ERR);
	} while (access(file, F_OK) == 0);
    }
    fd = open(file, FILE_CREATE, FILE_MODE);
    if (fd < 0 || write(fd, buf, len) != len) {
	perror(file);
	return(ERR);
    }
    if (mode == MODE_RDONLY)
	fchmod(fd, FILE_RDONLY);
    close(fd);
    return(OK);
}

static int load(file, buf, len)
char *file, *buf;
int len;
{
    int	fd, r;

    fd = open(file, O_RDONLY);
    if (fd < 0 || (r = read(fd, buf, len)) <= 0) {
	perror(file);
	return(ERR);
    }
    close(fd);
    if (opt_debug)
	fprintf(stderr, "ipl-code: %s, %d bytes\n", file, r);
    if (r == 0) {
	fprintf(stderr, "%s: empty file\n");
	r = ERR;
    }
    return(r);
}

static int isfdready(name, device)
char *name, *device;
{
    printf("Please insert blank diskette in FLOPPY unit.\n");
    printf("Write [%s] to \"%s\"", name, device);
    return(sure(NULL) ? OK : ERR);
}

static int chk_fddev(device)
char *device;
{
    regex_t exp;
    int n;

    if ((n = regcomp(&exp, FD_DEVICE, REG_EXTENDED | REG_NOSUB)) == 0) {
	n = regexec(&exp, device, (size_t)0, NULL, 0);
	regfree(&exp);
    }
    if (n != 0) {
	printf("\"%s\" is correct name of floppy device", device);
	if (!sure(NULL)) return(ERR);
    }
    return(OK);
}

static int save_prompt(dev, name, buff, len, flag)
char *dev, *name, *buff;
int len, flag;
{
    char *filename, tmpname[LBUF_SIZE];
    switch(flag) {
    case 0:
	printf("*** Before exchange the master boot program,\n");
	printf("*** You had better keep the original IPL code.\n");
	filename = ask("Enter file name to save:");
	if (save(filename, buff, len, MODE_RDONLY) != OK)
	    return(ERR);
	printf("Current IPL saved to '%s'.\n", filename);
	printf("Install [%s] to \"%s\"", name, dev);
	if (!sure(NULL))
	    return(ERR);
	break;
    case 1:
	*tmpname = 0;
	save(tmpname, buff, len, MODE_RDONLY);
	break;
    }
    return(OK);
}

static struct inneripl_s *choose_iplcode(name)
char *name;
{
    struct inneripl_s *p;
    int i;

    for(i = 0, p = inner_ipl; i < sizeof(inner_ipl)/sizeof(struct inneripl_s); i++, p++) {
	if (strcmp(p->iplname, name) == 0) return(p);
    }
    fprintf(stderr, "@%s: unknown loader\n", name);
    return(NULL);
}

static int copy_innercode(target, iplbuf, hdcode, name)
char *target, *iplbuf, **name;
int hdcode;
{
    struct inneripl_s *p;
    struct incode_s *q;
    int len;

    if ((p = choose_iplcode(target)) == NULL)
	return(ERR);

    q = hdcode ? p->hd_code : p->fd_code;
    memcpy(iplbuf, q->code, q->len);
    len = SECTOR_SIZE;
    if (p->nr_part > 1) {
	q++;
	iplbuf += SECTOR_SIZE;
	if (memcmp(iplbuf, q->code, 64) == 0) {		/* gemini only */
	    memcpy(iplbuf + SECTOR_SIZE, q->code + SECTOR_SIZE , q->len - SECTOR_SIZE);
	} else {
	    memcpy(iplbuf, q->code, q->len);
	}
	len += q->len;
    }
    *name = p->iplname;
    return(len);
}

static int load_extcode(ac, arg, iplbuf, name)
int ac;
char **arg, *iplbuf, **name;
{
    int len;
    static char iplname[LBUF_SIZE];

    *name = iplname;
    *iplname = 0;
    len = 0;
    if (ac >= 1) {
	len = load(*arg, iplbuf, SECTOR_SIZE);
	strncpy(iplname, basename(*arg), IPLNAME_LEN);
	if (len <= 0) return(ERR);
    }
    if (ac >= 2) {
	arg++;
	len = load(*arg, iplbuf + SECTOR_SIZE, GEMINI_CODELN);
	strcat(iplname, "&");
	strncat(iplname, basename(*arg), IPLNAME_LEN);
	if (len <= 0) return(ERR);
	len += SECTOR_SIZE;
    } else {
	len = SECTOR_SIZE;
    }
    return(len);
}

static int chk_holdblocks(buff, len, name)
char *buff, *name;
int len;
{
    struct partition_s table[NR_PARTITION];
    int holdblks;

    holdblks = (len + SECTOR_SIZE - 1) / SECTOR_SIZE;
    memcpy((char *)table, buff + TABLE_OFFSET, TABLE_SIZE);
    tblpack(table);
    if (opt_debug) {
	printf("#1 sysid = 0x%02X, start = %ld, len = %ld\n",
		table[0].sysind, table[0].sector_offset, table[0].nr_sector);
    }
    if (table[0].sector_offset < holdblks) {
	fprintf(stderr, "No sectors enough to install [%s] loader\n", name);
	return(ERR);
    }
    return(OK);
}

static int install(device, ac, arg)
char *device, **arg;
int ac;
{
    char *sectbuf, *iplbuf, *name;
    int len, r;

    if ((sectbuf = malloc(GEMINI_SIZE * 2)) == NULL) {
	perror("malloc");
	return(ERR);
    }

    iplbuf = sectbuf + GEMINI_SIZE;
    if (rdipl(device, sectbuf, &mbr, GEMINI_SIZE) != GEMINI_SIZE) {
	free(sectbuf);
	return(ERR);
    }

    if (*(unsigned short *)(sectbuf + IPL_MAGIC_POS) != IPL_MAGIC) {
	fprintf(stderr, "%s: no IPL magic, corrupted boot sector\n", device);
	free(sectbuf);
	return(ERR);
    }

    memcpy(iplbuf, sectbuf, GEMINI_SIZE);
    len = 0;
    if (ac == 0 || (ac == 1 && **arg == '@')) {
	len = copy_innercode(ac == 0 ? DEFAULT_INSTALL : *arg + 1, iplbuf, 1, &name);
    } else {
	len = load_extcode(ac, arg, iplbuf, &name);
    }

    if (len >= SECTOR_SIZE && chk_holdblocks(sectbuf, len, name) == OK) {
	r = save_prompt(device, name, sectbuf, len > SECTOR_SIZE ? GEMINI_SIZE : SECTOR_SIZE, opt_force);
	if (r == OK) {
	    len = adj_sectorbound(len);
	    r = (wripl(device, iplbuf, &mbr, len) == len ? OK: ERR);
	}
    } else
	r = ERR;
    free(sectbuf);
    return(r);
}

static int fdtest(device, ac, arg)
char *device, **arg;
int ac;
{
    char *sectbuf, *name;
    int len, r;

    if (chk_fddev(device) == ERR)
	return(ERR);

    if ((sectbuf = malloc(GEMINI_SIZE)) == NULL) {
	perror("malloc");
	return(ERR);
    }

    memset(sectbuf, 0, GEMINI_SIZE);
    len = 0;
    if (ac == 0 || (ac == 1 && **arg == '@')) {
	len = copy_innercode(ac == 0 ? DEFAULT_INSTALL: *arg + 1, sectbuf, 0, &name);
    } else {
	len = load_extcode(ac, arg, sectbuf, &name);
    }

    if (len >= SECTOR_SIZE) {
	*(unsigned short *)(sectbuf + IPL_MAGIC_POS) = IPL_MAGIC;
	r = isfdready(name, device);
	len = adj_sectorbound(len);
	if (r == OK)
	    r = (wripl(device, sectbuf, &mbr, len) == len ? OK: ERR);
    } else
	r = ERR;
    free(sectbuf);
    return(r);
}

static int saveipl(device, ac, arg)
char *device, **arg;
int ac;
{
    struct inneripl_s *p;
    char *sectbuf;
    int len, r;

    if ((p = choose_iplcode(DEFAULT_INSTALL)) == NULL)
	return(ERR);

    len = p->nr_part == 1 ? 1 : GEMINI_BLKLEN;
    if (ac >= 1 && **arg == '@') {
	len = atoi(*arg + 1);
	if (!between(len, 1, GEMINI_BLKLEN)) {
	    fprintf(stderr, "%s: invalid argument\n", *arg);
	    return(ERR);
	}
	ac--;
	arg++;
    }

    if (ac == 0) {
	fprintf(stderr, "Please specify filename to save.\n");
	return(ERR);
    }

    len *= SECTOR_SIZE;
    if ((sectbuf = malloc(len)) == NULL) {
	perror("malloc");
	return(ERR);
    }

    if (rdipl(device, sectbuf, &mbr, len) == len)
	r = save(*arg, sectbuf, len, MODE_RDONLY);
    else
	r = ERR;
    free(sectbuf);
    return(r);
}

static int restore(device, ac, arg)
char *device, **arg;
int ac;
{
    char *buff, *sectbuf, *wrbuf, ch;
    int len, r;

    if ((buff = malloc(GEMINI_SIZE * 2)) == NULL) {
	perror("malloc");
	return(ERR);
    }

    sectbuf = buff + GEMINI_SIZE;
    if (ac == 0 || (len = load(*arg, buff, GEMINI_SIZE)) < SECTOR_SIZE) {
	free(buff);
	return(ERR);
    }

    if (opt_debug) {
	char *p;
	int i;
	for(p = buff + TABLE_OFFSET, i = 0; i < 4; i++, p +=TBL_ENTRY_SIZE)
	    hexdump(p, TBL_ENTRY_SIZE);
	hexdump(buff + 510, 2);
    }

    r = 0;
    if (*(unsigned short *)(buff + IPL_MAGIC_POS) != IPL_MAGIC) {
	fprintf(stderr, "%s: no IPL magic, corrupted image file\n", arg);
	r++;
    }

    if (r == 0 && rdipl(device, sectbuf, &mbr, len) != len)
	r++;

    if (r == 0 && *(unsigned short *)(sectbuf + IPL_MAGIC_POS) != IPL_MAGIC) {
	fprintf(stderr, "%s: no IPL magic, corrupted boot sector\n", device);
	r++;
    }

    if (r != 0) {
	free(buff);
	return(ERR);
    }

    printf("\nC)ode:  restore ipl code only");
    printf("\nT)able: restore partition table only");
    printf("\nA)ll:   restore ipl code and partition table");
    ch = tolower(*ask("\n  Restore(c/t/a)?"));
    wrbuf = buff;
    switch(ch) {
    case 'c':
	printf("\nRestore ipl code only");
	memcpy(buff + TABLE_OFFSET, sectbuf + TABLE_OFFSET, TABLE_SIZE);
	break;
    case 't':
	printf("\nRestore partition table only");
	memcpy(sectbuf + TABLE_OFFSET, buff + TABLE_OFFSET, TABLE_SIZE);
	wrbuf = sectbuf;
	len = SECTOR_SIZE;
	break;
    case 'a':
	printf("\nOver write whole data(ipl-code and partition table)");
	break;
    default:
	free(buff);
	return(ERR);
    }
    len = adj_sectorbound(len);
    if (sure(NULL) && wripl(device, wrbuf, &mbr, len) == len)
	r = OK;
    else
	r = ERR;
    free(buff);
    return(r);
}

static int clrboot(device, ac, arg)
char *device, **arg;
int ac;
{
    char diskbuf[SECTOR_SIZE];
    struct partition_s *p;
    int i;

    if (rdipl(device, diskbuf, &mbr, SECTOR_SIZE) != SECTOR_SIZE)
	return(ERR);

    p = (struct partition_s *)(diskbuf + TABLE_OFFSET);
    for(i = 0; i < NR_PARTITION; i++, p++) {
	p->bootind &= 0x7f;
    }
    return(wripl(device, diskbuf, &mbr, SECTOR_SIZE) == SECTOR_SIZE ? OK: ERR);
}

static struct sysident_s *get_sysidnt(id)
int id;
{
    struct sysident_s *si;

    for(si = sysident; si->id >= 0; si++) {
	if (si->id == id) break;
    }
    return(si);
}

static void free_extinfo(base)
struct extinfo_s *base;
{
    struct extinfo_s *p;
    while(base != NULL) {
	p = base->next;
	free(base);
	base = p;
    }
}

static int rd_extpart(device, base)
char *device;
struct extinfo_s *base;
{
    int count, i;
    char buf[SECTOR_SIZE];
    struct extinfo_s *this,*q;
    struct partition_s *p;

    count = 0;
    this = base;
    while(this != NULL) {
	this->next = NULL;
	count++;
	if (rdipl(device, buf, &this->offset, SECTOR_SIZE) != SECTOR_SIZE) {
	    free_extinfo(base->next);
	    return(ERR);
	}
	memcpy((char *)this->table, buf + TABLE_OFFSET, TABLE_SIZE);
	if (count >= DIG_LIMIT) break;
	q = NULL;
	for(p = this->table, i = 0; i < 4; i++, p++) {
	    if (!is_extpart(p->sysind)) continue;
	    if ((q = malloc(sizeof(struct extinfo_s))) == NULL) {
		perror("malloc");
		free_extinfo(base->next);
		return(ERR);
	    }
	    this->next = q;
	    q->offset.lba = base->offset.lba + (ul_long)p->sector_offset;
	    q->offset.chs = p->start_chs;
	    break;
	}
	this = q;
    }
    return(count);
}

static int wr_extpart(device, base)
char *device;
struct extinfo_s *base;
{
    char buf[SECTOR_SIZE];

    while(base != NULL) {
	if (rdipl(device, buf, &base->offset, SECTOR_SIZE) != SECTOR_SIZE)
	    return(ERR);
	memcpy(buf + TABLE_OFFSET, (char *)base->table, TABLE_SIZE);
	if (wripl(device, buf, &base->offset, SECTOR_SIZE) != SECTOR_SIZE)
	    return(ERR);
	base = base->next;
    }
    return(OK);
}

static int show_extpart(base)
struct extinfo_s *base;
{
    int i, j, count, bootp;
    struct partition_s *p;
    struct sysident_s *si;

    bootp = count = 0;
    while(base != NULL) {
	count++;
	for(p = base->table, i = 0; i < NR_PARTITION; i++, p++) {
	    if (p->sysind == 0 || (opt_debug == 0 && is_extpart(p->sysind))) continue;
	    if (opt_debug) {
		for(j = 0; j < count; j++) printf("   ");
	    } else
		printf("    ");
	    si = get_sysidnt(p->sysind);
	    printf("%c(%2d) %02X: ", p->bootind & 0x80 ? '*' : ' ', count, p->sysind);
	    if (opt_debug)
		printf("%s", si == NULL ? "unknown" : si->name);
	    else
		printf("%-30s ", si == NULL ? "unknown" : si->name);
	    if (is_extpart(p->sysind))
		printf("\n");
	    else {
		if (bootp == 0 && p->bootind & 0x80) bootp = count;
		if (opt_debug == 0)
		    printf(" %8luMB\n", p->nr_sector >> 11);
		else
		    printf(" ;%luMB@%lu\n", p->nr_sector >> 11, base->offset.lba);
	    }
	}
	base = base->next;
    }
    return(bootp);
}

static int show_mastertable(device, table, flag)
char *device;
struct partition_s *table;
int flag;
{
    struct sysident_s *si;
    struct extinfo_s base;
    int i, n;

    if (*(unsigned short *)(table + 4) != IPL_MAGIC) {
	fprintf(stderr, "%s: IPL Magic not found.\n", device);
	exit(-1);
    }

    n = 0;
    printf("\n=========\nPartition TABLE on \"%s\"\n=========\n", device);
    for(i = 0; i < NR_PARTITION; i++, table++) {
	if (table->sysind == 0) {
	    printf(" [%d]\n", i + 1);
	    continue;
	}
	n++;
	printf("%c[%d] ", table->bootind & 0x80 ? 'A' : ' ', i + 1);
	si = get_sysidnt(table->sysind);
	printf("%02X: %-36s ", table->sysind, si == NULL ? "unknown" : si->name);
	if (is_extpart(table->sysind)) {
	    printf("%10s\n", "--");
	    if (flag) {
		base.offset.chs = table->start_chs;
		base.offset.lba = (ul_long)table->sector_offset;
		if (rd_extpart(device, &base) != ERR) {
		    show_extpart(&base);
		    free_extinfo(base.next);
		}
	    }
	} else {
	    printf("%8luMB\n", table->nr_sector >> 11);
	}
    }
    if (n == 0) {
	fprintf(stderr, "table is empty.\n");
	exit(0);
    }
    return(0);
}

static int showtbl(device, ac, arg)
char *device, **arg;
int ac;
{
    char buf[SECTOR_SIZE];

    if (rdipl(device, buf, &mbr, SECTOR_SIZE) == SECTOR_SIZE) {
	show_mastertable(device, (struct partition_s *)(buf + TABLE_OFFSET), 1);
	return(OK_NOMSG);
    }
    return(ERR);
}

static int get_bootable(start, end)
int start, end;
{
    static char *helpmsg[] = {
	"     c : Clear all bootable flag",
	"     w : Write this table and quit",
	"     q : Quit without write",
	NULL
	};
    char *s, c;
    int r, i;

    while(1) {
	s = ask("\n>>> Select partition to make bootable (? for help):");
	if (isdigit(*s)) {
	    r = atoi(s);
	    if (between(r, start, end)) return(r);
	    c = 0;
	} else
	    c = *s;
	switch(c) {
	case 'c':
		return(0);
	case 'w':
	case 'q':
	case 'b':
		return(c);
	default:
		bell();
	case 'h':
	case '?':
		printf("\n\t%d .. %d : specified bootable partition number.", start, end);
		for(i = 0; helpmsg[i] != NULL; i++)
			printf("\n\t%s.", helpmsg[i]);
		break;
	}
    }
}

static int chk_systype(n, tbl)
int n;
struct partition_s *tbl;
{
    struct sysident_s *si;

    si = get_sysidnt(tbl->sysind);
    if (tbl->sysind == 0 || si->bootable & Boot_Disable) {
	bell();
	if (tbl->sysind == 0)
	    printf("\nWARNING: Partition #%d is empty, ignored.\n", n);
	else
	    printf("\nPartiton #%d(%s) can not bootable..\n", n, si->name);
	(void)sure("Press <Enter> to continue");
	return(0);
    }
    if ((si->bootable & Boot_Enable) == 0) {
	bell();
	printf("\nPartition #%d(%s) specified, \n", n, si->name);
	return(sure("Do you make bootable(y/n)?") ? 1 : 0);
    }
    return(1);
}

static int touch_extflag(device, table, part)
char *device;
struct partition_s *table;
int part;
{
    struct extinfo_s base, *bp;
    struct partition_s *p;
    struct sysident_s *si;
    int count, n, r, i, bootp;

    base.offset.chs = table->start_chs;
    base.offset.lba = (ul_long)table->sector_offset;
    r = rd_extpart(device, &base);
    if (r <= 0) return(-1);

    si = get_sysidnt(table->sysind);
    while(1) {
	printf("\n### Inside [%d] %s\n", part, si->name);
	bootp = show_extpart(&base);
	n = get_bootable(1, r);
	if (n == 'b' || n == 'q' || n == 'w') break;
	if (n > 0) {
	    for(bp = &base, i = 1; i < n; i++) bp = bp->next;
	    for(p = bp->table, i = 0; i < NR_PARTITION; i++) {
		if (p->sysind == 0) continue;
		if (!is_extpart(p->sysind)) break;
	    }
	    if (chk_systype(n, p) == 0) continue;
	}
	bp = &base;
	count = 0;
	do {
	    count++;
	    for(p = bp->table, i = 0; i < NR_PARTITION; i++, p++) {
		if (p->sysind == 0) continue;
		if ((is_extpart(p->sysind) && count < n)
			|| (!is_extpart(p->sysind) && count == n)) {
		    p->bootind |= 0x80;
		} else {
		    p->bootind &= 0x7f;
		}
	    }
	    bp = bp->next;
	} while(bp != NULL);
    }
    if (n == 'w')
	r = wr_extpart(device, &base);
    free_extinfo(base.next);
    if (r == ERR) {
	fprintf(stderr, "%s: can not change active flag\n", device);
	return(-1);
    }
    return(bootp == 0 ? 0 : n);
}

static int setboot(device, ac, arg)
char *device, **arg;
int ac;
{
    char buf[SECTOR_SIZE];
    struct partition_s *mastertbl, *p;
    int n, r, i, loop;

    if (rdipl(device, buf, &mbr, SECTOR_SIZE) != SECTOR_SIZE)
	return(ERR);

    mastertbl = (struct partition_s *)(buf + TABLE_OFFSET);
    loop = 1;
    while(loop) {
	show_mastertable(device, mastertbl, 0);
	n = get_bootable(1, 4);    
	if (n == 'b') continue;
	if (n == 'q' || n == 'w') break;
	if (n > 0) {
	    p = mastertbl + n - 1;
	    if (chk_systype(n, p) == 0) continue;
	    if (is_extpart(p->sysind)) {
		r = touch_extflag(device, p, n);
		if (r < 0) return(ERR);
		if (r == 0 || r == 'b') continue;
		if (r == 'w' || r == 'q') loop = 0;
	    }
	}
	for (p = mastertbl, i = 1; i <= NR_PARTITION; i++, p++) {
	    if (n == i)	p->bootind |= 0x80;
	    else	p->bootind &= 0x7f;
	}
    }
    if (loop == 0) n = r;
    if (n == 'w') {
	return(wripl(device, buf, &mbr, SECTOR_SIZE) == SECTOR_SIZE ? OK: ERR);
    }
    return(OK_NOMSG);
}

static void tblsort(table)
struct partition_s *table;
{
    struct partition_s *p, q;
    int n = NR_PARTITION;

    if (opt_debug)
	printf("sorting partition table\n");
    do {
	for(p = table; p < table + NR_PARTITION - 1; p++) {
	    if (p[0].sysind == 0 || 
	       (p[0].sector_offset > p[1].sector_offset && p[1].sysind != 0)) {
			q = p[0];
			p[0] = p[1];
			p[1] = q;
	    }
	}
    } while(--n > 0);
}

static void tblpack(table)
struct partition_s *table;
{
    struct partition_s *p, q;
    int n = NR_PARTITION;

    if (opt_debug)
	printf("packing partition table\n");
    do {
	for(p = table; p < table + NR_PARTITION - 1; p++) {
	    if (p[0].sysind == 0 || p[1].sysind != 0) {
		q = p[0];
		p[0] = p[1];
		p[1] = q;
	    }
	}
    } while(--n > 0);
}

static char *ask(prompt)
char *prompt;
{
    static char lbuf[LBUF_SIZE];
    char *p, *q;

    printf("%s ", prompt);
    fflush(stdout);
    *lbuf = 0;
    fgets(lbuf, LBUF_SIZE, stdin);
    *(lbuf + strlen(lbuf) - 1) = 0;
    p = lbuf;
    while(*p && isspace(*p)) p++;
    q = p;
    while(*p && !isspace(*p)) p++;
    if (isspace(*p)) *p = 0;
    return(q);
}

static int sure(s)
char *s;
{
    return(tolower(*ask((s == NULL) ? "... Sure(y/n)?" : s)) == 'y');
}

#ifdef __linux__
/* Hacked by Taketoshi Sano <xlj06203@nifty.ne.jp>  */
static int long_seek(fd, offset, whence)
int fd, whence;
off_t offset;
{
    loff_t loffset, result;
    unsigned long loff_hi, loff_lo;

    loffset = (loff_t)offset << 9;
    loff_hi = (unsigned long)(loffset>>32);
    loff_lo = (unsigned long)(loffset & 0xffffffff);
    if (opt_debug) {
	fprintf(stderr, " sector: %lu, loffset: %Lu, loff_hi: %lu, loff_lo: %lu\n",
		offset, loffset, loff_hi, loff_lo);
	fflush(stderr);
    }
    if(_llseek(fd, loff_hi, loff_lo, &result, whence) != 0) {
	perror("llseek");
        return(ERR);
    }
    if (opt_debug) {
	loffset = (ul_long)result >> 9;
	fprintf(stderr, " result: %Lu, sector: %Lu\n", result, loffset);
	fflush(stderr);
    }
    return(OK);
}
#endif


/*=============================================================================
 *		for DEBUG
 *=============================================================================*/
static void hexdump(buf, len)
char *buf;
int len;
{
    int i;
    printf("> %02X", *buf & 0xff);
    for(i = 1; i < len; i++)
	printf(" %02X", *(++buf) & 0xff);
    printf("\n");
}

static int debug_out(buff, offset, len)
char *buff;
struct offset_s *offset;
int len;
{
    char debugout[LBUF_SIZE];
    static int seq = 0;

    sprintf(debugout, "%s.%03d", DEBUG_OUT, seq++);
    fprintf(stderr, "<<DEBUG MODE>> write to \"%s\", %d bytes, offset=%lu\n", debugout, len, offset->lba);
    return(save(debugout, buff, len, MODE_RDWR) == OK ? len: ERR);
}
