/*
	FATSort, utility for sorting FAT directory structures
	Copyright (C) 2004 Boris Leidner <fatsort@formenos.de>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#if defined(linux) || defined(__linux) || defined(__linux__)
#undef __LINUX__
#define __LINUX__	1
#elif defined(__FreeBSD__) || defined(__DragonFly__) || defined(__NetBSD__) || defined(__OpenBSD__) || defined(__APPLE__)
#undef __BSD__
#define __BSD__		1
#endif

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <assert.h>
#include <errno.h>

// used to check if device is mounted
#if defined(__LINUX__)
#include <mntent.h>
#elif defined (__BSD__)
#include <sys/param.h>
#include <sys/ucred.h>
#include <sys/mount.h>
#endif

// file attributes
#define ATTR_READ_ONLY 0x01
#define ATTR_HIDDEN 0x02
#define ATTR_SYSTEM 0x04
#define ATTR_VOLUME_ID 0x08
#define ATTR_DIRECTORY 0x10
#define ATTR_ARCHIVE 0x20
#define ATTR_LONG_NAME (ATTR_READ_ONLY | ATTR_HIDDEN | ATTR_SYSTEM | ATTR_VOLUME_ID)
#define ATTR_LONG_NAME_MASK (ATTR_READ_ONLY | ATTR_HIDDEN | ATTR_SYSTEM | ATTR_VOLUME_ID | ATTR_DIRECTORY | ATTR_ARCHIVE)

// constants for the LDIR structure
#define DE_FREE (char) 0xe5
#define DE_FOLLOWING_FREE 0x00
#define LAST_LONG_ENTRY 0x40

#define DIR_ENTRY_SIZE 32

// maximum path len on FAT file systems (above specification)
#define MAX_PATH_LEN 512

// maximum file len
// (specification: file < 4GB which is 
// maximum clusters in chain * cluster size)
#define MAX_FILE_LEN 0xFFFFFFFF
#define MAX_CHAIN_LENGTH (MAX_FILE_LEN / 1024)

// program information
#define INFO_PROGRAM "FATSort Utility"
#define INFO_VERSION "0.9.7.1"
#define INFO_AUTHOR "Boris Leidner <fatsort(at)formenos.de>"
#define INFO_HEADER INFO_PROGRAM " " INFO_VERSION " by " INFO_AUTHOR
#define INFO_USAGE "\n"	"Usage: fatsort [options] device\n" \
			"\n" \
			"Options:\n" \
			"\t-c\t Ignore case of file names\n" \
			"\t-r\t Sort in reverse order\n" \
			"\t-h\t Print some help\n" \
			"\t-i\t Print file system information only\n" \
			"\t-l\t Print current order of files only\n" \
			"\t-o flag\t Sort order of files where flag is\n" \
			"\t\t\td : directories first (default)\n" \
			"\t\t\tf : files first\n" \
			"\t\t\ta : files and directories are not distinguished\n" \
			"\t-q\t Be quiet\n" \
			"\t-v\t Print version information\n" \
			"\n" \
			"Device must be a FAT16 or FAT32 file system. FAT12 is not supported yet.\n" \
			"\n" \
			"Example: fatsort /dev/sda\n" \
			"\n" \
			"NOTE: THE FILESYSTEM MUST BE CONSISTENT, OTHERWISE YOU MAY DAMAGE IT!\n" \
			"IF SOMEONE ELSE HAS ACCESS TO THE DEVICE HE MIGHT EXPLOIT FATSORT WITH\n" \
			"A FORGED CORRUPT FILESYSTEM! USE THIS PROGRAM AT YOUR OWN RISK!\n"

// macros
#define myerror(msg) fprintf(stderr, "%s: %s\n", __func__, msg);

// Directory entry structures
// Structure for long directory names
struct sLongDirEntry {
	u_char LDIR_Ord;		// Order of entry in sequence
	char LDIR_Name1[10];		// Chars 1-5 of long name
	u_char LDIR_Attr;		// Attributes (ATTR_LONG_NAME must be set)
	u_char LDIR_Type;		// Type
	u_char LDIR_Checksum;		// Short name checksum
	char LDIR_Name2[12];		// Chars 6-11 of long name
	u_int16_t LDIR_FstClusLO;	// Zero
	char LDIR_Name3[4];		// Chars 12-13 of long name
} __attribute__((packed));

// Structure for old short directory names
struct sShortDirEntry {
	char DIR_Name[11];		// Short name
	u_char DIR_Atrr;		// File attributes
	u_char DIR_NTRes;		// Reserved for NT
	u_char DIR_CrtTimeTenth;	// Time of creation in ms
	u_int16_t DIR_CrtTime;		// Time of creation
	u_int16_t DIR_CrtDate;		// Date of creation
	u_int16_t DIR_LstAccDate;	// Last access date
	u_int16_t DIR_FstClusHI;	// Hiword of first cluster
	u_int16_t DIR_WrtTime;		// Time of last write
	u_int16_t DIR_WrtDate;		// Date of last write
	u_int16_t DIR_FstClusLO;	// Loword of first cluster
	u_int32_t DIR_FileSize;		// file size in bytes
} __attribute__((packed));

union sDirEntry {
	struct sShortDirEntry ShortDirEntry;
	struct sLongDirEntry LongDirEntry;
} __attribute__((packed));

// Bootsector structures
// FAT12 and FAT16
struct sFAT12_16 {
	u_char BS_DrvNum;		// Physical drive number
	u_char BS_Reserved;		// Current head
	u_char BS_BootSig;		// Signature
	u_int32_t BS_VolID;		// Volume ID
	char BS_VolLab[11];		// Volume Label
	char BS_FilSysType[8];		// FAT file system type (e.g. FAT, FAT12, FAT16, FAT32)
	u_char unused[448];		// unused space in bootsector
} __attribute__((packed));

// FAT32
struct sFAT32 {
	u_int32_t BS_FATSz32;		// Sectors per FAT
	u_int16_t BS_ExtFlags;		// Flags
	u_int16_t BS_FSVer;		// Version
	u_int32_t BS_RootClus;		// Root Directory Cluster
	u_int16_t BS_FSInfo;		// Sector of FSInfo structure
	u_int16_t BS_BkBootSec;		// Sector number of the boot sector copy in reserved sectors
	char BS_Reserved[12];		// for future expansion
	char BS_DrvNum;			// see fat12/16
	char BS_Reserved1;		// see fat12/16
	char BS_BootSig;		// ...
	u_int32_t BS_VolID;
	char BS_VolLab[11];
	char BS_FilSysType[8];
	u_char unused[420];		// unused space in bootsector
} __attribute__((packed));

union sFATxx {
	struct sFAT12_16 FAT12_16;
	struct sFAT32 FAT32;
} __attribute__((packed));

// First sector = boot sector
struct sBootSector {
	u_char BS_JmpBoot[3];		// Jump instruction (to skip over header on boot)
	char BS_OEMName[8];		// OEM Name (padded with spaces)
	u_int16_t BS_BytesPerSec;	// Bytes per sector
	u_char BS_SecPerClus;		// Sectors per cluster
	u_int16_t BS_RsvdSecCnt;	// Reserved sector count (including boot sector)
	u_char BS_NumFATs;		// Number of file allocation tables
	u_int16_t BS_RootEntCnt;	// Number of root directory entries
	u_int16_t BS_TotSec16;		// Total sectors (bits 0-15)
	u_char BS_Media;		// Media descriptor
	u_int16_t BS_FATSz16;		// Sectors per file allocation table
	u_int16_t BS_SecPerTrk;	// Sectors per track
	u_int16_t BS_NumHeads;		// Number of heads
	u_int32_t BS_HiddSec;		// Hidden sectors
	u_int32_t BS_TotSec32;		// Total sectors (bits 16-47)
	union sFATxx FATxx;
	u_int16_t BS_EndOfBS;		// marks end of bootsector
} __attribute__((packed));

struct sLongDirEntryList {
/*
	list structures for directory entries
	list structure for a long name entry
*/
	struct sLongDirEntry *lde;
	struct sLongDirEntryList *next;
};

struct sDirEntryList {
/*
	list structure for every file with short
	name entries and long name entries
*/
	char *sname, *lname;		// short and long name strings
	struct sShortDirEntry *sde;	// short dir entry
	struct sLongDirEntryList *ldel;	// long name entries in a list
	int32_t entries;		// number of entries
	struct sDirEntryList *next;	// next dir entry
};

struct sClusterChain {
/*
	this structure contains cluster chains
*/
	u_int32_t cluster;
	struct sClusterChain *next;
};

const struct sDirEntryList __INITDIRLIST__ = {0};
const struct sClusterChain __INITCLUSTERCHAIN__ = {0, NULL};

// --------- option flags

u_int32_t OPT_VERSION, OPT_HELP, OPT_INFO, OPT_QUIET, OPT_IGNORE_CASE, OPT_ORDER, OPT_LIST, OPT_REVERSE;

// ----------


void infomsg(char *str, ...) {
/*
	info messages that can be muted with a command line option
*/
	va_list argptr;

	if (!OPT_QUIET) {
		va_start(argptr,str);
		vprintf(str,argptr);
		va_end(argptr);
	}

}

// ----------

// Endian swap functions
#ifndef __BIG_ENDIAN__
#define SwapInt16(i) i
#define SwapInt32(i) i
#else

u_int16_t SwapInt16(u_int16_t value) {

	union {
		u_int16_t ivalue;
		char cvalue[2];
	} u;

	u.ivalue=value;

	char tmp;
	tmp=u.cvalue[0];
	u.cvalue[0]=u.cvalue[1];
	u.cvalue[1]=tmp;

	return u.ivalue;
}

u_int32_t SwapInt32(u_int32_t value) {

	union {
		u_int32_t ivalue;
		char cvalue[4];
	} u;

	u.ivalue=value;

	char tmp;
	tmp=u.cvalue[0];
	u.cvalue[0]=u.cvalue[3];
	u.cvalue[3]=tmp;
	tmp=u.cvalue[1];
	u.cvalue[1]=u.cvalue[2];
	u.cvalue[2]=tmp;

	return u.ivalue;
}
#endif

// List functions

struct sDirEntryList *
	newDirEntry(char *sname, char *lname, struct sShortDirEntry *sde, struct sLongDirEntryList *ldel, u_int32_t entries) {
/*
	create a new directory entry holder
*/
	assert(sname != NULL);
	assert(lname != NULL);
	assert(sde != NULL);
	
	struct sDirEntryList *tmp;

	if ((tmp=malloc(sizeof(struct sDirEntryList)))==NULL) {
		perror(__func__);
		exit(1);
	}
	if ((tmp->sname=malloc(strlen(sname)+1))==NULL) {
		perror(__func__);
		exit(1);
	}
	strcpy(tmp->sname, sname);
	if ((tmp->lname=malloc(strlen(lname)+1))==NULL) {
		perror(__func__);
		exit(1);
	}
	strcpy(tmp->lname, lname);

	if ((tmp->sde=malloc(sizeof(struct sShortDirEntry)))==NULL) {
		perror(__func__);
		exit(1);
	}
	memcpy(tmp->sde, sde, DIR_ENTRY_SIZE);
	tmp->ldel=ldel;
	tmp->entries=entries;
	tmp->next = NULL;
	return tmp;
}

struct sLongDirEntryList *
	newLongDirEntry(struct sLongDirEntry *lde) {
/*
	create new long dir name list entry
*/
	assert(lde != NULL);

	struct sLongDirEntryList *tmp;

	if ((tmp=malloc(sizeof(struct sLongDirEntryList)))==NULL) {
		perror(__func__);
		exit(1);
	}
	if ((tmp->lde=malloc(sizeof(struct sLongDirEntry)))==NULL) {
		perror(__func__);
		exit(1);
	}
	memcpy(tmp->lde, lde, DIR_ENTRY_SIZE);
	tmp->next = NULL;
	return tmp;
}

struct sLongDirEntryList *
	insertLongDirEntryList(struct sLongDirEntryList *new, struct sLongDirEntryList *list) {
/*
	inserts a long directory entry to list
*/
	assert(new != NULL);

	struct sLongDirEntryList *tmp;

	if (list != NULL) {
		tmp=list;
		while(tmp->next != NULL) {
			tmp=tmp->next;
		}
		tmp->next=new;
		return list;
	} else {
		return new;
	}

}

int32_t cmpEntries(struct sDirEntryList *de1, struct sDirEntryList *de2) {
/*
	compare two directory entries
*/

	assert(de1 != NULL);
	assert(de2 != NULL);

	char *s1,*s2;

	if ((de1->lname != NULL) && (de1->lname[0] != '\0')) {
		s1=de1->lname;
	} else {
		s1=de1->sname;
	}
	if ((de2->lname != NULL) && (de2->lname[0] != '\0')) {
		s2=de2->lname;
	} else {
		s2=de2->sname;
	}

	// directories will be put above normal files
	if (OPT_ORDER == 0) {
		if ((de1->sde->DIR_Atrr & ATTR_DIRECTORY) &&
		   !(de2->sde->DIR_Atrr & ATTR_DIRECTORY)) {
			return -1;
		} else if (!(de1->sde->DIR_Atrr & ATTR_DIRECTORY) &&
			    (de2->sde->DIR_Atrr & ATTR_DIRECTORY)) {
			return 1;
		}
	} else if (OPT_ORDER == 1) {
		if ((de1->sde->DIR_Atrr & ATTR_DIRECTORY) &&
		   !(de2->sde->DIR_Atrr & ATTR_DIRECTORY)) {
			return 1;
		} else if (!(de1->sde->DIR_Atrr & ATTR_DIRECTORY) &&
			    (de2->sde->DIR_Atrr & ATTR_DIRECTORY)) {
			return -1;
		}
	}

	if (OPT_IGNORE_CASE) {
		return strcasecmp(s1, s2) * OPT_REVERSE;
	} else {
		return strcmp(s1, s2) * OPT_REVERSE;
	}
}

void insertDirEntryList(struct sDirEntryList *new, struct sDirEntryList *list) {
/*
	inserts a directory entry into list
*/

	assert(new != NULL);
	assert(list != NULL);

	struct sDirEntryList *tmp, *dummy;

	tmp=list;
	while ((tmp->next != NULL) &&
	       (cmpEntries(new, tmp->next) >= 0)) {
		tmp=tmp->next;
	}
	dummy=tmp->next;
	tmp->next=new;
	new->next=dummy;
}

// ----------

int32_t check_bootsector(struct sBootSector *bs) {
/*
	lazy check if this is really a FAT bootsector
*/

	assert(bs != NULL);

	if (!((bs->BS_JmpBoot[0] == 0xeb) &&
	     (bs->BS_JmpBoot[2] == 0x90))
	  && !(bs->BS_JmpBoot[0] == 0xe9)) {
		// boot sector does not begin with specific instruction
 		myerror("Boot sector does not begin with jump instruction!");
		return -1;
	} else if (SwapInt16(bs->BS_EndOfBS) != 0xaa55) {
		// end of bootsector marker is missing
		myerror("End of boot sector marker is missing!");
		return -1;
	} else if (SwapInt16(bs->BS_BytesPerSec) == 0) {
		myerror("Sectors have a size of zero! Corrupt boot sector!");
		return -1;
	} else if (SwapInt16(bs->BS_SecPerClus) == 0) {
		myerror("Clusters have a size of zero! Corrupt boot sector!");
		return -1;
	}

	return 0;
}

int32_t read_bootsector(FILE *fd, struct sBootSector *bs) {
/*
	reads bootsector
*/

	assert(fd != NULL);
	assert(bs != NULL);

	if (fread(bs, sizeof(struct sBootSector), 1, fd) < 1) {
		if (feof(fd)) {
			myerror("Boot sector is too short!");
		} else {
			myerror("Failed to read from file!");
		}
		return -1;
	}


	if (check_bootsector(bs)) {
		myerror("This is not a FAT bootsector!");
		return -1;
	}

	

	return 0;
}

int32_t getCountOfClusters(struct sBootSector *bs) {
/*
	calculates count of clusters
*/

	assert(bs != NULL);

	u_int32_t RootDirSectors, FATSz, TotSec, DataSec;
	int32_t retvalue;

	RootDirSectors = ((SwapInt16(bs->BS_RootEntCnt) * 32) + (SwapInt16(bs->BS_BytesPerSec) - 1));
	RootDirSectors = RootDirSectors / SwapInt16(bs->BS_BytesPerSec);

	if (bs->BS_FATSz16 != 0) {
		FATSz = SwapInt16(bs->BS_FATSz16);
	} else {
		FATSz = SwapInt32(bs->FATxx.FAT32.BS_FATSz32);
	}
	if (SwapInt16(bs->BS_TotSec16) != 0) {
		TotSec = SwapInt16(bs->BS_TotSec16);
	} else {
		TotSec = SwapInt32(bs->BS_TotSec32);
	}
	DataSec = TotSec - (SwapInt16(bs->BS_RsvdSecCnt) + (bs->BS_NumFATs * FATSz) + RootDirSectors);

	retvalue = DataSec / bs->BS_SecPerClus;
	if (retvalue <= 0) {
		return -1;
	}
	return retvalue;
}

int32_t getFATType(struct sBootSector *bs) {
/*
	retrieves FAT type
*/

	assert(bs != NULL);

	u_int32_t CountOfClusters;

	CountOfClusters=getCountOfClusters(bs);
	if (CountOfClusters == -1) {
		return -1;
	} else if (CountOfClusters < 4096) { // FAT12!
		return 12;
	} else if (CountOfClusters < 65525) { // FAT16!
		return 16;
	} else { // FAT32!
		return 32;
	}
}

int32_t getFATEntry(FILE *fd, struct sBootSector *bs, u_int32_t cluster, u_int32_t *data) {
/*
	retrieves FAT entry for a cluster number
*/

	assert(fd != NULL);
	assert(bs != NULL);
	assert(data != NULL);

	off_t FATOffset, FATSz, BSOffset;
	int32_t FATType;

	*data=0;

	if (bs->BS_FATSz16 != 0) {
		FATSz = SwapInt16(bs->BS_FATSz16);
	} else {
		FATSz = SwapInt32(bs->FATxx.FAT32.BS_FATSz32);
	}

	FATType = getFATType(bs);

	if (FATType == 16) {
		FATOffset = (off_t)cluster * 2;
		BSOffset = (off_t)SwapInt16(bs->BS_RsvdSecCnt) * SwapInt16(bs->BS_BytesPerSec) + FATOffset;
		if (fseeko(fd, BSOffset, SEEK_SET) == -1) {
			myerror("Seek error!");
			return -1;
		}
		if (fread(data, 2, 1, fd)<1) {
			myerror("Failed to read from file!");
			return -1;
		}
		*data=SwapInt32(*data);
	} else if (FATType == 32) {
		FATOffset = (off_t)cluster * 4;
		BSOffset = (off_t)SwapInt16(bs->BS_RsvdSecCnt) * SwapInt16(bs->BS_BytesPerSec) + FATOffset;
		if (fseeko(fd, BSOffset, SEEK_SET) == -1) {
			myerror("Seek error!");
			return -1;
		}
		if (fread(data, 4, 1, fd) < 1) {
			myerror("Failed to read from file!");
			return -1;
		}
		*data=SwapInt32(*data);
		*data = *data & 0x0fffffff;
	} else if (FATType == 12) {
		myerror("FAT12 is not supported!");
		return -1;
	} else {
		myerror("Failed to get FAT type!");
		return -1;
	}

	return 0;

}

int32_t putFATEntry(FILE *fd, struct sBootSector *bs, u_int32_t cluster, u_int32_t data) {
/*
	write a FAT entry
*/

	assert(fd != NULL);
	assert(bs != NULL);

	off_t FATOffset, FATSz, BSOffset;
	u_int32_t value, i;
	int32_t FATType;

	if (bs->BS_FATSz16 != 0) {
		FATSz = SwapInt16(bs->BS_FATSz16);
	} else {
		FATSz = SwapInt32(bs->FATxx.FAT32.BS_FATSz32);
	}

	FATType = getFATType(bs);

	if (FATType == 16) {
		FATOffset = (off_t)cluster * 2;
		BSOffset = (off_t)SwapInt16(bs->BS_RsvdSecCnt) * SwapInt16(bs->BS_BytesPerSec) + FATOffset;
		fseeko(fd, BSOffset, SEEK_SET);
		data=SwapInt32(data);
		if (fwrite(&data, 2, 1, fd)<1) {
			myerror("Failed to write to file!");
			return -1;
		}
	} else if (FATType == 32) {
		FATOffset = (off_t)cluster * 4;
		BSOffset = (off_t)SwapInt16(bs->BS_RsvdSecCnt) * SwapInt16(bs->BS_BytesPerSec) + FATOffset;
		if (getFATEntry(fd, bs, cluster, &value)==-1) {
			myerror("Failed to get FAT entry!");
			return -1;
		}
		value = (value & 0xf0000000) | (data & 0x0fffffff);
		for(i=0; i<bs->BS_NumFATs; i++) {
			fseeko(fd, BSOffset+i*FATSz*bs->BS_BytesPerSec, SEEK_SET);
			value=SwapInt32(value);
			if (fwrite(&value, 4, 1, fd) < 1) {
				myerror("Failed to write to file!");
				return -1;
			}
		}
	} else if (FATType == 12) {
		myerror("FAT12 is not supported!");
		return -1;
	} else {
		myerror("Failed to get FAT type!");
		return -1;
	}

	return 0;

}

void parseLongFilenamePart(struct sLongDirEntry *lde, char *str) {
/*
	retrieves a part of a long filename from a
	directory entry
	(thanks to M$ for this ugly hack...)
*/

	assert(lde != NULL);
	assert(str != NULL);

	u_int32_t len=0;//strlen(str);
	str[len]=(char) (*(&lde->LDIR_Ord+1));
	str[len+1]=(char) (*(&lde->LDIR_Ord+3));
	str[len+2]=(char) (*(&lde->LDIR_Ord+5));
	str[len+3]=(char) (*(&lde->LDIR_Ord+7));
	str[len+4]=(char) (*(&lde->LDIR_Ord+9));
	str[len+5]=(char) (*(&lde->LDIR_Ord+14));
	str[len+6]=(char) (*(&lde->LDIR_Ord+16));
	str[len+7]=(char) (*(&lde->LDIR_Ord+18));
	str[len+8]=(char) (*(&lde->LDIR_Ord+20));
	str[len+9]=(char) (*(&lde->LDIR_Ord+22));
	str[len+10]=(char) (*(&lde->LDIR_Ord+24));
	str[len+11]=(char) (*(&lde->LDIR_Ord+28));
	str[len+12]=(char) (*(&lde->LDIR_Ord+30));
	str[len+13]=0;
}

void parseShortFilename(struct sShortDirEntry *sde, char *str) {
/*
	parses the short name of a file
*/

	assert(sde != NULL);
	assert(str != NULL);

	char *s;
	strncpy(str, sde->DIR_Name, 8);
	str[8]='\0';
	s=strchr(str, ' ');
	if (s!=NULL) s[0]='\0';
	if ((char)(*(sde->DIR_Name+8)) != ' ') {
		strcat(str, ".");
		strncat(str, sde->DIR_Name+8, 3);
		str[12]='\0';
	}
}

int32_t parseEntry(FILE *fd, union sDirEntry *de) {
/*
	parses one directory entry
*/

	assert(fd != NULL);
	assert(de != NULL);

	if ((fread(de, DIR_ENTRY_SIZE, 1, fd)<1)) {
		myerror("Failed to read from file!");
		return -1;
	}

	if (de->LongDirEntry.LDIR_Attr == 0) return 0; // no more entries

	// long dir entry
	if ((de->LongDirEntry.LDIR_Attr & ATTR_LONG_NAME_MASK) == ATTR_LONG_NAME) return 2;

	return 1; // short dir entry
}

off_t getClusterOffset(struct sBootSector *bs, u_int32_t cluster) {
/*
	returns the offset of a specific cluster in the
	data region of the file system
*/

	assert(bs != NULL);
	assert(cluster > 1);

	u_int32_t FATSz, RootDirSectors, FirstDataSector;

	if (bs->BS_FATSz16 != 0) {
		FATSz = SwapInt16(bs->BS_FATSz16);
	} else {
		FATSz = SwapInt32(bs->FATxx.FAT32.BS_FATSz32);
	}

	RootDirSectors = ((SwapInt16(bs->BS_RootEntCnt) * DIR_ENTRY_SIZE) + (SwapInt16(bs->BS_BytesPerSec) - 1)) / SwapInt16(bs->BS_BytesPerSec);
	FirstDataSector = (SwapInt16(bs->BS_RsvdSecCnt) + (bs->BS_NumFATs * FATSz) + RootDirSectors);

	return (((off_t)(cluster - 2) * bs->BS_SecPerClus) + FirstDataSector) * SwapInt16(bs->BS_BytesPerSec);

}

int32_t parseClusterChain(FILE *fd, struct sBootSector *bs, struct sClusterChain *chain, struct sDirEntryList *list) {
/*
	parses a cluster chain and puts directory entries to list
*/

	assert(fd != NULL);
	assert(bs != NULL);
	assert(chain != NULL);
	assert(list != NULL);

	int32_t j, ret;
	u_int32_t maxEntries, entries=0;
	union sDirEntry de;
	struct sLongDirEntryList *llist;
	char tmp[MAX_PATH_LEN+1], dummy[MAX_PATH_LEN+1], sname[MAX_PATH_LEN+1], lname[MAX_PATH_LEN+1];

	maxEntries = bs->BS_SecPerClus * SwapInt16(bs->BS_BytesPerSec) / DIR_ENTRY_SIZE;

	chain=chain->next;	// head element

	llist = NULL;
	lname[0]='\0';
	while (chain != NULL) {
		fseeko(fd, getClusterOffset(bs, chain->cluster), SEEK_SET);
		for (j=1;j<=maxEntries;j++) {
			entries++;
			ret=parseEntry(fd, &de);
			if (ret == -1) {
				myerror("Failed to parse directory entry!");
				return -1;
			} else if (ret == 0) {
				break;
			} else if (ret == 2) {
				parseLongFilenamePart(&de.LongDirEntry, tmp);
				
				// insert long dir entry in list
				llist=insertLongDirEntryList(newLongDirEntry(&de.LongDirEntry), llist);
				strncpy(dummy, tmp, MAX_PATH_LEN);
				dummy[MAX_PATH_LEN]='\0';
				strncat(dummy, lname, MAX_PATH_LEN - strlen(dummy));
				dummy[MAX_PATH_LEN]='\0';
				strncpy(lname, dummy, MAX_PATH_LEN);
				dummy[MAX_PATH_LEN]='\0';
			} else {
				parseShortFilename(&de.ShortDirEntry, sname);
				if (OPT_LIST && strcmp(sname, ".") &&
				   strcmp(sname, "..") &&
				   (sname[0] != DE_FREE) &&
				  !(de.ShortDirEntry.DIR_Atrr & ATTR_VOLUME_ID)) {
					printf("%s\n", (lname[0] != '\0') ? lname : sname);
				}
				insertDirEntryList(newDirEntry(sname, lname, &de.ShortDirEntry, llist, entries), list);
				entries=0;
				llist = NULL;
				lname[0]='\0';
			}
		}
		chain=chain->next;
	}

	return 0;
}

int32_t parseDirEntry(FILE *fd, struct sDirEntryList *list, u_int32_t *entries) {
/*
	parses an entry of a directory structure
*/

	assert(fd != NULL);
	assert(list != NULL);

	char sname[MAX_PATH_LEN+1], lname[MAX_PATH_LEN+1], tmp[MAX_PATH_LEN+1], dummy[MAX_PATH_LEN+1];
	union sDirEntry de;
	u_int32_t count=0;

	struct sLongDirEntryList *llist = NULL;

	// read a directory entry from file
	if ((fread(&de, DIR_ENTRY_SIZE, 1, fd)<1)) {
		myerror("Failed to read from file!");
		return -1;
	}

	// if directory entry is empty, return
	if (de.LongDirEntry.LDIR_Ord == 0) {
		return 1;
	}

	lname[0]='\0';
	*entries=0;
	while (((de.LongDirEntry.LDIR_Attr & ATTR_LONG_NAME_MASK) == ATTR_LONG_NAME) ){

		parseLongFilenamePart(&de.LongDirEntry, tmp);

		// insert long dir entry in list
		llist=insertLongDirEntryList(newLongDirEntry(&de.LongDirEntry), llist);

		count++;
		(*entries)++;
		strncpy(dummy, tmp, MAX_PATH_LEN);
		dummy[MAX_PATH_LEN]='\0';
		strncat(dummy, lname, MAX_PATH_LEN - strlen(dummy));
		dummy[MAX_PATH_LEN]='\0';
		strncpy(lname, dummy, MAX_PATH_LEN);
		dummy[MAX_PATH_LEN]='\0';
		if (fread(&de, DIR_ENTRY_SIZE, 1, fd)<1) {
			myerror("Failed to read from file!");
			return -1;
		}
	}
	
	/* short-filename directory entry appears after long-filename directory entries
	   well, i should check such things in some future version */
	parseShortFilename(&de.ShortDirEntry, sname);
	(*entries)++;

	insertDirEntryList(newDirEntry(sname, lname, &de.ShortDirEntry, llist, *entries), list);

	if (OPT_LIST && strcmp(sname, ".") && strcmp (sname, "..") && (sname[0] != DE_FREE) && !(de.ShortDirEntry.DIR_Atrr & ATTR_VOLUME_ID)) {
		printf("%s\n", (lname[0] != '\0') ? lname : sname);
	}


	return 0;

}

int32_t writeList(FILE *fd, struct sDirEntryList *list) {
/*
	writes directory entries to file
*/

	assert(fd != NULL);
	assert(list != NULL);

	struct sLongDirEntryList *tmp;

	while(list->next!=NULL) {
		tmp=list->next->ldel;
		while(tmp != NULL) {
			if (fwrite(tmp->lde, DIR_ENTRY_SIZE, 1, fd)<1) {
				perror(__func__);
				return -1;
			}
			tmp=tmp->next;
		}
		if (fwrite(list->next->sde, DIR_ENTRY_SIZE, 1, fd)<1) {
			perror(__func__);
			return -1;
		}
		list=list->next;
	}

	return 0;
}

int32_t insertCluster(struct sClusterChain *chain, u_int32_t cluster) {
/*
	allocate memroy and insert cluster into cluster chain
*/
	assert(chain != NULL);

	while (chain->next != NULL) chain=chain->next;

	if  ((chain->next = malloc(sizeof(struct sClusterChain))) == NULL) {
		myerror("Failed to allocate memory!");
		return -1;
	}
	chain->next->cluster=cluster;
	chain->next->next=NULL;
	
	return 0;
}

void freeClusterChain(struct sClusterChain *chain) {
/*
	free cluster chain
*/

	assert(chain != NULL);

	struct sClusterChain *tmp, *next;
	
	for (tmp = chain->next; tmp; tmp = next) {
		next = tmp -> next;
		free (tmp);
	}
}

int32_t getClusterChain(FILE *fd, struct sBootSector *bs, u_int32_t startCluster, struct sClusterChain *chain) {
/*
	retrieves an array of all clusters in a cluster chain
	starting with startCluster
*/

	assert(fd != NULL);
	assert(bs != NULL);
	assert(chain != NULL);

	u_int32_t cluster;
	u_int32_t data,i=0;
	int32_t FATType;

	cluster=startCluster;

	FATType=getFATType(bs);

	if (FATType == 12) {
		myerror("FAT12 is not supported!");
		return -1;
	} else if (FATType == 16) {
		do {
			if (i == MAX_CHAIN_LENGTH) {
				myerror("Cluster chain is too long!");
				return -1;
			}
			if (insertCluster(chain, cluster) == -1) {
				myerror("Failed to insert cluster!");
				return -1;
			}
			i++;
			if (getFATEntry(fd, bs, cluster, &data)) {
				myerror("Failed to get FAT entry!");
				return -1;
			}
			cluster=data;
		} while (cluster < 0xfff8);	// end of cluster
	} else if (FATType == 32){
		do {
			if (i == MAX_CHAIN_LENGTH) {
				myerror("Cluster chain is too long!");
				return -1;
			}
			if (insertCluster(chain, cluster) == -1) {
				myerror("Failed to insert cluster!");
				return -1;
			}
			i++;
			if (getFATEntry(fd, bs, cluster, &data)) {
				myerror("Failed to get FAT entry");
				return -1;
			}
			cluster=data;
		} while (((cluster & 0x0fffffff) != 0x0ff8fff8) &&
			 ((cluster & 0x0fffffff) < 0x0ffffff8));	// end of cluster
	} else {
		myerror("Failed to get FAT type!");
		return -1;
	}

	return i;
}

int32_t writeClusterChain(FILE *fd, struct sBootSector *bs, struct sDirEntryList *list, struct sClusterChain *chain) {
/*
	writes all entries from list to the cluster chain
*/

	assert(fd != NULL);
	assert(bs != NULL);
	assert(list != NULL);
	assert(chain != NULL);

	int32_t i=0, entries=0;
	u_int32_t MaxEntries;
	struct sLongDirEntryList *tmp;
	struct sDirEntryList *p=list->next;
	char empty[DIR_ENTRY_SIZE]={0};

	chain=chain->next;	// we don't need to look at the head element

	MaxEntries = bs->BS_SecPerClus * SwapInt16(bs->BS_BytesPerSec) / DIR_ENTRY_SIZE;
	if (fseeko(fd, getClusterOffset(bs, chain->cluster), SEEK_SET)==-1) {
		myerror("Seek error!");
		return -1;
	}

	while(p != NULL) {
		if (entries+p->entries <= MaxEntries) {
			tmp=p->ldel;
			for (i=1;i<p->entries;i++) {
				if (fwrite(tmp->lde, DIR_ENTRY_SIZE, 1, fd)<1) {
					perror(__func__);
					return -1;
				}
				tmp=tmp->next;
			}
			if (fwrite(p->sde, DIR_ENTRY_SIZE, 1, fd)<1) {
				perror(__func__);
				return -1;
			}
			entries+=p->entries;
		} else {
			tmp=p->ldel;
			for (i=1;i<=MaxEntries-entries;i++) {
				if (fwrite(tmp->lde, DIR_ENTRY_SIZE, 1, fd)<1) {
					perror(__func__);
					return -1;
				}
				tmp=tmp->next;
			}
			chain=chain->next; entries=p->entries - (MaxEntries - entries);	// next cluster
			if (fseeko(fd, getClusterOffset(bs, chain->cluster), SEEK_SET)==-1) {
				myerror("Seek error!");
				return -1;
			}
			while(tmp!=NULL) {
				if (fwrite(tmp->lde, DIR_ENTRY_SIZE, 1, fd)<1) {
					perror(__func__);
					return -1;
				}
				tmp=tmp->next;
			}
			if (fwrite(p->sde, DIR_ENTRY_SIZE, 1, fd)<1) {
				perror(__func__);
				return -1;
			}
		}
		p=p->next;

	}
	if (entries < MaxEntries) {
		if (fwrite(empty, DIR_ENTRY_SIZE, 1, fd)<1) {
			perror(__func__);
			return -1;
		}
	}

	return 0;

}

int32_t sortClusterChain(FILE *fd, struct sBootSector *bs, u_int32_t cluster, char *path) {
/*
	sorts the directory entries in a cluster
*/

	assert(fd != NULL);
	assert(bs != NULL);
	assert(path != NULL);

	u_int32_t clen, value, c;
	struct sClusterChain ClusterChain=__INITCLUSTERCHAIN__;
	char newpath[MAX_PATH_LEN+1]={0};

	struct sDirEntryList list = __INITDIRLIST__;
	struct sDirEntryList *p;

	if ((clen=getClusterChain(fd, bs, cluster, &ClusterChain)) == -1 ) {
		myerror("Failed to get cluster chain!");
		return -1;
	}

	// DEBUG
/*	struct sClusterChain *ch=&ClusterChain;
	while(ch != NULL) {
		printf("%u-", ch->cluster);
		ch=ch->next;
	}
*/
	if (!OPT_LIST) {
		if (parseClusterChain(fd, bs, &ClusterChain, &list) == -1) {
			myerror("Failed to parse cluster chain!");
			return -1;
		}
		infomsg("Sorting directory %s\n", path);
		if (writeClusterChain(fd, bs, &list, &ClusterChain) == -1) {
			myerror("Failed to write cluster chain!");
			return -1;
		}
	} else {
		printf("%s\n", path);
		if (parseClusterChain(fd, bs, &ClusterChain, &list) == -1) {
			myerror("Failed to parse cluster chain!");
			return -1;
		}
		printf("\n");
	}

	freeClusterChain(&ClusterChain);

	// sort sub directories
	p=list.next;
	while (p != NULL) {
		if ((p->sde->DIR_Atrr & ATTR_DIRECTORY) &&
			(p->sde->DIR_Name[0] != DE_FREE) &&
			!(p->sde->DIR_Atrr & ATTR_VOLUME_ID) &&
			(strcmp(p->sname, ".")) && strcmp(p->sname, "..")) {

			c=(SwapInt16(p->sde->DIR_FstClusHI) * 65536 + SwapInt16(p->sde->DIR_FstClusLO));
			if (getFATEntry(fd, bs, c, &value) == -1) {
				myerror("Failed to get FAT entry!");
				return -1;
			}

			strncpy(newpath, path, MAX_PATH_LEN - strlen(newpath));
			newpath[MAX_PATH_LEN]='\0';
			if ((p->lname != NULL) && (p->lname[0] != '\0')) {
				strncat(newpath, p->lname, MAX_PATH_LEN - strlen(newpath));
				newpath[MAX_PATH_LEN]='\0';
				strncat(newpath, "/", MAX_PATH_LEN - strlen(newpath));
				newpath[MAX_PATH_LEN]='\0';
			} else {
				strncat(newpath, p->sname, MAX_PATH_LEN - strlen(newpath));
				newpath[MAX_PATH_LEN]='\0';
				strncat(newpath, "/", MAX_PATH_LEN - strlen(newpath));
				newpath[MAX_PATH_LEN]='\0';
			}

			if (sortClusterChain(fd, bs, c, newpath) == -1) {
				myerror("Failed to sort cluster chain!");
				return -1;
			}

		}
		p=p->next;
	}

	return 0;
}

int32_t sort_FAT16_rootdir(FILE *fd, struct sBootSector *bs) {
/*
	sorts the root directory of a FAT16 file system
*/

	assert(fd != NULL);
	assert(bs != NULL);

	off_t BSOffset;
	u_int32_t FATSz, i, ret;
	u_int32_t entries,count=0, c, value;
	char newpath[MAX_PATH_LEN+1]={0};
	char empty[32]={0};

	struct sDirEntryList list = __INITDIRLIST__;
	struct sDirEntryList *p;

	if (bs->BS_FATSz16 != 0) {
		FATSz = SwapInt16(bs->BS_FATSz16);
	} else {
		FATSz = SwapInt32(bs->FATxx.FAT32.BS_FATSz32);
	}

	BSOffset = ((off_t)SwapInt16(bs->BS_RsvdSecCnt) + bs->BS_NumFATs * FATSz)* SwapInt16(bs->BS_BytesPerSec);

	fseeko(fd, BSOffset, SEEK_SET);

	if (OPT_LIST) {
		printf("/\n");
	}

	// read all entries and parse it to the list
	for (i=1;i<=SwapInt16(bs->BS_RootEntCnt);i++) {
		ret=parseDirEntry(fd, &list, &entries);
		count+=entries;
		if (ret == 1) {
			break;
		} else if (ret == -1) {
			myerror("Failed to parse directory entries!");
			return -1;
		}
		// well, one entry read, but not matched
		if (entries == 0) count++;
	}

	if (!OPT_LIST) {
		infomsg("Sorting directory /\n");

		fseeko(fd, BSOffset, SEEK_SET);

		// write the sorted entries back to the fs
		if (writeList(fd, &list) == -1) {
			myerror("Failed to write FAT entries!");
			return -1;
		}

		// wipe all follwing entries
		if (SwapInt16(bs->BS_RootEntCnt) - count > 0) {
			if (fwrite(&empty, DIR_ENTRY_SIZE, 1, fd)<1) {
				perror(__func__);
				return -1;
			}
		}
	} else {
		printf("\n");
	}

	// sort sub directories
	p=list.next;
	while (p != NULL) {
		if ((p->sde->DIR_Atrr & ATTR_DIRECTORY) &&
			(p->sde->DIR_Name[0] != DE_FREE) &&
			!(p->sde->DIR_Atrr & ATTR_VOLUME_ID) &&
			(strcmp(p->sname, ".")) && strcmp(p->sname, "..")) {
			c= (SwapInt16(p->sde->DIR_FstClusHI) * 65536 + SwapInt16(p->sde->DIR_FstClusLO));
			if (getFATEntry(fd, bs, c, &value) == -1) {
				myerror("Failed to get FAT entry!");
				return -1;
			}

			strncpy(newpath, "/", MAX_PATH_LEN - strlen(newpath));
			newpath[MAX_PATH_LEN]='\0';
			if ((p->lname != NULL) && (p->lname[0] != '\0')) {
				strncat(newpath, p->lname, MAX_PATH_LEN - strlen(newpath));
				newpath[MAX_PATH_LEN]='\0';
				strncat(newpath, "/", MAX_PATH_LEN - strlen(newpath));
				newpath[MAX_PATH_LEN]='\0';
			} else {
				strncat(newpath, p->sname, MAX_PATH_LEN - strlen(newpath));
				newpath[MAX_PATH_LEN]='\0';
				strncat(newpath, "/", MAX_PATH_LEN - strlen(newpath));
				newpath[MAX_PATH_LEN]='\0';
			}

			if (sortClusterChain(fd, bs, c, newpath) == -1) {
				myerror("Failed to sort cluster chain!");
				return -1;
			}
		}
		p=p->next;
	}

	return 0;
}

int32_t printFSInfo(char *filename) {
/*
	print file system information
*/

	assert(filename != NULL);

	u_int32_t FATSz, value;
	int32_t FATType, cluster;
	FILE *fd;
	struct sBootSector bs;

	printf("\t- File system information -\n");

	if ((fd=fopen(filename, "r")) == NULL) {
		perror(__func__);
		return -1;
	}

	// read boot sector
	if (read_bootsector(fd, &bs)) {
		myerror("Failed to read boot sector!");
		return -1;
	}

	if (bs.BS_FATSz16 != 0) {
		FATSz = SwapInt16(bs.BS_FATSz16);
	} else {
		FATSz = SwapInt32(bs.FATxx.FAT32.BS_FATSz32);
	}

	FATType = getFATType(&bs);
	if (FATType == -1) {
		myerror("Failed to get FAT type!");
		return -1;
	}

	cluster=getCountOfClusters(&bs);
	if (cluster == -1) {
		myerror("Failed to get count of cluster!");
		return -1;
	}

	printf("Device:\t\t\t\t%s\n", filename);
	fflush(stdout);
	printf("Type:\t\t\t\tFAT%u\n", getFATType(&bs));
	fflush(stdout);
	printf("Sector size:\t\t\t%u bytes\n", SwapInt16(bs.BS_BytesPerSec));
	fflush(stdout);
	printf("FAT size:\t\t\t%u sectors (%u bytes)\n", FATSz, FATSz * SwapInt16(bs.BS_BytesPerSec));
	printf("Cluster size:\t\t\t%u bytes\n", bs.BS_SecPerClus * SwapInt16(bs.BS_BytesPerSec));
	printf("Cluster count:\t\t\t%u\n", cluster);
	printf("FS size:\t\t\t%.2f MiBytes\n", (float)cluster * bs.BS_SecPerClus * SwapInt16(bs.BS_BytesPerSec) / (1024.0*1024));
	if (FATType == 32) {
		if (getFATEntry(fd, &bs, SwapInt32(bs.FATxx.FAT32.BS_RootClus), &value) == -1) {
			myerror("Failed to get FAT enry!");
			return -1;
		}
		printf("FAT32 root directory first cluster: 0x%x, Data offset: 0x%llx, FAT entry: 0x%x\n",
			SwapInt32(bs.FATxx.FAT32.BS_RootClus),
			(unsigned long long)getClusterOffset(&bs, SwapInt32(bs.FATxx.FAT32.BS_RootClus)), value);
	}

	fclose(fd);

	return 0;

}

u_int32_t check_mounted(char *filename) {

#if defined(__LINUX__)
	FILE *fd;
	struct mntent *mnt;
	u_int32_t ret = 0;
	
	if ((fd = setmntent("/etc/mtab", "r")) == NULL) {
		myerror((const char *) strerror(errno));
		return 0;
	}
	
	while ((mnt = getmntent(fd)) != NULL) {
		if (strcmp(mnt->mnt_fsname, filename) == 0) {
			ret = 1;
			break;
		}
	}
	
	if (endmntent(fd) != 1) {
		myerror("Closing mtab failed!");
		return 0;
	}

	return ret;
#elif defined(__BSD__)
	struct statfs *mntbuf;
	int i, mntsize;
	u_int32_t ret = 0;

	mntsize = getmntinfo(&mntbuf, MNT_NOWAIT);

	if (mntsize == 0) {
		myerror(strerror (errno));
		return 0;
	}
	
	for (i = mntsize - 1; i >= 0; i--) {
		if (strcmp(mntbuf[i].f_mntfromname, filename) == 0) {
			ret = 1;
			break;
		}
	}
	
	return ret;
#else
	return 0;
#endif
}


int32_t sort_fs(char *filename) {
/*
	sort FAT file system
*/

	assert(filename != NULL);

	FILE *fd;
	struct sBootSector bs;

	int32_t FATType;

	if (check_mounted (filename) && !OPT_LIST) {
		myerror("Filesystem is mounted!");
		return -1;
	}

	if ((fd=fopen(filename, (OPT_LIST) ? "r" : "r+")) == NULL) {
		perror(__func__);
		return -1;
	}

	// read boot sector
	if (read_bootsector(fd, &bs)) {
		myerror("Failed to read boot sector!");
		return -1;
	}

	FATType = getFATType(&bs);

	if (FATType == 12) {
		// FAT12
		// sorry, too complicated ;)
		myerror("FAT12 is not supported!");
		return -1;
	} else if (FATType == 16) {
		// FAT16
		// root directory has fixed size and position
		infomsg("File system: FAT16.\n\n");
		if (sort_FAT16_rootdir(fd, &bs) == -1) {
			myerror("Failed to sort FAT16 root directory!");
			return -1;
		}
	} else if (FATType == 32) {
		// FAT32
		// root directory lies in cluster chain,
		// so sort it like all other directories
		infomsg("File system: FAT32.\n\n");
		if (sortClusterChain(fd, &bs, SwapInt32(bs.FATxx.FAT32.BS_RootClus), "/") == -1) {
			myerror("Failed to sort cluster chain!");
			return -1;
		}
	} else {
		myerror("Failed to get FAT type!");
		return -1;
	}

	fclose(fd);
	return 0;
}

int main(int argc, char *argv[]) {
/*
	parse arguments and options and start sorting
*/

	int8_t c;

	/*
		DEBUG:
	*/
/*	printf("SizeofBS: %u\n", sizeof(struct sBootSector));
	printf("SizeofFAT12_16: %u\n", sizeof(struct sFAT12_16));
	printf("SizeofFAT32: %u\n", sizeof(struct sFAT32));
	return 0;*/

	/* Default (1) is normal order, use -1 for reverse order. */
	
	OPT_REVERSE = 1;
	
	opterr=0;
	while ((c=getopt(argc, argv, "ivhqco:lr")) != -1) {
		switch(c) {
			case 'i' : OPT_INFO = 1; break;
			case 'v' : OPT_VERSION = 1; break;
			case 'h' : OPT_HELP = 1; break;
			case 'q' : OPT_QUIET = 1; break;
			case 'c' : OPT_IGNORE_CASE = 1; break;
			case 'l' : OPT_LIST = 1; break;
			case 'r' : OPT_REVERSE = -1; break;
			case 'o' :
				switch(optarg[0]) {
					case 'd': OPT_ORDER=0; break;
					case 'f': OPT_ORDER=1; break;
					case 'a': OPT_ORDER=2; break;
					default:
						fprintf(stderr, "Unknown flag '%c' for option 'o'.\n", optarg[0]);
						fprintf(stderr, "Use -h for more help.\n");
						return -1;
				}
				break;
			default :
				fprintf(stderr, "Unknown option '%c'.\n", optopt);
				fprintf(stderr, "Use -h for more help.\n");
				return -1;
		}
	}

	if (OPT_HELP) {
		printf(INFO_HEADER INFO_USAGE);
		return 0;
	} else if (OPT_VERSION) {
		printf(INFO_HEADER "\n" );
		return 0;
	} else if (optind < argc -1) {
		fprintf(stderr, "Too many arguments!\n");
		fprintf(stderr, "Use -h for more help.\n");
		return -1;
	} else if (optind == argc) {
		fprintf(stderr, "Device must be given!\n");
		fprintf(stderr, "Use -h for more help.\n");
		return -1;
	}

	if (OPT_INFO) {
		infomsg(INFO_HEADER "\n\n");
		if (printFSInfo(argv[optind]) == -1) {
			myerror("Failed to print file system information");
			return -1;
		}
	} else {
		infomsg(INFO_HEADER "\n\n");
		if (sort_fs(argv[optind]) == -1) {
			myerror("Failed to sort file system!");
			return -1;
		}
	}

	return 0;
}
