#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/kd.h>

#define MAXFONTSIZE 512

///////////////////////////////////////////////////////////////////////////////

/* From Yann Dirson's <ydirson@a2points.com> console-tools */

int get_kernel_font (int fd, struct consolefontdesc *cfd)
{
	if (!cfd)
	{
		errno = EINVAL;
		return -1;
	}
  
	cfd->charheight = 0;

#ifndef GIO_FONTX
	cfd->chardata = (char *) malloc (32 * 256);
#else
	cfd->chardata = (char *) malloc (32 * MAXFONTSIZE);
	cfd->charcount = MAXFONTSIZE;
	if (ioctl (fd, GIO_FONTX, cfd))
#endif
	{
		cfd->charcount = 256;
		if (ioctl (fd, GIO_FONT, cfd->chardata))
			return -1;
	}

	/* If we had to rely on GIO_FONT, find the most efficient char-height
	 * from the data we got.
	 * FIXME: should be set to console cell-height ?
	 */
	if (cfd->charheight == 0)
		for (cfd->charheight = 32; cfd->charheight > 0; cfd->charheight--)
			for (unsigned i = 0; i < cfd->charcount; i++)
				if (cfd->chardata[32 * i + cfd->charheight - 1])
					goto nonzero;
nonzero:
 
	return 0;
}

int set_kernel_font (int fd, struct consolefontdesc *cfd)
{
#if defined (PIO_FONTX) && !defined (sparc)
	if (ioctl (fd, PIO_FONTX, cfd))
#endif
		if (ioctl (fd, PIO_FONT, cfd->chardata))
			return -1;
  
	return 0;
}

int is_a_console (int fd) 
{
	char arg;
  
	arg = 0;
	return ioctl (fd, KDGKBTYPE, &arg) == 0 &&
		(arg == KB_101 || arg == KB_84);
}

int open_a_console (char fnam[])
{
	int fd;
  
	/* try read-write */
	fd = open (fnam, O_RDWR);
  
	/* if failed, try read-only */
	if (fd < 0 && errno == EACCES)
		fd = open (fnam, O_RDONLY);
  
	/* if failed, try write-only */
	if (fd < 0 && errno == EACCES)
		fd = open (fnam, O_WRONLY);
  
	/* if failed, fail */
	if (fd < 0)
		return -1;
  
	/* if not a console, fail */
	if (! is_a_console (fd))
	{
		close (fd);
		return -1;
	}
  
	/* success */
	return fd;
}

/*
 * Get an fd for use with kbd/console ioctls.
 * We try several things because opening /dev/console will fail
 * if someone else used X (which does a chown on /dev/console).
 *
 * if tty_name is non-NULL, try this one instead.
 */

int get_console_fd (char tty_name[])
{
	int fd;

	if (tty_name)
	{
		if ((fd = open_a_console (tty_name)) == -1) return -1;
		return fd;
	}
    
	fd = open_a_console ("/dev/tty");
	if (fd >= 0) return fd;
  
	fd = open_a_console ("/dev/console");
	if (fd >= 0) return fd;

	for (fd = 0; fd < 3; fd++)
		if (is_a_console (fd))
			return fd;
  
	cerr << "Couldn't get a file descriptor referring to the console\n";
	return -1;		/* total failure */
}

int set_kernel_unimap (int fd, struct unimapdesc *descr)
{
	struct unimapinit advice;

	advice.advised_hashsize = 0;
	advice.advised_hashstep = 0;
	advice.advised_hashlevel = 1;	       /* FIXME: 0 or 1 ? */

again:
	if (ioctl (fd, PIO_UNIMAPCLR, &advice)) 
		return -1;
  
	if (ioctl (fd, PIO_UNIMAP, descr))
	{
		if (errno == ENOMEM && advice.advised_hashlevel < 100) 
		{
			advice.advised_hashlevel++;
			goto again;
		}
		return -1;
	}
  
	return 0;
}

int get_kernel_unimap (int fd, struct unimapdesc *descr)
{
	descr->entry_ct = 0;
	descr->entries = NULL;
  
	/* get entry_ct */
	if (ioctl (fd, GIO_UNIMAP, (unsigned long) descr)) /* failed */
	{
		if (errno != ENOMEM || descr->entry_ct == 0)
			return -1;
      
		/* alloc place for entry_ct unipairs */
		descr->entries = (struct unipair *)
			malloc (descr->entry_ct * sizeof (struct unipair));
      
		/* actually get unipairs */
		if (ioctl (fd, GIO_UNIMAP, (unsigned long) descr))
			return -1;

		return 0;
	}
	else
	{
		/* no valid SFM currently loaded */
		errno = ENXIO;
		return -1;
	}
}

///////////////////////////////////////////////////////////////////////////////

#define MAX_DECOMPOSITION 16

struct numer
{
	unsigned long nr[MAX_DECOMPOSITION];
	int fragmentow;
	int b, ul;
	numer () : fragmentow (0), b (0), ul (0) {}
	numer (unsigned long nr1) : fragmentow (1), b (0), ul (0) {*nr = nr1;}
	int operator == (const numer &n1) const;
	int operator != (const numer &n1) const {return !(*this == n1);}
	int numer::hash () const;
};

int numer::operator == (const numer &n1) const
{
	if (fragmentow != n1.fragmentow || b != n1.b || ul != n1.ul) return 0;
	for (int i = 0; i < fragmentow; i++)
		if (nr[i] != n1.nr[i]) return 0;
	return 1;
}

int numer::hash () const
{
	int h = 0;
	for (int i = 0; i < fragmentow; i++)
		h += nr[i] & 0xFFFF;
	return h & 0xFFFF;
}

template <class T>
struct listaznakow
{
	numer Z;
	T d;
	listaznakow <T> *nast;
};

template <class T>
class tablicaznakow
{
private:
	listaznakow <T> *A[0x1000];
public:
	tablicaznakow ();
	T *utworzznak (const numer &Z);
	T *dajznak (const numer &Z) const;
};

template <class T>
tablicaznakow <T>::tablicaznakow ()
{
	for (int Z = 0; Z < 0x1000; Z++)
		A[Z] = NULL;
}

template <class T>
T *tablicaznakow <T>::utworzznak (const numer &Z)
{
	listaznakow <T> **B = &A[Z.hash () & 0xFFF];
	while (*B)
	{
		if ((*B)->Z == Z) return &(*B)->d;
		B = &(*B)->nast;
	}
	*B = new listaznakow <T>;
	(*B)->Z = Z;
	(*B)->nast = NULL;
	return &(*B)->d;
}

template <class T>
T *tablicaznakow <T>::dajznak (const numer &Z) const
{
	listaznakow <T> *B = A[Z.hash () & 0xFFF];
	while (B)
	{
		if (B->Z == Z) return &B->d;
		B = B->nast;
	}
	return NULL;
}

///////////////////////////////////////////////////////////////////////////////

int console_fd;
struct consolefontdesc oryginalny_font, font_na_ekranie;
struct unimapdesc oryginalne_sfm, sfm_na_ekranie;

int znakow = 512;
int wysokosc = 0;
int ascii = 1;
int pogrubiamy = 0, podkreslamy = 0;

struct znak
{
	char *rysunek;
	int ramka;
};

enum category {cat_normal, cat_Mn};

struct znakfontu
{
	znak *z;
	category cat;
	int comb;
	numer nast;
};

tablicaznakow<znakfontu> font;

unsigned long czytajhex (istream &f, int &c)
{
	unsigned long Z = 0;
	int i;
	for (i = 0; i < 8; i++)
	{
		if (c >= '0' && c <= '9')
			Z = (Z << 4) + (c - '0');
		else if (c >= 'A' && c <= 'F')
			Z = (Z << 4) + (10 + (c - 'A'));
		else
			break;
		c = f.get ();
	}
	return Z;
}

void czytajfont (char plik[])
{
	ifstream f (plik);
	znak *z = NULL;
	int y = 0;
	numer poprz_Z;
	int c = f.get ();
	for(;;)
	{
		switch (c)
		{
			case '[':
				if ((c = f.get ()) == 'U')
				{
					c = f.get ();
					numer Z;
					while (c == '+')
					{
						if (Z.fragmentow == MAX_DECOMPOSITION)
							goto ignorujemy;
						c = f.get ();
						if (!(c >= '0' && c <= '9' || c >= 'A' && c <= 'F'))
							goto ignorujemy;
						Z.nr[Z.fragmentow++] = czytajhex (f, c);
					}
					if (c == ']' && Z.fragmentow && Z != 0xFFFF && !font.dajznak (Z))
					{
						if (!z)
						{
							z = new znak;
							z->rysunek = new char[wysokosc ? wysokosc : 32];
							z->ramka = 0;
							y = 0;
							poprz_Z = 0xFFFF;
						}
						font.dajznak (poprz_Z)->nast = Z;
						poprz_Z = Z;
						znakfontu *zf = font.utworzznak (Z);
						zf->z = z;
						zf->cat = cat_normal;
					}
				}
				goto ignorujemy;
			case '\n':
			case -1:
				if (z)
				{
					if (y)
						if (wysokosc)
						{
							if (y != wysokosc)
							{
								cerr << "Different character heights in " << plik << "\n";
								exit (1);
							}
						}
						else
							wysokosc = y;
					else
						z->rysunek = NULL;
					font.dajznak (poprz_Z)->nast = font.dajznak (0xFFFF)->nast;
					z = NULL;
				}
				if (c == -1) goto koniec;
				c = f.get ();
				break;
			case 'C':
				if (z)
				{
					if ((c = f.get ()) == 'a')
					{
						if
						(
							(c = f.get ()) == 't' &&
							(c = f.get ()) == 'e' &&
							(c = f.get ()) == 'g' &&
							(c = f.get ()) == 'o' &&
							(c = f.get ()) == 'r' &&
							(c = f.get ()) == 'y' &&
							(c = f.get ()) == ':'
						)
						{
							while ((c = f.get ()) == ' ' || c == '\t');
							if
							(
								c == 'M' &&
								(c = f.get ()) == 'n'
							)
							{
								znakfontu *zf = font.dajznak (poprz_Z);
								zf->cat = cat_Mn;
								zf->comb = 0;
								c = f.get ();
							}
						}
					}
					else
					if
					(
						c == 'o' &&
						(c = f.get ()) == 'm' &&
						(c = f.get ()) == 'b' &&
						(c = f.get ()) == 'i' &&
						(c = f.get ()) == 'n' &&
						(c = f.get ()) == 'i' &&
						(c = f.get ()) == 'n' &&
						(c = f.get ()) == 'g' &&
						(c = f.get ()) == '-' &&
						(c = f.get ()) == 'C' &&
						(c = f.get ()) == 'l' &&
						(c = f.get ()) == 'a' &&
						(c = f.get ()) == 's' &&
						(c = f.get ()) == 's' &&
						(c = f.get ()) == ':'
					)
					{
						while ((c = f.get ()) == ' ' || c == '\t');
						if (c >= '0' && c <= '9')
						{
							znakfontu *zf = font.dajznak (poprz_Z);
							zf->comb = 0;
							while (c >= '0' && c <= '9')
							{
								zf->comb = zf->comb * 10 + (c - '0');
								c = f.get ();
							}
						}
					}
				}
				goto ignorujemy;
			case '.':
			case '0':
				if (z)
				{
					int wiersz = 0;
					int x = 0;
					do
					{
						wiersz = (wiersz << 1) + (c == '0');
						c = f.get ();
						if (++x == 9) break;
					}
						while (c == '.' || c == '0');
					while (c == ' ' || c == '\t') c = f.get ();
					if (x < 8 || c != '\n' && c != -1)
					{
						cerr << "Character width must be 8 or 9 in " << plik << "\n";
						exit (1);
					}
					if (c == '\n') c = f.get ();
					if (x == 9)
					{
						if (wiersz & 1) z->ramka |= 1;
						wiersz >>= 1;
					}
					if (y < 32) z->rysunek[y++] = wiersz;
					break;
				}
			default:
			ignorujemy:
				while (c != '\n' && c != -1) c = f.get ();
				if (c == '\n') c = f.get ();
		}
	}
koniec:
	for (unsigned long Z = ' '; Z < 127; Z++)
	{
		znakfontu *zf = font.dajznak (Z);
		if (!zf)
		{
			zf = font.utworzznak (Z);
			zf->z = new znak;
			zf->z->rysunek = NULL;
			zf->z->ramka = 0;
			zf->cat = cat_normal;
			zf->nast = Z;
		}
		if (!zf->z->rysunek)
			zf->z->rysunek = oryginalny_font.chardata + Z * 32;
	}
	znakfontu *zf = font.dajznak (0xFFFD);
	if (!zf)
	{
		zf = font.utworzznak (0xFFFD);
		zf->z = font.dajznak ('?')->z;
		zf->cat = cat_normal;
		zf->nast = font.dajznak ('?')->nast;
		font.dajznak ('?')->nast = 0xFFFD;
	}
	if (!zf->z->rysunek)
		zf->z->rysunek = font.dajznak ('?')->z->rysunek;
	font.dajznak (0xFFFF)->nast = 0xFFFF;
}

///////////////////////////////////////////////////////////////////////////////

tablicaznakow<unsigned short> nanowe;
numer nastare[512];
unsigned short poprzedni[514];
unsigned short nastepny[514];

void instaluj_sfm ()
{
	sfm_na_ekranie.entry_ct = 0;
	for (unsigned short Z = 0; Z < znakow; Z++)
	{
		numer Z1 = nastare[Z];
		do
		{
			if (Z1.fragmentow == 1 && *Z1.nr < 0xFFFF && !Z1.b && !Z1.ul)
			{
				sfm_na_ekranie.entries[sfm_na_ekranie.entry_ct].unicode = *Z1.nr;
				sfm_na_ekranie.entries[sfm_na_ekranie.entry_ct].fontpos = Z;
				sfm_na_ekranie.entry_ct++;
			}
			Z1 = font.dajznak (Z1)->nast;
		}
			while (Z1 != nastare[Z]);
	}
	set_kernel_unimap (console_fd, &sfm_na_ekranie);
}

void inicjalizacja ()
{
	cout << "\33%G";

	font_na_ekranie.charcount = znakow;
	font_na_ekranie.charheight = wysokosc;
	font_na_ekranie.chardata = new char[znakow * 32];
	memcpy
	(
		font_na_ekranie.chardata, oryginalny_font.chardata,
		(
			oryginalny_font.charcount < znakow ?
				oryginalny_font.charcount
			:
				znakow
		)
			* 32
	);
	sfm_na_ekranie.entries = new unipair[0x10000];
	for (unsigned short Z = 0; Z < znakow; Z++)
		nastare[Z] = 0xFFFF;
	for (unsigned short Z = ' '; Z < 127; Z++)
	{
		if (ascii || !font.dajznak (Z)->z->ramka)
		{
			numer Z1 = Z;
			do
			{
				*nanowe.utworzznak (Z1) = Z;
				Z1 = font.dajznak (Z1)->nast;
			}
				while (Z1 != Z);
			nastare[Z] = Z;
		}
		memcpy (font_na_ekranie.chardata + Z * 32,
			font.dajznak (Z)->z->rysunek, wysokosc);
	}
	numer Z1 = 0xFFFD;
	do
	{
		*nanowe.utworzznak (Z1) = 0;
		Z1 = font.dajznak (Z1)->nast;
	}
		while (Z1 != 0xFFFD);
	nastare[0] = 0xFFFD;
	memcpy (font_na_ekranie.chardata + 0 * 32,
		font.dajznak (0xFFFD)->z->rysunek, wysokosc);
	set_kernel_font (console_fd, &font_na_ekranie);
	instaluj_sfm ();

	for (unsigned short Z = 2; Z < znakow; Z++)
		poprzedni[Z] = Z - 1;
	poprzedni[ascii ? 0x7F : 0x21] = 512;
	poprzedni[0xC0] = 513;
	poprzedni[0xE0] = 0xBF;
	poprzedni[512] = 0x1F;
	switch (znakow)
	{
		case 256:
			poprzedni[1] = 0xFF;
			poprzedni[513] = 0xDF;
			break;
		case 512:
			poprzedni[1] = 0x1FF;
			poprzedni[0x1C0] = 0xDF;
			poprzedni[0x1E0] = 0x1BF;
			poprzedni[513] = 0x1DF;
			break;
	}
	for (unsigned short Z = 1; Z < znakow - 1; Z++)
		nastepny[Z] = Z + 1;
	nastepny[0x1F] = 512;
	nastepny[0xBF] = 0xE0;
	nastepny[512] = ascii ? 0x7F : 0x21;
	nastepny[513] = 0xC0;
	switch (znakow)
	{
		case 256:
			nastepny[0xDF] = 513;
			nastepny[0xFF] = 1;
			break;
		case 512:
			nastepny[0xDF] = 0x1C0;
			nastepny[0x1BF] = 0x1E0;
			nastepny[0x1DF] = 513;
			nastepny[0x1FF] = 1;
			break;
	}
}

unsigned short potrzeba_znaku (const numer &znak)
{
	znakfontu *zf = font.dajznak (znak);
	numer stary = zf && zf->z->rysunek ? znak : 0xFFFD;
	unsigned short *nowy1 = nanowe.dajznak (stary);
	unsigned short nowy = nowy1 ? *nowy1 : 0xFFFF;
	if
	(
		nowy != 0x20 && nowy != 0 &&
		!(ascii && nowy >= 0x20 && nowy <= 0x7E)
	)
	{
		unsigned short poczatek = 512 + font.dajznak (stary)->z->ramka;
		if (nowy == 0xFFFF)
		{
			nowy = poprzedni[poczatek];
			numer Z = nastare[nowy];
			do
			{
				*nanowe.utworzznak (Z) = 0xFFFF;
				Z = font.dajznak (Z)->nast;
			}
				while (Z != nastare[nowy]);
			Z = stary;
			do
			{
				*nanowe.utworzznak (Z) = nowy;
				Z = font.dajznak (Z)->nast;
			}
				while (Z != stary);
			nastare[nowy] = stary;
			memcpy (font_na_ekranie.chardata + nowy * 32,
				font.dajznak (stary)->z->rysunek, wysokosc);
			set_kernel_font (console_fd, &font_na_ekranie);
			instaluj_sfm ();
		}
		if (nowy != nastepny[poczatek])
		{
			poprzedni[nastepny[nowy]] = poprzedni[nowy];
			nastepny[poprzedni[nowy]] = nastepny[nowy];
			poprzedni[nowy] = poczatek;
			nastepny[nowy] = nastepny[poczatek];
			poprzedni[nastepny[poczatek]] = nowy;
			nastepny[poczatek] = nowy;
		}
	}
	return znak.fragmentow == 1 && *znak.nr < 0x10000 && !znak.b && !znak.ul ?
		*znak.nr : 0xF000 | nowy;
}

///////////////////////////////////////////////////////////////////////////////

int narysuj (const numer &Z)
{
	znakfontu *zf1 = font.dajznak (Z);
	if (zf1 && zf1->z->rysunek) return 1;
	numer Z1 = Z;
	if (Z1.fragmentow <= 1)
	{
		if (!zf1) return 0;
		do
		{
			Z1 = font.dajznak (Z1)->nast;
			if (Z1 == Z) return 0;
		}
			while (Z1.fragmentow <= 1);
	}
	znakfontu *zf2 = font.dajznak (Z1.nr[Z1.fragmentow - 1]);
	if (zf2 && zf2->cat == cat_Mn && zf2->z->rysunek)
	{
		Z1.fragmentow--;
		if (narysuj (Z1))
		{
			znakfontu *zf1 = font.dajznak (Z1);
			znakfontu *zf = font.dajznak (Z);
			if (!zf)
			{
				zf = font.utworzznak (Z);
				zf->z = new znak;
				zf->z->rysunek = NULL;
				zf->z->ramka = zf1->z->ramka;
				zf->cat = zf1->cat;
				zf->comb = zf1->comb;
				zf->nast = Z;
			}
			zf->z->rysunek = new char[wysokosc];
			memcpy (zf->z->rysunek, zf1->z->rysunek, wysokosc);
			int gora2 = 0;
			while (gora2 < wysokosc && !zf2->z->rysunek[gora2]) gora2++;
			if (gora2 != wysokosc)
			{
				int dol2 = wysokosc - 1;
				while (!zf2->z->rysunek[dol2]) dol2--;
				int gora1 = 0;
				while (gora1 < wysokosc && !zf->z->rysunek[gora1]) gora1++;
				int dol1 = wysokosc - 1;
				while (dol1 > 0 && !zf->z->rysunek[dol1]) dol1--;
				int przesuniecie = 0;
				if
				(
					zf2->comb == 200 || zf2->comb == 202 || zf2->comb == 204 ||
					zf2->comb == 218 || zf2->comb == 220 || zf2->comb == 222
				)
				{
					int odstep = zf2->comb < 218 ?
						zf2->comb == 202 ? 0 : -1
					:
						1;
					przesuniecie =
						gora1 == wysokosc ? 0 :
						wysokosc - 1 - dol1 >= dol2 - gora2 + 1 + odstep ?
							dol1 - gora2 + 1 + odstep
						:
							wysokosc - 1 - dol2;
				}
				else
				if
				(
					zf2->comb == 212 || zf2->comb == 214 || zf2->comb == 216 ||
					zf2->comb == 228 || zf2->comb == 230 || zf2->comb == 232
				)
				{
					int odstep = zf2->comb < 228 ?
						zf2->comb == 214 ? 0 : -1
					:
						1;
					przesuniecie =
						gora1 == wysokosc ? 0 :
						gora1 >= dol2 - gora2 + 1 + odstep ?
							gora1 - dol2 - 1 - odstep
						:
							-gora2;
				}
				for (int i = gora2; i <= dol2; i++)
					zf->z->rysunek[przesuniecie + i] |= zf2->z->rysunek[i];
			}
			return 1;
		}
	}
	return 0;
}

void pogrub (const numer &Z)
{
	if (font.dajznak (Z)) return;
	numer Z1 = Z;
	Z1.b = 0;
	znakfontu *zf1 = font.dajznak (Z1);
	if (!zf1) return;
	znak *z = new znak;
	z->rysunek = new char[wysokosc];
	memcpy (z->rysunek, zf1->z->rysunek, wysokosc);
	int wlewo = 1, wprawo = 1;
	for (int y = 0; y < wysokosc; y++)
	{
		if (z->rysunek[y] & 0x80) wlewo  = 0;
		if (z->rysunek[y] &    1) wprawo = 0;
	}
	if (wprawo || !wlewo)
		for (int y = 0; y < wysokosc; y++)
			z->rysunek[y] |= z->rysunek[y] >> 1;
	else
		for (int y = 0; y < wysokosc; y++)
			z->rysunek[y] |= z->rysunek[y] << 1;
	z->ramka = zf1->z->ramka;
	numer poprz_Z = 0xFFFF;
	Z1 = Z;
	do
	{
		font.dajznak (poprz_Z)->nast = Z1;
		poprz_Z = Z1;
		znakfontu *zf = font.utworzznak (Z1);
		zf->z = z;
		zf->cat = cat_normal;
		Z1.b = 0;
		Z1 = font.dajznak (Z1)->nast;
		Z1.b = 1;
	}
		while (Z1 != Z);
	font.dajznak (poprz_Z)->nast = font.dajznak (0xFFFF)->nast;
	font.dajznak (0xFFFF)->nast = 0xFFFF;
}

void podkresl (const numer &Z)
{
	if (font.dajznak (Z)) return;
	numer Z1 = Z;
	Z1.ul = 0;
	znakfontu *zf1 = font.dajznak (Z1);
	if (!zf1) return;
	znak *z = new znak;
	z->rysunek = new char[wysokosc];
	memcpy (z->rysunek, zf1->z->rysunek, wysokosc);
	int wlewo = 0, mozna = 1;
	if (!zf1->z->ramka)
	{
		for (int y = 0; y < wysokosc; y++)
		{
			if (z->rysunek[y] & 1) wlewo = 1;
			if (z->rysunek[y] & 0x80) mozna = 0;
		}
		if (wlewo && mozna)
			for (int y = 0; y < wysokosc; y++)
				z->rysunek[y] <<= 1;
	}
	znakfontu *podkreslenie = font.dajznak ('_');
	for (int y = 0; y < wysokosc; y++)
		z->rysunek[y] |= podkreslenie->z->rysunek[y];
	z->ramka = zf1->z->ramka || !wlewo || mozna;
	numer poprz_Z = 0xFFFF;
	Z1 = Z;
	do
	{
		font.dajznak (poprz_Z)->nast = Z1;
		poprz_Z = Z1;
		znakfontu *zf = font.utworzznak (Z1);
		zf->z = z;
		zf->cat = cat_normal;
		Z1.ul = 0;
		Z1 = font.dajznak (Z1)->nast;
		Z1.ul = 1;
	}
		while (Z1 != Z);
	font.dajznak (poprz_Z)->nast = font.dajznak (0xFFFF)->nast;
	font.dajznak (0xFFFF)->nast = 0xFFFF;
}

///////////////////////////////////////////////////////////////////////////////

/* Based on linux-2.0.35/drivers/char/console.c */

#define NPAR 16

int utf_count;
unsigned long utf_char;
int
	intensity = 1, underline = 0, s_intensity, s_underline,
	npar, par[NPAR];
numer Znak;

enum
{
	ESnormal, ESesc, ESsquare, ESgetpars, ESgotpars, ESfunckey,
	EShash, ESsetG0, ESsetG1, ESpercent, ESignore, ESnonstd,
	ESpalette
}
	vc_state;

void csi_m ()
{
	int bylo = 0;
	for (int i = 0; i <= npar; i++)
	{
		switch (par[i])
		{
			case 0:
				if (pogrubiamy)
					intensity = 1;
				if (podkreslamy)
					underline = 0;
				break;
			case 1:
				if (pogrubiamy)
					intensity = 2;
				break;
			case 2:
				if (pogrubiamy)
					intensity = 0;
				break;
			case 4:
				if (podkreslamy)
				{
					underline = 1;
					continue;
				}
				break;
			case 21:
			case 22:
				if (pogrubiamy)
					intensity = 1;
				break;
			case 24:
				if (podkreslamy)
				{
					underline = 0;
					continue;
				}
				break;
			case 38:
				if (podkreslamy)
				{
					underline = 1;
					par[i] = 49;
				}
				break;
			case 39:
				if (podkreslamy)
				{
					underline = 0;
					par[i] = 49;
				}
				break;
		}
		if (bylo) cout << ';';
		bylo = 1;
		cout << par[i];
	}
	cout << 'm';
}

void con_write (unsigned char c)
{
	unsigned long tc;
	if (c & 0x80)
	{
		if ((c & 0xC0) == 0x80)
		{
			if (utf_count > 0)
			{
				utf_char = (utf_char << 6) | (c & 0x3F);
				utf_count--;
				if (utf_count == 0)
					tc = utf_char;
				else
					return;
			}
			else
				return;
		}
		else
		{
			if ((c & 0xE0) == 0xC0)
			{
				utf_count = 1;
				utf_char = c & 0x1F;
			}
			else if ((c & 0xF0) == 0xE0)
			{
				utf_count = 2;
				utf_char = c & 0x0F;
			}
			else if ((c & 0xF8) == 0xF0)
			{
				utf_count = 3;
				utf_char = c & 0x07;
			}
			else if ((c & 0xFC) == 0xF8)
			{
				utf_count = 4;
				utf_char = c & 0x03;
			}
			else if ((c & 0xFE) == 0xFC)
			{
				utf_count = 5;
				utf_char = c & 0x01;
			}
			else
				utf_count = 0;
			return;
		}
	}
	else
	{
		tc = c;
		utf_count = 0;
	}

	if (tc < 32 || tc == 127 || tc == 128 + 27)
	{
		Znak.fragmentow = 0;
		switch (tc)
		{
			case 24:
			case 26:
				vc_state = ESnormal;
				break;
			case 27:
				vc_state = ESesc;
				break;
			case 128 + 27:
				vc_state = ESsquare;
				break;
		}
	}
	else
	{
		if (vc_state != ESnormal) Znak.fragmentow = 0;
		switch (vc_state)
		{
			case ESnormal:
				{
					znakfontu *zf = font.dajznak (tc);
					int jest = 0;
					if (Znak.fragmentow && Znak.fragmentow < MAX_DECOMPOSITION)
					{
						numer Znak1 = Znak;
						Znak1.nr[Znak1.fragmentow++] = tc;
						if (font.dajznak (Znak1))
							jest = 1;
					}
					if
					(
						jest ||
						zf && zf->cat == cat_Mn &&
						Znak.fragmentow
					)
						cout << '\b';
					else
						Znak.fragmentow = 0;
					if (Znak.fragmentow < MAX_DECOMPOSITION)
						Znak.nr[Znak.fragmentow++] = tc;
					narysuj (Znak);
					numer Znak1 = Znak;
					if (intensity == 2)
					{
						Znak1.b = 1;
						pogrub (Znak1);
					}
					if (underline)
					{
						Znak1.ul = 1;
						podkresl (Znak1);
					}
					unsigned short znak = potrzeba_znaku (Znak1);
					if (znak < 0x80)
						cout << (char) znak;
					else if (znak < 0x800)
						cout <<
							(char) (0xC0 | znak >> 6 & 0x1F) <<
							(char) (0x80 | znak & 0x3F);
					else
						cout <<
							(char) (0xE0 | znak >> 12 & 0x0F) <<
							(char) (0x80 | znak >> 6 & 0x3F) <<
							(char) (0x80 | znak & 0x3F);
				}
				return;
			case ESesc:
				vc_state = ESnormal;
				switch (tc)
				{
					case '[':
						vc_state = ESsquare;
						break;
					case ']':
						vc_state = ESnonstd;
						break;
					case '%':
						vc_state = ESpercent;
						break;
					case '(':
						vc_state = ESsetG0;
						break;
					case ')':
						vc_state = ESsetG1;
						break;
					case '#':
						vc_state = EShash;
						break;
					case 'c':
						if (pogrubiamy) intensity = 1;
						if (podkreslamy) underline = 0;
						break;
				}
				break;
			case ESnonstd:
				vc_state = tc == 'P' ? npar = 0, ESpalette : ESnormal;
				break;
			case ESpalette:
				if
				(
					(tc >= '0' && tc <= '9') ||
					(tc >= 'A' && tc <= 'F') ||
					(tc >= 'a' && tc <= 'f')
				)
				{
					if (++npar == 7) vc_state = ESnormal;
				}
				else
					vc_state = ESnormal;
				break;
			case ESsquare:
				for (npar = 0; npar < NPAR; npar++)
					par[npar] = 0;
				npar = 0;
				vc_state = ESgetpars;
				if (tc == '[')
				{
					vc_state = ESfunckey;
					break;
				}
				if (tc == '?') break;
			case ESgetpars:
				if (tc == ';' && npar < NPAR - 1)
				{
					npar++;
					return;
				}
				else if (tc >= '0' && tc <= '9')
				{
					par[npar] *= 10;
					par[npar] += tc - '0';
					return;
				}
				else
					vc_state = ESgotpars;
			case ESgotpars:
				vc_state = ESnormal;
				switch (tc)
				{
					case 'm':
						csi_m ();
						return;
					case 's':
						if (pogrubiamy)
							s_intensity = intensity;
						if (podkreslamy)
							s_underline = underline;
						break;
					case 'u':
						if (pogrubiamy)
							intensity = s_intensity;
						if (podkreslamy)
							underline = s_underline;
						break;
				}
				for (int i = 0; i <= npar; i++)
				{
					cout << par[i];
					if (i < npar) cout << ';';
				}
				break;
			default:
				vc_state = ESnormal;
		}
	}
	cout << c;
}

///////////////////////////////////////////////////////////////////////////////

main (int argc, char *argv[])
{
	char *nazwaprogramu = *argv++;
	while (*argv)
	{
		if (!strcmp (*argv, "-256"))
		{
			znakow = 256;
			argv++;
		}
		if (!strcmp (*argv, "-ascii"))
		{
			ascii = 0;
			argv++;
		}
		if (!strcmp (*argv, "-bold"))
		{
			pogrubiamy = 1;
			argv++;
		}
		if (!strcmp (*argv, "-underline"))
		{
			podkreslamy = 1;
			argv++;
		}
		else
			break;
	}
	if (!*argv)
	{
		cerr << "Usage: " << nazwaprogramu << " [-256] [-ascii] [-bold] [-underline] font_file.sbf [...]\n";
		exit (1);
	}

	if ((console_fd = get_console_fd (NULL)) == -1) exit (1);
	get_kernel_font (console_fd, &oryginalny_font);
	get_kernel_unimap (console_fd, &oryginalne_sfm);
	font.utworzznak (0xFFFF)->nast = 0xFFFF;
	while (*argv) czytajfont (*argv++);
	inicjalizacja ();

	int c;
	while ((c = cin.get ()) != -1) con_write (c);

	set_kernel_font (console_fd, &oryginalny_font);
	set_kernel_unimap (console_fd, &oryginalne_sfm);

	return 0;
}
