#include "ganglia.h"
#include "metric_typedefs.h"
#include "kstat.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/utsname.h>
#include <strings.h>

/* used for swap space determination - swapctl() 
 * and anon.h has the data structure  needed for swapctl to be useful
 */
 
#include <sys/stat.h>
#include <sys/swap.h>
#include <vm/anon.h>

/* we MUST use kvm.  make sure you use the right one, too.
 * otherwise you will get errors. (32 vs 64-bit)
 */

#include <kvm.h>
#include <fcntl.h>

/* we get the cpu struct from cpuvar, maybe other mojo too
 *
 */

#include <sys/var.h>
#include <sys/cpuvar.h>
#include <sys/time.h>

/*
 * functions spackled in by swagner -- the CPU-percentage-specific code is
 * largely an imitation (if not a shameless copy) of the Solaris-specific
 * code for top.
 */

/*  number of seconds to wait before refreshing/recomputing values off kstat */

#define TICK_SECONDS 		30

#ifndef FSCALE
#define FSHIFT  8               /* bits to right of fixed binary point */
#define FSCALE  (1<<FSHIFT)
#endif /* FSCALE */

/*  also imported from top, i'm basically using the same CPU cycle summing algo */

#define CPUSTATES	5
#define CPUSTATE_IOWAIT 3
#define CPUSTATE_SWAP   4

/* definitions for indices in the nlist array */
#define X_V                      0
#define X_MPID                   1
#define X_ANONINFO               2
#define X_MAXMEM                 3
#define X_SWAPFS_MINFREE         4
#define X_FREEMEM                5
#define X_AVAILRMEM              6
#define X_AVENRUN                7
#define X_CPU                    8
#define X_NPROC                  9
#define X_NCPUS                 10

int first_run = 1;

static struct nlist nlst[] =
{
  {"v"},                        /* 0 */ /* replaced by dynamic allocation */
  {"mpid"},                     /* 1 */
#if OSREV >= 56
  /* this structure really has some extra fields, but the first three match */
  {"k_anoninfo"},               /* 2 */
#else
  {"anoninfo"},                 /* 2 */
#endif
  {"maxmem"},                   /* 3 */ /* use sysconf */
  {"swapfs_minfree"},           /* 4 */ /* used only w/ USE_ANONINFO */
  {"freemem"},                  /* 5 */ /* available from kstat >= 2.5 */
  {"availrmem"},                /* 6 */ /* available from kstat >= 2.5 */
  {"avenrun"},                  /* 7 */ /* available from kstat */
  {"cpu"},                      /* 8 */ /* available from kstat */
  {"nproc"},                    /* 9 */ /* available from kstat */
  {"ncpus"},                    /* 10 */ /* available from kstat */
  {0}
};


/* support macros for the percentage computations */

#define loaddouble(la) ((double)(la) / FSCALE)

static struct utsname unamedata;
static unsigned long *cpu_offset;
struct cpu_info {
    unsigned int 		bread;
    unsigned int 		bwrite;
    unsigned int 		lread;
    unsigned int 		lwrite;
    unsigned int 		phread;
    unsigned int 		phwrite;
};

struct g_kvmdata {
    char *      label;
    long        offset;
    long        value;
};

static struct nlist nlst[];

// static struct g_kvmdata kvmdatalist[10];

struct g_metrics_struct {
    g_val_t		boottime;
    g_val_t		cpu_wio;
    g_val_t		cpu_idle;
    g_val_t		cpu_nice;
    g_val_t		cpu_system;
    g_val_t		cpu_user;
    g_val_t		cpu_num;
    g_val_t		cpu_speed;
    g_val_t		load_one;
    g_val_t		load_five;
    g_val_t		load_fifteen;
    char *		machine_type;
    g_val_t		mem_buffers;
    g_val_t		mem_cached;
    g_val_t		mem_free;
    g_val_t		mem_shared;
    g_val_t		mem_total;
    char *		os_name;
    char *		os_release;
    g_val_t 		proc_run;
    g_val_t 		proc_total;
    g_val_t		swap_free;
    g_val_t		swap_total;
    g_val_t		sys_clock;
    g_val_t		bread_sec;
    g_val_t		bwrite_sec;
    g_val_t		lread_sec;
    g_val_t		lwrite_sec;
    g_val_t		phread_sec;
    g_val_t		phwrite_sec;
    g_val_t		rcache;
    g_val_t		wcache; 



};

static struct g_metrics_struct metriclist;
kvm_t *kd;
static int ncpus;

/* begin of "ripped off from top" section 
 */

/*
 *  getkval(offset, ptr, size, refstr) - get a value out of the kernel.
 *      "offset" is the byte offset into the kernel for the desired value,
 *      "ptr" points to a buffer into which the value is retrieved,
 *      "size" is the size of the buffer (and the object to retrieve),
 *      "refstr" is a reference string used when printing error meessages,
 *          if "refstr" starts with a '!', then a failure on read will not
 *          be fatal (this may seem like a silly way to do things, but I
 *          really didn't want the overhead of another argument).
 *
 */
int
getkval (unsigned long offset,
         int *ptr,
         int size,
         char *refstr)
{
  if (kvm_read (kd, offset, (char *) ptr, size) != size)
    {
    debug_msg("There was a problem!  I'll try to reload.");
    kvm_close(kd);
    kd = kvm_open (NULL, NULL, NULL, O_RDONLY, NULL);
      if (*refstr == '!')
        {
          debug_msg("gmond: kvm_read for %s: %s", refstr, strerror(errno));
          return (1);
        }
      else
        {
          debug_msg ("gmond: kvm_read for %s: %s", refstr, strerror(errno));
          exit (23);
        }
    }
  return (1);

}


/* end of "ripped off from top" section
 */

int
get_metric_val(g_val_t *val, char *ks_name, char *name)
{
   /* Warning.. always assuming a KSTAT_DATA_ULONG here */
   int rval;
   kstat_t *ks;
   kstat_ctl_t *kc;
   kstat_named_t *kn;

   kc = kstat_open();
   if (kc == NULL)
      {
         err_ret("get_metric_val() kstat_open() error");
         return SYNAPSE_FAILURE;
      }

   ks = kstat_lookup(kc, "unix", 0, ks_name);
   if (ks == NULL)
      {
         err_ret("get_metric_val() kstat_lookup() error");
         return SYNAPSE_FAILURE;
      }

   rval = kstat_read(kc, ks, 0);
   if ( rval <0)
      {
         err_ret("get_metric_val() kstat_read() error");
         return SYNAPSE_FAILURE;
      }

   kn = kstat_data_lookup(ks, name);
   if ( kn == NULL )
      {
         err_ret("kstat_data_lookup() kstat_read() error");
         return SYNAPSE_FAILURE;
      }

   val->uint32 = kn->value.ui32;
   kstat_close(kc);

   return SYNAPSE_SUCCESS;
}

int
get_kstat_val(g_val_t *val, char *km_name, char *ks_name, char *name)
{
   /* Warning.. always assuming a KSTAT_DATA_ULONG here */
   kstat_t *ks;
   kstat_ctl_t *kc;
   kstat_named_t *kn;

/*  run the open call before we start worrying there was an error */
   kc = kstat_open();
   if (kc == NULL)
      {
         debug_msg("couldn't open kc...");
         err_ret("get_kstat_val() kstat_open() error");
         return SYNAPSE_FAILURE;
      }

   debug_msg( "Lookup up kstat:  km (unix?)='%s', ks (system_misc?)='%s',kn (resulting metric?)='%s'", km_name, ks_name, name);
   debug_msg( "%s: kc is %p", name, kc);
   ks = kstat_lookup(kc, km_name, 0, ks_name);
   debug_msg("%s: Just did kstat_lookup().",name);
   if (ks == NULL)
      {
      perror("ks");
      }
   debug_msg("%s: Looked up.", name);
    if (kstat_read(kc, ks, 0) == -1) {
        perror("kstat_read");
        return SYNAPSE_FAILURE;
    }
   kn = kstat_data_lookup(ks, name);
   if ( kn == NULL )
      {
         err_ret("get_kstat_val() kstat_data_lookup() kstat_read() error");
         return SYNAPSE_FAILURE;
      }
   debug_msg( "%s: Kstat data type:  %d, Value returned: %u, %d %u %d", name, (int)kn->data_type, (int)kn->value.ui32, (int)kn->value.l, kn->value.ul, kn->value.ui64);
//    ks = kstat_lookup(kc, "unix", 0, "system_misc");

   if (kn->value.ui32 == 0)
      val->uint32 = (unsigned long)kn->value.ul;
   else
      val->uint32 = (int)kn->value.ui32;
   sleep(0);
   debug_msg("%s: Kernel close.  Val returned: %d", name, val->uint32);
   kstat_close(kc);

   return SYNAPSE_SUCCESS;
}

unsigned int
pagetok( int pageval )
{
    unsigned int foo;
    foo = pageval;
    foo = foo * (sysconf(_SC_PAGESIZE) / 1024);
    debug_msg("PageToK():  %u * PAGESIZE (%u) / 1024 (Kb conversion) = %u", pageval, sysconf(_SC_PAGESIZE), foo);
    return foo;
}


/*
 *  there's too much legwork for each function to handle its metric.
 *  hence the updater.
 */

void
determine_swap_space( unsigned int *total, unsigned int *fr )
{
    struct anoninfo anon;
    if (swapctl(SC_AINFO, &anon)  == -1 )  {
       *total = *fr = 0;
       return;
    }
    /*  we are going from swap pages to kilobytes, so the conversion works... */
    *total = pagetok(anon.ani_max);
    *fr = pagetok((anon.ani_max - anon.ani_resv));
    debug_msg("Old/new:  Total = %u/%u , Free = %u/%u", anon.ani_max,*total,(anon.ani_max - anon.ani_resv),*fr);
    return;
}

int
update_metric_data ( void )
{
   debug_msg("running update_metric_data() ... ");
   get_kstat_val(&metriclist.boottime, "unix","system_misc","boot_time");
   get_kstat_val(&metriclist.cpu_speed,    "cpu_info","cpu_info0","clock_MHz");
   get_kstat_val(&metriclist.load_fifteen, "unix", "system_misc","avenrun_15min");
   get_kstat_val(&metriclist.load_five,    "unix", "system_misc","avenrun_5min");
   get_kstat_val(&metriclist.load_one,     "unix", "system_misc","avenrun_1min");
   get_kstat_val(&metriclist.proc_total,   "unix", "system_misc", "nproc");
/*  
 * memory usage stats are arguably VERY broken.
 */
   get_kstat_val(&metriclist.mem_free,     "unix", "system_pages", "pagesfree");
   get_kstat_val(&metriclist.mem_total,    "unix", "system_pages", "pagestotal");
   debug_msg("Before PageToK():  mem_free = %u, mem_total = %u",metriclist.mem_free.uint32,metriclist.mem_total.uint32);
   metriclist.mem_free.uint32 = pagetok(metriclist.mem_free.uint32);
   metriclist.mem_total.uint32 = pagetok(metriclist.mem_total.uint32);
   determine_swap_space(&metriclist.swap_total.uint32,&metriclist.swap_free.uint32);
//   (void)determine_cpu_percentages();
//   sleep(5);

   /*  update the timestamp.  we use this to determine freshening times as well. */
   metriclist.sys_clock.uint32 = time(NULL);
   return 0;
}

/* another function ripped from top.  after all we want the CPU percentage
 * stuff to match.
 */

long percentages(cnt, out, new, old, diffs)

int cnt;
int *out;
register long *new;
register long *old;
long *diffs;

{
    register int i;
    register long change;
    register long total_change;
    register long *dp;
    long half_total;

    /* initialization */
    total_change = 0;
    dp = diffs;

    /* calculate changes for each state and the overall change */
    for (i = 0; i < cnt; i++)
    {
        if ((change = *new - *old) < 0)
        {
            /* this only happens when the counter wraps */
            change = (int)
                ((unsigned long)*new-(unsigned long)*old);
        }
        total_change += (*dp++ = change);
        *old++ = *new++;
    }

    /* avoid divide by zero potential */
    if (total_change == 0)
    {
        total_change = 1;
    }

    /* calculate percentages based on overall change, rounding up */
    half_total = total_change / 2l;
    for (i = 0; i < cnt; i++)
    {
        *out++ = (int)((*diffs++ * 1000 + half_total) / total_change);
    }

    /* return the total in case the caller wants to use it */
    return(total_change);
}

/*
 * the process for figuring out CPU usage is a little involved, so it has
 * been folded into this function.  also because this way it's easier to
 * rip off top.  :)
 */

int
determine_cpu_percentages ( void )
{
/*  hopefully this doesn't get too confusing.
 *  cpu_snap is a structure from <sys/cpuvar.h> and is the container into which
 *  we read the current CPU metrics.
 *  the static array "cpu_old" contains the last iteration's summed cycle
 *  counts.
 *  the array "cpu_now" contains the current iteration's summed cycle
 *  counts.
 *  "cpu_diff" holds the delta.
 *  across CPUs of course. :)
 *  buffers[0..2] holds past, present and diff info for the "other" CPU stats.
 */
   struct cpu cpu_snap;
   static struct cpu_info buffers[2];
   static struct timeval lasttime = {0, 0};
   struct timeval thistime;
   double timediff;
   int cpu_states[CPUSTATES];
   unsigned int ncpus;
   static unsigned long cpu_old[CPUSTATES];
   unsigned long cpu_now[CPUSTATES];
   unsigned int cpu_diff[CPUSTATES];
   float floaty;
   unsigned long diff_cycles = 0L;
   unsigned int time_delta = 0L;
   double alpha, beta;  // lambda lambda lambda!!!
   static unsigned long last_refresh;
   register int i, j;

/*  ripped from top by swagner in the hopes of getting
 *  top-like CPU percentages ...
 */
   gettimeofday (&thistime, NULL);

   if (lasttime.tv_sec)
     timediff = ((double) thistime.tv_sec * 1.0e7 +
		 ((double) thistime.tv_usec * 10.0)) -
	((double) lasttime.tv_sec * 1.0e7 +
	 ((double) lasttime.tv_usec * 10.0));
   else
     timediff = 1.0e7;

  /*
     * constants for exponential average.  avg = alpha * new + beta * avg
     * The goal is 50% decay in 30 sec.  However if the sample period
     * is greater than 30 sec, there's not a lot we can do.
     */ 
  if (timediff < 30.0e7)
    {
      alpha = 0.5 * (timediff / 30.0e7);
      beta = 1.0 - alpha;
      debug_msg("* * * * Setting alpha to %f and beta to %f because timediff = %d",alpha,beta,timediff);
    }           
  else          
    {
      alpha = 0.5;
      beta = 0.5;
    }   

    lasttime = thistime;

/*  END SECTION RIPPED BLATANTLY FROM TOP :) */

   ncpus = metriclist.cpu_num.uint32;

/* cpu_offset defined in the init function as an array of kernel memory offsets
 * containing cpuinfo structures.  we walk through all available CPUs.
 * but first we hose out cpu_now, which is static after all...
 */
   for (j = 0; j < CPUSTATES; j++)
      cpu_now[j] = 0;

// will C let me do this?  :)

   buffers[0].bread = 0L;
   buffers[0].bwrite = 0L;
   buffers[0].lread = 0L;
   buffers[0].lwrite = 0L;
   buffers[0].phread = 0L;
   buffers[0].phwrite = 0L;

   if (first_run == 1)
      {
   debug_msg("Initializing old read/write buffer... ");
   buffers[1].bread = 0L;
   buffers[1].bwrite = 0L;
   buffers[1].lread = 0L;
   buffers[1].lwrite = 0L;
   buffers[1].phread = 0L;
   buffers[1].phwrite = 0L;
   time_delta = 0;
      }

   for (i = 0; i < ncpus; i++)
      {
      /* it places the kval in the basket ... errr cpu struct */
      if (getkval (cpu_offset[i], &cpu_snap, sizeof (struct cpu), "cpu") )
         {
         perror("cpustuff");
         }
      
      /* sum up to the wait state counter, the last two we determine ourselves */

      for (j = 0; j < CPU_WAIT; j++)
          cpu_now[j] += (unsigned long) cpu_snap.cpu_stat.cpu_sysinfo.cpu[j];
      cpu_now[CPUSTATE_IOWAIT] += (unsigned long) cpu_snap.cpu_stat.cpu_sysinfo.wait[W_IO] +
				  (unsigned long) cpu_snap.cpu_stat.cpu_sysinfo.wait[W_PIO];
      cpu_now[CPUSTATE_SWAP] += (unsigned long) cpu_snap.cpu_stat.cpu_sysinfo.wait[W_SWAP];

      debug_msg("offset = %d, cpu_now[1] = %d", cpu_offset[i], cpu_now[1]);

      buffers[0].bread += (long)cpu_snap.cpu_stat.cpu_sysinfo.bread;
      buffers[0].bwrite += (long)cpu_snap.cpu_stat.cpu_sysinfo.bwrite;
      buffers[0].lread += (long)cpu_snap.cpu_stat.cpu_sysinfo.lread;
      buffers[0].lwrite += (long)cpu_snap.cpu_stat.cpu_sysinfo.lwrite;
      buffers[0].phread += (long)cpu_snap.cpu_stat.cpu_sysinfo.phread;
      buffers[0].phwrite += (long)cpu_snap.cpu_stat.cpu_sysinfo.phwrite;

      }
/* now we have our precious "data" and have to manipulate it - compare new to old
 * and calculate percentages and sums.
 */
   buffers[2].bread = buffers[0].bread - buffers[1].bread;
   buffers[2].bwrite = buffers[0].bwrite - buffers[1].bwrite;
   buffers[2].lread = buffers[0].lread - buffers[1].lread;
   buffers[2].lwrite = buffers[0].lwrite - buffers[1].lwrite;
   buffers[2].phread = buffers[0].phread - buffers[1].phread;
   buffers[2].phwrite = buffers[0].phwrite - buffers[1].phwrite;

   debug_msg("Raw:  bread / bwrite / lread / lwrite / phread / phwrite\n%u,%u,%u / %u,%u,%u / %u,%u,%u / %u,%u,%u / %u,%u,%u / %u,%u,%u",
   buffers[0].bread,buffers[1].bread,buffers[2].bread,
   buffers[0].bwrite,buffers[1].bwrite,buffers[2].bwrite,
   buffers[0].lread,buffers[1].lread,buffers[2].lread,
   buffers[0].lwrite,buffers[1].lwrite,buffers[2].lwrite,
   buffers[0].phread,buffers[1].phread,buffers[2].phread,
   buffers[0].phwrite,buffers[1].phwrite,buffers[2].phwrite);

   time_delta = (unsigned long)time(NULL) - (unsigned long)last_refresh;
   if (time_delta == 0)           
      time_delta = 1; 

/* decay stuff
 * semi-stolen from top.  :)  added by swagner on 8/20/02
 */
   if (time_delta < 30)
	{
	alpha = 0.5 * (time_delta / 30);
	beta = 1.0 - alpha;
	}
   else
	{
	alpha = 0.5;
	beta = 0.5;
	}
   metriclist.bread_sec.f = (float)buffers[2].bread / (float)time_delta;
   if (buffers[1].bread == buffers[0].bread)
      metriclist.bread_sec.f = 0;

   metriclist.bwrite_sec.f  = (float)buffers[2].bwrite / (float)time_delta;
   if (buffers[1].bwrite == buffers[0].bwrite)
      metriclist.bwrite_sec.f = 0L;

   metriclist.lread_sec.f   = (float)buffers[2].lread / (float)time_delta;
   if (buffers[1].bwrite == buffers[0].lread)
      metriclist.lread_sec.f = 0L;       

   metriclist.lwrite_sec.f  = (float)buffers[2].lwrite / (float)time_delta;
   if (buffers[1].bwrite == buffers[0].lwrite)                    
      metriclist.lwrite_sec.f = 0L;                           

   metriclist.phread_sec.f  = (float)buffers[2].phread / (float)time_delta;
   if (buffers[1].bwrite == buffers[0].phread)                    
      metriclist.phread_sec.f = 0L;                           

   metriclist.phwrite_sec.f = (float)buffers[2].phwrite / (float)time_delta;
   if (buffers[1].bwrite == buffers[0].phwrite)                    
      metriclist.phwrite_sec.f = 0L; 

   debug_msg("Aftermath: %f %f %f %f %f %f delta = %u", 
           metriclist.bread_sec.f, metriclist.bwrite_sec.f,
           metriclist.lread_sec.f, metriclist.lwrite_sec.f,
           metriclist.phread_sec.f, metriclist.phwrite_sec.f,
	   time_delta
   );

   buffers[1].bread = buffers[0].bread;
   buffers[1].bwrite = buffers[0].bwrite;
   buffers[1].lread = buffers[0].lread;
   buffers[1].lwrite = buffers[0].lwrite;
   buffers[1].phread = buffers[0].phread;
   buffers[1].phwrite = buffers[0].phwrite;

   diff_cycles = percentages(CPUSTATES, cpu_states, cpu_now, cpu_old, cpu_diff);

   debug_msg ("** ** ** ** ** Are percentages electric?  Try %d%%, %d%% , %d%% , %d%% , %d%% %d%%", cpu_states[0],cpu_states[1],cpu_states[2],cpu_states[3],cpu_states[4]);

/* i don't know how you folks do things in new york city, but around here folks
 * don't go around dividing by zero.
 */
   if (diff_cycles < 1)
       {
       debug_msg("diff_cycles < 1 ... == %f %u!", diff_cycles, diff_cycles);
       diff_cycles = 1;
       }

/* could this be ANY HARDER TO READ?  sorry.  through hacking around i found that
 * explicitly casting everything as floats seems to work...
 */
   metriclist.cpu_idle.f = (float) cpu_states[0] / 10;
   metriclist.cpu_user.f = (float) cpu_states[1] / 10;
   metriclist.cpu_system.f = (float)(cpu_states[2] + cpu_states[4]) / 10;
   metriclist.cpu_wio.f = (float) cpu_states[CPUSTATE_IOWAIT] / 10;

   metriclist.rcache.f = 100.0 * ( 1.0 - ( (float)buffers[0].bread / (float)buffers[0].lread ) );
   metriclist.wcache.f = 100.0 * ( 1.0 - ( (float)buffers[0].bwrite / (float)buffers[0].lwrite ) );

   last_refresh = time(NULL);
//   sleep(5);
   return(0);
}

/*
 * This function is called only once by the gmond.  Use to 
 * initialize data structures, etc or just return SYNAPSE_SUCCESS;
 */
g_val_t
metric_init( void )
{

/* swagner's stuff below, initialization for reading running kernel data ...
 */

   g_val_t val;
   register int i;
   int offset;

   /* ripped from top - perform the kvm_open - suppress error here */
   kd = kvm_open (NULL, NULL, NULL, O_RDONLY, NULL);
    if (kd == NULL)
      {
        /* Print the failure message here */
        (void) kvm_open (NULL, NULL, NULL, O_RDONLY, "gmond");
        perror("kvm_open");
        debug_msg(" *** WARNING!!!!  kvm_open() failed.  prepare for a segfault ... *** ");
        printf("*** kvm_open() failed, are you running gmond as root?\n");
      }
    if (kvm_nlist (kd, nlst) < 0)
      {
        perror ("kvm_nlist");
//        return (-1);
      }
   get_kstat_val(&metriclist.cpu_num, "unix","system_misc","ncpus");
   debug_msg("metric_init: Assigning cpu_num value (%d) to ncpus.",(int)metriclist.cpu_num.uint32);
   ncpus = metriclist.cpu_num.uint32;
/* cpu_offset is an array of longs containing the kvm offsets of cpu stat arrays
 * for all CPUs installed (SMP!) ... this is needed for determine_cpu_percentages().
 */
   cpu_offset = (unsigned long *) malloc (ncpus * sizeof (unsigned long ));
   for (i = offset = 0; i < ncpus; offset += sizeof(unsigned long))
      {
      (void) getkval (nlst[X_CPU].n_value + offset, &cpu_offset[i], 
                      sizeof (unsigned long), nlst[X_CPU].n_name );
      if (cpu_offset[i] != 0)
         i++;

      }
   // kvm_close(kd);

/* first we get the uname data (hence my including <sys/utsname.h> ) */
   (void) uname( &unamedata );
/* these values don't change from tick to tick.  at least, they shouldn't ... 
 * also, these strings don't use the ganglia metric struct!
 */
   metriclist.os_name = unamedata.sysname;
   metriclist.os_release = unamedata.release;
   metriclist.machine_type = unamedata.machine;
   update_metric_data();
   debug_msg("solaris.c: metric_init() ok.");
   val.int32 = SYNAPSE_SUCCESS;
   first_run = 0;
   return val;
}

void
metric_tick ( void )
{
   double thetime = time(NULL);
   /*  update every 30 seconds */
   if ( thetime >= ( metriclist.sys_clock.uint32 + TICK_SECONDS) )
      update_metric_data();
}

g_val_t
cpu_num_func ( void )
{
   g_val_t val;

   val.uint16 = metriclist.cpu_num.uint32;
   return val;
}

/* 
g_val_t
mtu_func ( void )
{
   g_val_t val;

   val.uint32 = 601;
   return val;
}
*/

/* the following funcs were written by the almighty Federico Sacerdoti */
/* ===================================================== */
/* Using the UNIX-independant libdnet library */
#include "dnet.h"

static int
find_mtu(const struct intf_entry *entry, void *arg)
{
   unsigned int mtu;
   unsigned int *min = (unsigned int *) arg;

   /* Only consider interfaces that are up. */
   if (! entry->intf_flags & INTF_FLAG_UP)
      return 0;

   mtu=entry->intf_mtu;
   if ( !*min || *min>mtu)
      *min=mtu;

   return 0;
}

/* --------------------------------------------------------------------------- */
g_val_t
mtu_func ( void )
{
   /* We want to find the minimum MTU (Max packet size) over all UP interfaces. */
   unsigned int min=0;
   g_val_t val;

   intf_t *intf;
   intf = intf_open();
   intf_loop(intf, find_mtu, &min);
   intf_close(intf);
   val.uint32 = min;

   /* A val of 0 means there are no UP interfaces. Shouldn't happen. */
   return val;
}

/* --- snip!  preceding code lifted from linux.c --- */

g_val_t
cpu_speed_func ( void )     
{
   g_val_t val;

   val.uint32 = metriclist.cpu_speed.uint32;
   return val;
}

g_val_t
mem_total_func ( void )
{
   g_val_t val;

   val.uint32 = metriclist.mem_total.uint32;
//   val.uint32 = pagetok(sysconf(_SC_PHYS_PAGES));
   return val;
}

g_val_t
swap_total_func ( void )
{
   g_val_t val;
   metric_tick();
   val.uint32 = metriclist.swap_total.uint32;
   return val;
}

g_val_t
boottime_func ( void )
{
   g_val_t val;

   val.uint32 = metriclist.boottime.uint32;
   return val;
}

g_val_t
sys_clock_func ( void )
{
   g_val_t val;

   metric_tick();
   val.uint32 = (uint32_t)time(NULL);
   return val;
}

g_val_t
machine_type_func ( void )
{
   g_val_t val;

   strncpy( val.str, metriclist.machine_type, MAX_G_STRING_SIZE );
   return val;
}

g_val_t
os_name_func ( void )
{
   g_val_t val;

   strncpy( val.str, unamedata.sysname, MAX_G_STRING_SIZE );
   return val;
}        

g_val_t
os_release_func ( void )
{
   g_val_t val;
 
   strncpy( val.str, unamedata.release, MAX_G_STRING_SIZE );
   return val;
}        

g_val_t
cpu_user_func ( void )
{
   g_val_t val;

   determine_cpu_percentages();
   val.f = metriclist.cpu_user.f;
   return val;
}

g_val_t
cpu_nice_func ( void )
{
   g_val_t val;

   val.f = 0.0;   /*  no more mr. nice procs ... */
   return val;
}

g_val_t 
cpu_system_func ( void )
{
   g_val_t val;

   val.f = metriclist.cpu_system.f;
   return val;
}

g_val_t 
cpu_idle_func ( void )
{
   g_val_t val;

   val.f = metriclist.cpu_idle.f;
   return val;
}

g_val_t 
cpu_wio_func ( void )
{
   g_val_t val;
   
   val.f = metriclist.cpu_wio.f;
   return val;
}

g_val_t
load_one_func ( void )
{
   g_val_t val;

   metric_tick();
   val.f = metriclist.load_one.uint32;

   val.f = loaddouble(val.f);
   return val;
}

g_val_t
load_five_func ( void )
{
   g_val_t val;
 
   metric_tick();
   val.f = metriclist.load_five.uint32;
   val.f = loaddouble(val.f);
   return val;
}

g_val_t
load_fifteen_func ( void )
{
   g_val_t val;

   metric_tick();
   val.f = metriclist.load_fifteen.uint32;
   val.f = loaddouble(val.f);
   return val;
}

g_val_t
proc_run_func( void )
{
   g_val_t val;

   metric_tick();
   val.uint32 = 0;
   return val;
}

g_val_t
proc_total_func ( void )
{
   g_val_t val;

   metric_tick();
   val = metriclist.proc_total;
   return val;
}

g_val_t
mem_free_func ( void )
{
   g_val_t val;

   metric_tick();
   val.uint32 = metriclist.mem_free.uint32;
   return val;
}

g_val_t
mem_shared_func ( void )
{
   g_val_t val;

   val.uint32 = 0;
   return val;
}

g_val_t
mem_buffers_func ( void )
{
   g_val_t val;

   val.uint32 = 0;
   return val;
}

g_val_t
mem_cached_func ( void )
{
   g_val_t val;

   val.uint32 = 0;
   return val;
}

g_val_t
swap_free_func ( void )
{
   g_val_t val;

   metric_tick();
   val.uint32 = metriclist.swap_free.uint32;
   return val;
}

/* some solaris-specific stuff.  enjoy. */

g_val_t
bread_sec_func(void)
{
   g_val_t val;
   
   val.f = metriclist.bread_sec.f;
   return val;
}
g_val_t
bwrite_sec_func(void)
{
   g_val_t val;

   val.f = metriclist.bwrite_sec.f;
   return val;
}
g_val_t
lread_sec_func(void)
{
   g_val_t val;

   val.f = metriclist.lread_sec.f;
   return val;
}
g_val_t
lwrite_sec_func(void)
{
   g_val_t val;

   val.f = metriclist.lwrite_sec.f;
   return val;
}
g_val_t
phread_sec_func(void)
{
   g_val_t val;

   val.f = metriclist.phread_sec.f;
   return val;
}
g_val_t
phwrite_sec_func(void)
{
   g_val_t val;

   val.f = metriclist.phwrite_sec.f;
   return val;
}
g_val_t
rcache_func(void)
{
   g_val_t val;

   val.f = metriclist.rcache.f;
   return val;
}
g_val_t
wcache_func(void)
{
   g_val_t val;

   val.f = metriclist.wcache.f;
   return val;
}

g_val_t
cpu_aidle_func(void)
{
   g_val_t val;
   val.f = 0;
   return val;
}
