/*
 *
 * gnome-scancontext.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include "gnomescancontext.h"
#include "gnomescanutils.h"
#include "gnomescanner.h"


#define	GNOME_SCAN_CONTEXT_PARENT_CLASS(klass)	(G_OBJECT_CLASS (g_type_class_peek_parent ((klass))))
#define GNOME_SCAN_CONTEXT_ERROR		(g_type_qname (GNOME_TYPE_SCAN_CONTEXT))
#define GET_PRIVATE(obj)			(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_CONTEXT, GnomeScanContextPrivate))

typedef struct _GnomeScanContextPrivate		GnomeScanContextPrivate;

struct _GnomeScanContextPrivate {
  gboolean		dispose_has_run;
  GError		*error;

  /* PROPERTIES */
  GnomeScanBackend	*backend;
  GSList		*scanners;
  GnomeScanContextInfo	*info;
  GdkPixbufRotation	rotation;
};

enum {
  PROP_0,
  PROP_BACKEND,
  PROP_SCANNERS,
  PROP_SCANNER,
  PROP_RESOLUTION,
  PROP_AREA,
  PROP_ROTATION
};

enum {
  CHANGED,
  PROBE_DONE,
  SCANNER_SELECTED,
  ACQUISITION_STARTED,
  DATA_RECEIVED,
  ACQUISITION_PAUSED,
  ACQUISITION_RESUMED,
  ACQUISITION_TERMINATED,
  IMAGE_RECEIVED,
  PREVIEW_STARTED,
  PREVIEW_TERMINATED,
  PREVIEW_RECEIVED,
  ERROR,
  N_SIGNALS
};

static guint signals[N_SIGNALS];

void				gsc_trigger_error				(GnomeScanContext *context,
										 GError *error);

void				gsc_acquire					(GnomeScanContext *context);

void				gsc_data_received				(GnomeScanBackend *backend,
										 gsize data_size,
										 GnomeScanContext *context);

void				gsc_image_acquired				(GnomeScanBackend *backend,
										 GnomeScanResult *result,
										 GnomeScanContext *context);

/********************************
 * 	      GOBJECT		*
 ********************************/


void				gnome_scan_context_dispose 			(GObject *obj);

void				gnome_scan_context_finalize			(GObject *obj);

void				gnome_scan_context_set_property 		(GObject *obj,
										 guint property_id,
										 const GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_context_get_property 		(GObject *obj,
										 guint property_id,
										 GValue *value,
										 GParamSpec *pspec);

G_DEFINE_TYPE (GnomeScanContext, gnome_scan_context, G_TYPE_OBJECT);

void
gnome_scan_context_class_init (GnomeScanContextClass* klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property = gnome_scan_context_set_property;
  gobject_class->get_property = gnome_scan_context_get_property;
  gobject_class->dispose = gnome_scan_context_dispose;
  gobject_class->finalize = gnome_scan_context_finalize;

  g_type_class_add_private (gobject_class,
			    sizeof (GnomeScanContextPrivate));

  /* Properties */

  g_object_class_install_property (gobject_class,
				   PROP_BACKEND,
				   g_param_spec_object ("backend",
							"Scan Backend",
							"a GnomeScanBackend",
							GNOME_TYPE_SCAN_BACKEND,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_SCANNERS,
				   g_param_spec_pointer ("scanners",
							 "Scanners",
							 "Contain scanners list.",
							 G_PARAM_READABLE));

  g_object_class_install_property (gobject_class,
				   PROP_SCANNER,
				   g_param_spec_object ("scanner",
							"Scanner",
							"The current selected scanner.",
							GNOME_TYPE_SCANNER,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
				   PROP_RESOLUTION,
				   g_param_spec_int ("resolution",
						     "Resolution",
						     "The final image X resolution.",
						     0,
						     G_MAXINT,
						     100,
						     G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
				   PROP_AREA,
				   g_param_spec_pointer ("area",
							 "Area",
							 "The final image area to be scanned.",
							 G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
				   PROP_ROTATION,
				   g_param_spec_int ("rotation",
						     "Rotation",
						     "The rotation in degree to be applied to the final image.",
						     GDK_PIXBUF_ROTATE_NONE,
						     GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE,
						     GDK_PIXBUF_ROTATE_NONE,
						     G_PARAM_READWRITE));

  /* Signals */

  /**
   * GnomeScanContext::changed:
   * @context:	The emitting #GnomeScanContext
   * @property:	The changed property name
   *
   * Emitted when a scan option has been changed.
   */
  signals[CHANGED] = g_signal_new ("changed",
				   GNOME_TYPE_SCAN_CONTEXT,
				   G_SIGNAL_RUN_FIRST,
				   G_STRUCT_OFFSET (GnomeScanContextClass, changed),
				   NULL,
				   NULL,
				   g_cclosure_marshal_VOID__STRING,
				   G_TYPE_NONE,
				   1,
				   G_TYPE_STRING);

  /**
   * GnomeScanContext::probe-done:
   * @context: The emitting #GnomeScanContext
   *
   * Emitted when the backend terminate the scanner probe.
   */
  signals[PROBE_DONE] = g_signal_new ("probe-done",
				      GNOME_TYPE_SCAN_CONTEXT,
				      G_SIGNAL_RUN_FIRST,
				      G_STRUCT_OFFSET (GnomeScanContextClass, probe_done),
				      NULL,
				      NULL,
				      g_cclosure_marshal_VOID__VOID,
				      G_TYPE_NONE,
				      0);

  /**
   * GnomeScanContext::scanner-selected:
   * @context: The emitting #GnomeScanContext
   * @scanner: The newly selected scanner
   *
   * Emitted when the current a scanner is selected (wheter or not a
   * previous scanner were selected).
   */
  signals[SCANNER_SELECTED] = g_signal_new ("scanner-selected",
					    GNOME_TYPE_SCAN_CONTEXT,
					    G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST,
					    G_STRUCT_OFFSET (GnomeScanContextClass, scanner_selected),
					    NULL,
					    NULL,
					    g_cclosure_marshal_VOID__OBJECT,
					    G_TYPE_NONE,
					    1,
					    GNOME_TYPE_SCANNER);

  /**
   * GnomeScanContext::preview-started:
   * @context: The emitting #GnomeScanContext
   * @forecast: The image size forecast.
   *
   * Emitted when a preview acquisition is started.
   * 
   */
  signals[PREVIEW_STARTED] = g_signal_new ("preview-started",
					   GNOME_TYPE_SCAN_CONTEXT,
					   G_SIGNAL_RUN_FIRST,
					   G_STRUCT_OFFSET (GnomeScanContextClass, preview_started),
					   NULL,
					   NULL,
					   g_cclosure_marshal_VOID__BOXED,
					   G_TYPE_NONE,
					   1,
					   GNOME_TYPE_SCAN_FORECAST);

  /**
   * GnomeScanContext::preview-terminated:
   * @context: The emitting #GnomeScanContext
   *
   * Emitted when a preview acquisition is terminated.
   * 
   */
  signals[PREVIEW_TERMINATED] = g_signal_new ("preview-terminated",
					      GNOME_TYPE_SCAN_CONTEXT,
					      G_SIGNAL_RUN_FIRST,
					      G_STRUCT_OFFSET (GnomeScanContextClass, preview_received),
					      NULL,
					      NULL,
					      g_cclosure_marshal_VOID__VOID,
					      G_TYPE_NONE,
					      0);

  /**
   * GnomeScanContext::preview-received:
   * @context: The emitting #GnomeScanContext
   * @pixbuf: The preview image in a #GdkPixbuf
   *
   * Emitted when a preview image is received.
   * 
   */
  signals[PREVIEW_RECEIVED] = g_signal_new ("preview-received",
					    GNOME_TYPE_SCAN_CONTEXT,
					    G_SIGNAL_RUN_FIRST,
					    G_STRUCT_OFFSET (GnomeScanContextClass, preview_received),
					    NULL,
					    NULL,
					    g_cclosure_marshal_VOID__BOXED,
					    G_TYPE_NONE,
					    1,
					    GNOME_TYPE_SCAN_RESULT);

  /**
   * GnomeScanContext::acquisition-started:
   * @context: The emitting #GnomeScanContext
   * @forecast: The image size forecast.
   *
   * Emitted when the acquisition is started. Use that signal to show
   * a popup before actualy receiving data.
   */
  signals[ACQUISITION_STARTED] = g_signal_new ("acquisition-started",
					       GNOME_TYPE_SCAN_CONTEXT,
					       G_SIGNAL_RUN_FIRST,
					       G_STRUCT_OFFSET (GnomeScanContextClass, acquisition_started),
					       NULL,
					       NULL,
					       g_cclosure_marshal_VOID__BOXED,
					       G_TYPE_NONE,
					       1,
					       GNOME_TYPE_SCAN_FORECAST);

  /**
   * GnomeScanContext::data-received:
   * @context: The emitting #GnomeScanContext
   * @data_size: The current number of bytes received.
   *
   * Emitted when the backend send data. Either for a final picture
   * acquisition or for a preview refresh.
   */
  signals[DATA_RECEIVED] = g_signal_new ("data-received",
					 GNOME_TYPE_SCAN_CONTEXT,
					 G_SIGNAL_RUN_FIRST,
					 G_STRUCT_OFFSET (GnomeScanContextClass, data_received),
					 NULL,
					 NULL,
					 g_cclosure_marshal_VOID__INT,
					 G_TYPE_NONE,
					 1,
					 G_TYPE_INT);

  /**
   * GnomeScanContext::acquisition-paused:
   * @context: The emitting #GnomeScanContext
   *
   * Emitted when the acquisition process has been paused.
   */
  signals[ACQUISITION_PAUSED] = g_signal_new ("acquisition-paused",
					      GNOME_TYPE_SCAN_CONTEXT,
					      G_SIGNAL_RUN_FIRST,
					      G_STRUCT_OFFSET (GnomeScanContextClass, acquisition_paused),
					      NULL,
					      NULL,
					      g_cclosure_marshal_VOID__VOID,
					      G_TYPE_NONE,
					      0);

  /**
   * GnomeScanContext::acquisition-resumed:
   * @context: The emitting #GnomeScanContext
   *
   * Emitted when the acquisition process has been paused.
   */
  signals[ACQUISITION_RESUMED] = g_signal_new ("acquisition-resumed",
					       GNOME_TYPE_SCAN_CONTEXT,
					       G_SIGNAL_RUN_FIRST,
					       G_STRUCT_OFFSET (GnomeScanContextClass, acquisition_resumed),
					       NULL,
					       NULL,
					       g_cclosure_marshal_VOID__VOID,
					       G_TYPE_NONE,
					       0);

  /**
   * GnomeScanContext::acquisition-terminated:
   * @context: The emitting #GnomeScanContext
   *
   * Emitted when the backend has no more data to send. This signal is
   * emitted even if the acquisition has been canceled.
   */
  signals[ACQUISITION_TERMINATED] = g_signal_new ("acquisition-terminated",
						  GNOME_TYPE_SCAN_CONTEXT,
						  G_SIGNAL_RUN_FIRST,
						  G_STRUCT_OFFSET (GnomeScanContextClass, acquisition_terminated),
						  NULL,
						  NULL,
						  g_cclosure_marshal_VOID__VOID,
						  G_TYPE_NONE,
						  0);

  /**
   * GnomeScanContext::image-received:
   * @context: The emitting #GnomeScanContext
   * @pixbuf: The final image in a #GdkPixbuf
   *
   * Emitted when a final image has been acquired. The pixbuf is then
   * sent with that signal.
   * 
   */
  signals[IMAGE_RECEIVED] = g_signal_new ("image-received",
					  GNOME_TYPE_SCAN_CONTEXT,
					  G_SIGNAL_RUN_FIRST,
					  G_STRUCT_OFFSET (GnomeScanContextClass, image_received),
					  NULL,
					  NULL,
					  g_cclosure_marshal_VOID__BOXED,
					  G_TYPE_NONE,
					  1,
					  GNOME_TYPE_SCAN_RESULT);

  /**
   * GnomeScanContext::error:
   * @context: The emitting #GnomeScanContext
   * @error: a #GError containing the backend error.
   *
   * This signal allow the application to avert he user that a backend
   * error occured.
   */
  signals[ERROR] = g_signal_new ("error",
				 GNOME_TYPE_SCAN_CONTEXT,
				 G_SIGNAL_RUN_FIRST,
				 G_STRUCT_OFFSET (GnomeScanContextClass, error),
				 NULL,
				 NULL,
				 g_cclosure_marshal_VOID__POINTER,
				 G_TYPE_NONE,
				 1,
				 G_TYPE_POINTER);
}

void
gnome_scan_context_init (GnomeScanContext *context)
{
  GnomeScanContextPrivate *priv = GET_PRIVATE (context);

  priv->scanners = NULL;
  priv->info	= g_new0 (GnomeScanContextInfo, 1);
  priv->info->area = g_new0 (GnomeScanArea, 1);
  priv->rotation = GDK_PIXBUF_ROTATE_NONE;
  priv->dispose_has_run = FALSE;
}

void
gnome_scan_context_dispose (GObject *obj)
{
  GnomeScanContext *context = GNOME_SCAN_CONTEXT (obj);
  GnomeScanContextPrivate *priv = GET_PRIVATE (context);
  GnomeScanContextClass *klass = GNOME_SCAN_CONTEXT_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  gnome_scan_backend_destroy (priv->backend);
  priv->dispose_has_run = TRUE;

  /* chain */
  GNOME_SCAN_CONTEXT_PARENT_CLASS (klass)->dispose (obj);
}

void
gnome_scan_context_finalize (GObject *obj)
{
  GnomeScanContext *context = GNOME_SCAN_CONTEXT (obj);
  GnomeScanContextPrivate *priv = GET_PRIVATE (context);
  GnomeScanContextClass *klass = GNOME_SCAN_CONTEXT_GET_CLASS (obj);

  g_slist_free (priv->scanners);
  g_free (priv->info);

  GNOME_SCAN_CONTEXT_PARENT_CLASS (klass)->finalize (obj);
}


void
gnome_scan_context_set_property (GObject *obj,
				 guint property_id,
				 const GValue *value,
				 GParamSpec *pspec)
{
  GnomeScanContextPrivate *priv = GET_PRIVATE (GNOME_SCAN_CONTEXT (obj));

  switch (property_id) {
  case PROP_BACKEND:
    priv->backend = GNOME_SCAN_BACKEND (g_value_dup_object (value));
    break;
  case PROP_SCANNERS:
    priv->scanners = g_value_get_pointer (value);
    break;
  case PROP_SCANNER:
    priv->info->scanner = g_value_get_pointer (value);
    break;
  case PROP_RESOLUTION:
    priv->info->resolution = g_value_get_double (value);
    break;
  case PROP_AREA:
    priv->info->area = g_value_get_pointer (value);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id,pspec);
    break;
  }
}

void
gnome_scan_context_get_property (GObject *obj,
				 guint property_id,
				 GValue *value,
				 GParamSpec *pspec)
{
  GnomeScanContextPrivate *priv = GET_PRIVATE (GNOME_SCAN_CONTEXT(obj));

  switch (property_id) {
  case PROP_BACKEND:
    g_value_set_object (value, priv->backend);
    break;
  case PROP_SCANNERS:
    g_value_set_pointer (value, priv->scanners);
    break;
  case PROP_SCANNER:
    g_value_set_pointer (value, priv->info->scanner);
    break;
  case PROP_RESOLUTION:
    g_value_set_double (value, priv->info->resolution);
    break;
  case PROP_AREA:
    g_value_set_pointer (value, priv->info->area);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,property_id,pspec);
    break;
  }
}




/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_context_new:
 * 
 * Create a new #GnomeScanContext
 * 
 * Return value: the new #GnomeScanContext
 */
GnomeScanContext*
gnome_scan_context_new (void)
{
  GError *error = NULL;
  GnomeScanBackend *backend = gnome_scan_backend_new (&error);
  GnomeScanContextInfo *info;

  if (error) {
    g_critical ("Failed to initialise GnomeScanBackend.");
    return NULL;
  }

  GnomeScanContext *context = g_object_new (GNOME_TYPE_SCAN_CONTEXT,
					    "backend", backend,
					    NULL);

  /* Init info */
  info = GET_PRIVATE (context)->info;
  info->depth = 8;
  info->colorspace = GNOME_SCAN_COLORSPACE_RGB;

  /* Is here the right place to do that ? */
/*   if (!g_thread_supported ()) g_thread_init (NULL); */

  g_signal_connect (backend,
		    "data-received",
		    (GCallback) gsc_data_received,
		    context);

  g_signal_connect (backend,
		    "image-acquired",
		    (GCallback) gsc_image_acquired,
		    context);

  return context;
}

/**
 * gnome_scan_context_destroy:
 * @context: a #GnomeScanContext
 * 
 * Unref a #GnomeScanContext
 */
void
gnome_scan_context_destroy (GnomeScanContext *context)
{
  g_return_if_fail (GNOME_IS_SCAN_CONTEXT (context));

  g_object_unref (context);
}

/**
 * gnome_scan_context_probe_scanners:
 * @context: a #GnomeScanContext
 * 
 * Since probe can be very long, a newly created #GnomeScanContext
 * contains an empty list of scanners. The application must call
 * gnome_scan_context_probe_scanners and wait for the signal
 * #GnomeScancontext::probe-done before query the scanners list. This
 * allow the application to show a waiting dialog.
 *
 * This function will be drop as soon as a dbus service will be implemented.
 */
void
gnome_scan_context_probe_scanners (GnomeScanContext *context)
{
  g_return_if_fail (GNOME_IS_SCAN_CONTEXT (context));
  GnomeScanContextPrivate *priv = GET_PRIVATE (context);
  GError *error = NULL;
  priv->scanners = gnome_scan_backend_probe_scanners (priv->backend, &error);

  if (error) {
    gsc_trigger_error (context, error);
  }
  
  g_signal_emit_by_name (context,
			 "probe-done");
}

/**
 * gnome_scan_context_get_scanners:
 * @context: a #GnomeScanContext
 *
 * Return the list of probed scanners. Don't forget to do a
 * gnome_scan_context_probe_scanners() before calling
 * gnome_scan_context_get_scanners().
 * 
 * Return value: a #GSList of #GnomeScanner.
 */
GSList*
gnome_scan_context_get_scanners (GnomeScanContext *context)
{
  g_return_val_if_fail (GNOME_IS_SCAN_CONTEXT (context), NULL);
  return GET_PRIVATE (context)->scanners;
}

/**
 * gnome_scan_context_get_scanner:
 * @context: a #GnomeScanContext
 * 
 * Return the current selected scanner as a #GnomeScanner.
 * 
 * Return value: the current selected scanner
 */
GnomeScanner*
gnome_scan_context_get_scanner (GnomeScanContext *context)
{
  g_return_val_if_fail (GNOME_IS_SCAN_CONTEXT (context), NULL);
  return GET_PRIVATE (context)->info->scanner;
}

/**
 * gnome_scan_context_select_scanner:
 * @context: a #GnomeScanContext
 * @scanner: a #GnomeScanner
 *
 * Select a scanner for scan. Use a scanner pointed in the scanners
 * list. Beware, no check the scanner were in the scanners list is done.
 *
 * @see_also: gnome_scan_context_get_scanners()
 */
void
gnome_scan_context_select_scanner (GnomeScanContext *context,
				   GnomeScanner *scanner)
{
  GnomeScanContextPrivate *priv;
  GValueArray *sources;
  GValue *source;

  g_return_if_fail (GNOME_IS_SCAN_CONTEXT (context));
  g_return_if_fail (GNOME_IS_SCANNER (scanner));

  priv = GET_PRIVATE (context);

  /* adapt info to new scanner. */
  priv->info->scanner = scanner;
  sources = gnome_scanner_get_sources (scanner);
  source = g_value_array_get_nth (sources, 0);
  priv->info->source = g_value_get_enum (source);

  g_debug ("Scanner %s selected", gnome_scanner_get_product (scanner));

  g_signal_emit_by_name (context,
			 "scanner-selected",
			 scanner);
}

/**
 * gnome_scan_context_set_resolution:
 * @context: a #GnomeScanContext
 * @resolution: the new resolution
 * 
 * Set the resolution of the scan. Beware of fiting the range of
 * allowed value.
 */
void
gnome_scan_context_set_resolution (GnomeScanContext *context,
				   gdouble resolution)
{
  g_return_if_fail (GNOME_IS_SCAN_CONTEXT (context));
  GET_PRIVATE (context)->info->resolution = resolution;
  g_signal_emit_by_name (context, "changed", "resolution");
}

/**
 * gnome_scan_context_get_resolution:
 * @context: a #GnomeScanContext
 * 
 * Retreive the current resolution the image will be scan on.
 */
void
gnome_scan_context_get_resolution (GnomeScanContext *context,
				   gdouble *resolution)
{
  g_return_if_fail (GNOME_IS_SCAN_CONTEXT (context));
  *resolution = GET_PRIVATE (context)->info->resolution;
}

/**
 * gnome_scan_context_set_area:
 * @context: a #GnomeScanContext
 * @area: a #GnomeScanArea
 * 
 * Set the scan area of the picture. Using a scan area avoid people to
 * use Gimp or such tools to crop picture and save some time during
 * acquisition. This function is usually called by Preview
 * widgets. Beware, the value aren't checked against the image geometrys.
 */
void
gnome_scan_context_set_area (GnomeScanContext *context,
			     GnomeScanArea *area)
{
  g_return_if_fail (GNOME_IS_SCAN_CONTEXT (context));
  GnomeScanContextPrivate *priv = GET_PRIVATE (context);
  g_free (priv->info->area);
  priv->info->area = g_memdup (area, sizeof (GnomeScanArea));
  g_debug ("Area set to %f×%f+%f+%f (mm)", area->width, area->height, area->x, area->y);
  g_signal_emit_by_name (context, "changed", "area");
}

/**
 * gnome_scan_context_get_area:
 * @context: a #GnomeScanContext
 * 
 * Get the current area required. See gnome_scan_context_set_area()
 * for further informations. The returned #GnomeScanArea is a copy of
 * the area and should be freed using g_free().
 * 
 * Return value: the current #GnomeScanArea
 */
GnomeScanArea*
gnome_scan_context_get_area (GnomeScanContext *context)
{
  g_return_val_if_fail (GNOME_IS_SCAN_CONTEXT (context), NULL);
  return g_memdup (GET_PRIVATE (context)->info->area, sizeof (GnomeScanArea));
}

/**
 * gnome_scan_context_set_rotation:
 * @context: a #GnomeScanContext 
 * @rotation: the rotation.
 * 
 * Set the final image rotation using a value in #GdkPixbufRotation.
 */
void
gnome_scan_context_set_rotation (GnomeScanContext *context,
				 GdkPixbufRotation rotation)
{
  g_return_if_fail (GNOME_IS_SCAN_CONTEXT (context));
  GET_PRIVATE (context)->rotation = rotation;
  g_signal_emit_by_name (context, "changed", "rotation");
}

/**
 * gnome_scan_context_get_rotation:
 * @context: a #GnomeScanContext
 * 
 * Retreive the current final image rotation.
 * 
 * Return value: the image rotation
 **/
GdkPixbufRotation
gnome_scan_context_get_rotation (GnomeScanContext *context)
{
  g_return_val_if_fail (GNOME_IS_SCAN_CONTEXT (context), GDK_PIXBUF_ROTATE_NONE);
  return GET_PRIVATE (context)->rotation;
}

/**
 * gnome_scan_context_set_source:
 * @context:	a #GonmeScanContext
 * @source: 	the source to select
 * 
 * Change current selected source.
 **/
void
gnome_scan_context_set_source (GnomeScanContext *context,
			       GnomeScannerSource source)
{
  GET_PRIVATE (context)->info->source = source;
  g_signal_emit_by_name (context, "changed", "source");
}

/**
 * gnome_scan_context_get_source:
 * @context: a #GnomeScanContext
 * 
 * Get the selected scanner source.
 * 
 * Return value: the source
 */
GnomeScannerSource
gnome_scan_context_get_source (GnomeScanContext *context)
{
  return GET_PRIVATE (context)->info->source;
}

/**
 * gnome_scan_context_get_colorspace:
 * @context: a #GnomeScanContext
 * 
 * Get the colorspace
 * 
 * Returns: The selected colorspace
 **/
GnomeScanColorspace
gnome_scan_context_get_colorspace (GnomeScanContext *context)
{
  return GET_PRIVATE (context)->info->colorspace;
}

/**
 * gnome_scan_context_set_colorspace:
 * @context:		a #GnomeScanContext 
 * @colorspace: 	a #GnomeScanColorspace
 * 
 * Choose the colorspace of the final image.
 **/
void
gnome_scan_context_set_colorspace (GnomeScanContext *context,
				   GnomeScanColorspace colorspace)
{
  GET_PRIVATE (context)->info->colorspace = colorspace;
  g_signal_emit_by_name (context, "changed", "colorspace");
}

/**
 * gnome_scan_context_get_depth:
 * @context:	a #GnomeScanContext 
 * 
 * Get the final image depth
 * 
 * Returns: an integer
 **/
gint
gnome_scan_context_get_depth (GnomeScanContext *context)
{
  return GET_PRIVATE (context)->info->depth;
}

/**
 * gnome_scan_context_set_depth:
 * @context:	a #GnomeScanContext 
 * @depth: 	a depth
 * 
 * Set the image depth (e.g. bit per sample) of the final
 * image. Currently, only 8bit depth is allow due to GdkPixbuf
 * Limitation.
 **/
void
gnome_scan_context_set_depth (GnomeScanContext *context,
			      gint depth)
{
  g_return_if_fail (depth == 8);
  GET_PRIVATE (context)->info->depth = depth;
  g_signal_emit_by_name (context, "changed", "depth");
}



/**
 * gnome_scan_context_acquire_preview
 * @context: a #GnomeScanContext
 *
 * This function trigger a acquisition using full area and given
 * resolution, without changing the selection area and resolution.
 * 
 */
void
gnome_scan_context_acquire_preview (GnomeScanContext *context)
{
  GnomeScanner *scanner;
  GnomeScannerOption *option;
  GnomeScanGeometry *geometry;
  GnomeScanContextPrivate *priv;
  GnomeScanContextInfo *info;
  GnomeScanForecast *fc;
  GnomeScanResult *result;
  GError *error = NULL;

  g_return_if_fail (GNOME_IS_SCAN_CONTEXT (context));

  priv = GET_PRIVATE (context);
  scanner = priv->info->scanner;

  /* We create a temp ContextInfo suitable for preview. */
  /* resolution */
  info = g_memdup (priv->info, sizeof (GnomeScanContextInfo));
  info->resolution = 50;
  info->preview = TRUE;

  /* area */
  geometry = gnome_scanner_get_geometry (scanner);
  info->area = g_new0 (GnomeScanArea, 1);
  info->area->width = geometry->width;
  info->area->height = geometry->height;

  fc =  gnome_scan_backend_forecast (GET_PRIVATE (context)->backend, info, &error);
  if (error) gsc_trigger_error (context, error);

  g_signal_emit_by_name (context, "preview-started", fc); while (gtk_events_pending ())  gtk_main_iteration ();
  gnome_scan_backend_acquire (priv->backend, info, &error);
  g_signal_emit_by_name (context, "preview-terminated"); while (gtk_events_pending ())  gtk_main_iteration ();
}

/**
 * gnome_scan_context_forecast:
 * @context: a #GnomeScanContext
 * 
 * Forecast image size and geometry with current resolution and area.
 * 
 * Returns: a #GnomeScanForecast
 **/
GnomeScanForecast*
gnome_scan_context_forecast (GnomeScanContext *context)
{
  g_return_val_if_fail (GNOME_IS_SCAN_CONTEXT (context), NULL);
  GError *error = NULL;
  GnomeScanForecast *fc;

  fc =  gnome_scan_backend_forecast (GET_PRIVATE (context)->backend, GET_PRIVATE (context)->info, &error);
  if (error)
    gsc_trigger_error (context, error);

  return fc;
}

/**
 * gnome_scan_context_start_acquisition:
 * @context: a #GnomeScanContext
 * 
 * Acquire a picture. Cancel an acquisition is possible using
 * gnome_scan_context_stop_acquisition(). The resulted pixbuf is
 * sent using the #GnomeScanContext::"image-received" signal.
 */
void
gnome_scan_context_start_acquisition (GnomeScanContext *context)
{
  g_signal_emit_by_name (context, "acquisition-started", gnome_scan_context_forecast (context));
  while (gtk_events_pending ()) gtk_main_iteration ();
  gsc_acquire (context);
}

/**
 * gnome_scan_context_resume_acquisition:
 * @context: a #GnomeScanContext
 *
 * Continue an acquisition for multipage document.
 */
void
gnome_scan_context_resume_acquisition (GnomeScanContext *context)
{
  g_signal_emit_by_name (context, "acquisition-resumed");
  gsc_acquire (context);
}

/**
 * gnome_scan_context_stop_acquisition:
 * @context: a #GnomeScanContext
 * 
 * Stop an atomic multiple acquisition. If an image is being aquired,
 * the image acquisition is canceled.
 **/
void
gnome_scan_context_stop_acquisition (GnomeScanContext *context)
{
  g_return_if_fail (GNOME_IS_SCAN_CONTEXT (context));
  gnome_scan_backend_stop (GET_PRIVATE (context)->backend, GET_PRIVATE (context)->info->scanner);
  g_signal_emit_by_name (context, "acquisition-terminated");
  while (gtk_events_pending ())  gtk_main_iteration ();
}




/********************************
 * 	      OTHER		*
 ********************************/


void
gsc_trigger_error (GnomeScanContext *context,
		   GError *error)
{
  GET_PRIVATE (context)->error = error;
  g_signal_emit_by_name (context, "error", error);
}

void
gsc_acquire (GnomeScanContext *context)
{
  GnomeScanContextPrivate *priv = GET_PRIVATE (context);
  GError *error = NULL;
  gnome_scan_backend_acquire (priv->backend, priv->info, &error);
  g_signal_emit_by_name (context, "acquisition_paused");
}

/* This function just propagate data-received backend signal. */
void
gsc_data_received (GnomeScanBackend *backend,
		   gsize data_size,
		   GnomeScanContext *context)
{
  while (gtk_events_pending ())
    gtk_main_iteration ();

  g_signal_emit_by_name (context, "data-received", data_size);
}

void
gsc_image_acquired (GnomeScanBackend *backend,
		    GnomeScanResult *result,
		    GnomeScanContext *context)
{
  GnomeScanContextPrivate *priv = GET_PRIVATE (context);
  GdkPixbuf *image;

  if (result->preview) {
    g_signal_emit_by_name (context, "preview-received", result);
  }
  else {
    /* apply transformations */
    if (priv->rotation) {
      result = g_boxed_copy (GNOME_TYPE_SCAN_RESULT, result);
      image = result->image;
      result->image = gdk_pixbuf_rotate_simple (image, priv->rotation);
      g_debug ("%s: Unref image %p", __FUNCTION__, image);
      g_object_unref (image);
    }
    g_signal_emit_by_name (context, "image-received", result);
  }
}
