/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescanareaselector.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#define GNOME_SCAN_AREA_SELECTOR_ERROR			(g_type_qname (GNOME_TYPE_SCAN_AREA_SELECTOR))
#define	GNOME_SCAN_AREA_SELECTOR_PARENT_CLASS(klass)	(GTK_WIDGET_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)				(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_AREA_SELECTOR, GnomeScanAreaSelectorPrivate))

typedef struct _GnomeScanAreaSelectorPrivate		GnomeScanAreaSelectorPrivate;

struct _GnomeScanAreaSelectorPrivate {
  gboolean		dispose_has_run;
  GtkWidget		*combo;
  GtkWidget*		portrait;
  GtkWidget*		landscape;
  /* current selection cache */
  GtkPageOrientation	orientation;
  GtkPaperSize		*papersize;
};

enum {
  COLUMN_NAME,
  COLUMN_PAPERSIZE,
  COLUMN_SEPARATOR,
  N_COLUMNS
};

enum {
  /*   AREA_MAX, */
  AREA_CUSTOM,
  /*   AREA_AUTO, */
  N_PREDEF_AREAS
};

/********************************
 * 	      CALLBACKS		*
 ********************************/

void				gsas_portrait						(GtkToggleButton *button,
											 GnomeScanAreaSelector *gsas);

void				gsas_landscape						(GtkToggleButton *button,
											 GnomeScanAreaSelector *gsas);

void				gsas_combo_changed					(GtkComboBox *combo,
											 GnomeScanAreaSelector *gsas);

void				gsas_scanner_selected					(GnomeScanContext *context,
											 GnomeScanner *scanner,
											 GnomeScanAreaSelector *gsas);

/********************************
 * 	       OTHERS		*
 ********************************/

gboolean			gsas_row_separator					(GtkTreeModel *model,
											 GtkTreeIter *iter,
											 gpointer data);

void				gsas_formats_foreach					(gchar *format,
											 GnomeScanAreaSelector *gsas);

void				gsas_setup_context					(GnomeScanAreaSelector *gsas);

/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scan_area_selector_set_property 			(GObject *obj,
											 guint property_id,
											 const GValue *value,
											 GParamSpec *pspec);

void				gnome_scan_area_selector_get_property 			(GObject *obj,
											 guint property_id,
											 GValue *value,
											 GParamSpec *pspec);

void				gnome_scan_area_selector_finalize			(GObject *obj);

void				gnome_scan_area_selector_dispose			(GObject *obj);


enum {
  PROP_0,
  PROP_CONTEXT
};

enum {
  N_SIGNALS
};

static guint signals[N_SIGNALS];


G_DEFINE_TYPE (GnomeScanAreaSelector, gnome_scan_area_selector, GTK_TYPE_VBOX);


void
gnome_scan_area_selector_class_init (GnomeScanAreaSelectorClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property	= gnome_scan_area_selector_set_property;
  gobject_class->get_property	= gnome_scan_area_selector_get_property;
  gobject_class->dispose 	= gnome_scan_area_selector_dispose;
  /*   gobject_class->finalize 	= gnome_scan_area_selector_finalize; */

  g_type_class_add_private (gobject_class,
			    sizeof (GnomeScanAreaSelectorPrivate));

  klass->formats = NULL;
  /* add all stocks */
  klass->formats = g_slist_append (klass->formats, GTK_PAPER_NAME_A4);
  klass->formats = g_slist_append (klass->formats, GTK_PAPER_NAME_A5);
  klass->formats = g_slist_append (klass->formats, GTK_PAPER_NAME_B5);
  klass->formats = g_slist_append (klass->formats, GTK_PAPER_NAME_LETTER);
  klass->formats = g_slist_append (klass->formats, GTK_PAPER_NAME_EXECUTIVE);
  klass->formats = g_slist_append (klass->formats, GTK_PAPER_NAME_LEGAL);
  klass->formats = g_slist_append (klass->formats, "iso_dl");
  klass->formats = g_slist_append (klass->formats, "om_small-photo");
  /* TODO: add useful stock paper sizes. */

  /* Properties */
  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"The GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));

  /* Signals */
}

void
gnome_scan_area_selector_init (GnomeScanAreaSelector *gsas)
{
  GnomeScanAreaSelectorPrivate *priv = GET_PRIVATE (gsas);

  gsas->context = NULL;
  priv->dispose_has_run = FALSE;
  priv->combo		= NULL;
  priv->landscape	= NULL;
  priv->portrait	= NULL;
  priv->orientation	= GTK_PAGE_ORIENTATION_PORTRAIT;
}

void
gnome_scan_area_selector_dispose (GObject *obj)
{
  GnomeScanAreaSelector *gsas = GNOME_SCAN_AREA_SELECTOR (obj);
  GnomeScanAreaSelectorPrivate *priv = GET_PRIVATE (gsas);
  GnomeScanAreaSelectorClass *b_klass = GNOME_SCAN_AREA_SELECTOR_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref devices */
  g_object_unref (gsas->context);
  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_AREA_SELECTOR_PARENT_CLASS (klass)->dispose (obj); */
}

void
gnome_scan_area_selector_finalize (GObject *obj)
{
  GnomeScanAreaSelector *gsas = GNOME_SCAN_AREA_SELECTOR (obj);
  GnomeScanAreaSelectorClass *klass = GNOME_SCAN_AREA_SELECTOR_GET_CLASS (gsas);
  
  /*   GNOME_SCAN_AREA_SELECTOR_PARENT_CLASS (klass)->finalize (obj); */
}

void
gnome_scan_area_selector_set_property (GObject *obj,
				       guint property_id,
				       const GValue *value,
				       GParamSpec *pspec)
{
  GnomeScanAreaSelector *gsas = GNOME_SCAN_AREA_SELECTOR (obj);
  GnomeScanAreaSelectorPrivate *priv = GET_PRIVATE (gsas);

  switch (property_id) {
  case PROP_CONTEXT:
    gsas->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}

void
gnome_scan_area_selector_get_property (GObject *obj,
				       guint property_id,
				       GValue *value,
				       GParamSpec *pspec)
{
  GnomeScanAreaSelector *gsas = GNOME_SCAN_AREA_SELECTOR (obj);
  GnomeScanAreaSelectorPrivate *priv = GET_PRIVATE (gsas);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, gsas->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}




/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_area_selector_new:
 * @context: a #GnomeScanContext
 * 
 * Create a new #GnomeScanAreaSelector connected to @context.
 * 
 * Return value: a new #GnomeScanAreaSelector
 **/
GtkWidget*
gnome_scan_area_selector_new (GnomeScanContext *context)
{	
  GnomeScanAreaSelector *gsas;
  GnomeScanAreaSelectorPrivate *priv;
  GtkWidget *widget, *radio, *icon;
  GtkListStore *store;
  GtkCellRenderer *renderer;
  GtkTreeIter iter;
  GSList *stock_list;
  gint i;
  gchar *name;
  GtkStockItem item;

  widget = GTK_WIDGET (g_object_new (GNOME_TYPE_SCAN_AREA_SELECTOR,
				     "context", context,
				     NULL));

  gsas = GNOME_SCAN_AREA_SELECTOR (widget);
  priv = GET_PRIVATE (gsas);

  gtk_box_set_spacing (GTK_BOX (gsas),
		       6);

  /* COMBO */
  store = gtk_list_store_new (N_COLUMNS,
			      G_TYPE_STRING,
			      G_TYPE_POINTER,
			      G_TYPE_BOOLEAN);

  priv->combo = gtk_combo_box_new_with_model (GTK_TREE_MODEL (store));
  gtk_combo_box_set_row_separator_func (GTK_COMBO_BOX (priv->combo), gsas_row_separator, NULL, NULL);

  gtk_box_pack_start (GTK_BOX (widget), 
		      priv->combo,
		      FALSE, TRUE,
		      0);

  /* label renderer */
  renderer = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (priv->combo), renderer, TRUE);
  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (priv->combo),
				 renderer,
				 "markup", COLUMN_NAME);

  g_signal_connect (GTK_COMBO_BOX (priv->combo),
		    "changed",
		    (GCallback) gsas_combo_changed,
		    gsas);

  /* portrait */
  priv->portrait = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (gsas),
		      priv->portrait,
		      TRUE,
		      TRUE,
		      0);
  gtk_stock_lookup (GTK_STOCK_ORIENTATION_PORTRAIT,
		    &item);
  radio = gtk_radio_button_new_with_label (NULL,
					   item.label);
  g_signal_connect (GTK_RADIO_BUTTON (radio),
		    "toggled",
		    (GCallback) gsas_portrait,
		    gsas);
  gtk_box_pack_start (GTK_BOX (priv->portrait),
		      radio,
		      TRUE,
		      TRUE,
		      0);
  icon = gtk_image_new_from_stock (GTK_STOCK_ORIENTATION_PORTRAIT,
				   GTK_ICON_SIZE_SMALL_TOOLBAR);
  gtk_box_pack_start (GTK_BOX (priv->portrait),
		      icon,
		      FALSE,
		      TRUE,
		      0);
  
  

  /* landscape */
  priv->landscape = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (gsas),
		      priv->landscape,
		      TRUE,
		      TRUE,
		      0);
  gtk_stock_lookup (GTK_STOCK_ORIENTATION_LANDSCAPE,
		    &item);
  radio = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (radio),
						       item.label);
  g_signal_connect (GTK_RADIO_BUTTON (radio),
		    "toggled",
		    (GCallback) gsas_landscape,
		    gsas);

  gtk_box_pack_start (GTK_BOX (priv->landscape),
		      radio,
		      TRUE,
		      TRUE,
		      0);
  icon = gtk_image_new_from_stock (GTK_STOCK_ORIENTATION_LANDSCAPE,
				   GTK_ICON_SIZE_SMALL_TOOLBAR);
  gtk_box_pack_start (GTK_BOX (priv->landscape),
		      icon,
		      FALSE,
		      TRUE,
		      0);


  g_signal_connect (context,
		    "scanner-selected",
		    (GCallback) gsas_scanner_selected,
		    gsas);

  return widget;
}


/**
 * gnome_scan_area_selector_add_stock:
 * @gsas: 
 * @stock_id: 
 * 
 * 
 **/
void
gnome_scan_area_selector_add_stock (GnomeScanAreaSelector *gsas,
				    gchar *stock_id)
{
  GnomeScanAreaSelectorPrivate *priv = GET_PRIVATE (gsas);
  GtkListStore *store;
  GtkTreeIter iter;
  GtkPaperSize *papersize;

  if (papersize = gtk_paper_size_new (stock_id)) {
    store = GTK_LIST_STORE (gtk_combo_box_get_model (GTK_COMBO_BOX (priv->combo)));

    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter,
			COLUMN_NAME, gtk_paper_size_get_display_name (papersize),
			COLUMN_PAPERSIZE, papersize,
			-1);
  }
}

/********************************
 * 	      CALLBACKS		*
 ********************************/

void
gsas_portrait (GtkToggleButton *button,
	       GnomeScanAreaSelector *gsas)
{
  if (gtk_toggle_button_get_active (button)) {
    GET_PRIVATE (gsas)->orientation = GTK_PAGE_ORIENTATION_PORTRAIT;
    gsas_setup_context (gsas);
  }
}

/* resize and recenter */
void
gsas_landscape (GtkToggleButton *button,
		GnomeScanAreaSelector *gsas)
{
  if (gtk_toggle_button_get_active (button)) {
    GET_PRIVATE (gsas)->orientation = GTK_PAGE_ORIENTATION_LANDSCAPE;
    gsas_setup_context (gsas);
  }
}

void
gsas_combo_changed (GtkComboBox *combo,
		    GnomeScanAreaSelector *gsas)
{
  GnomeScanAreaSelectorPrivate *priv = GET_PRIVATE (gsas);
  guint active = gtk_combo_box_get_active (combo);
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *path;
  GValue *value;
  GtkPaperSize *papersize;

  if (active != AREA_CUSTOM) {
    gtk_widget_set_sensitive (priv->landscape, TRUE);
    gtk_widget_set_sensitive (priv->portrait, TRUE);

    /* get selected papersize */
    model = gtk_combo_box_get_model (combo);
    path = gtk_tree_path_new_from_indices (active, -1);
    gtk_tree_model_get_iter (model, &iter, path);
    value = g_new0 (GValue, 1);
    gtk_tree_model_get_value (model, &iter, COLUMN_PAPERSIZE, value);
    priv->papersize = g_value_get_pointer (value);

    /* trigger context setup */
    gsas_setup_context (gsas);
  }
  else {
    /*     case AREA_MAX: */
    /*       scanner = gnome_scan_context_get_scanner (gsas->context); */
    /*       area = gnome_scan_context_get_area (gsas->context); */
    /*       geometry = gnome_scanner_get_geometry (scanner); */

    /*       area->x = 0.; */
    /*       area->y = 0.; */
    /*       area->width = geometry->width; */
    /*       area->height = geometry->height; */

    /*       gnome_scan_context_set_area (gsas->context, */
    /* 				   area); */
    /*       break; */
    priv->papersize = NULL;
    gsas_setup_context (gsas);
    gtk_widget_set_sensitive (priv->landscape, FALSE);
    gtk_widget_set_sensitive (priv->portrait, FALSE);
  }    
}

void
gsas_scanner_selected (GnomeScanContext *context,
		       GnomeScanner *scanner,
		       GnomeScanAreaSelector *gsas)
{
  GnomeScanAreaSelectorClass *klass = GNOME_SCAN_AREA_SELECTOR_GET_CLASS (gsas);
  GnomeScanAreaSelectorPrivate *priv = GET_PRIVATE (gsas);
  GtkListStore *store;
  GtkTreeIter iter;

  store = GTK_LIST_STORE (gtk_combo_box_get_model (GTK_COMBO_BOX (priv->combo)));
  gtk_list_store_clear (store);

  /* add custom entry */
  gtk_list_store_append (store, &iter);
  gtk_list_store_set (store, &iter,
		      COLUMN_NAME, _("Custom size"),
		      -1);

  /* populate the store */
  g_slist_foreach (klass->formats,
		   (GFunc) gsas_formats_foreach,
		   gsas);

  gtk_combo_box_set_active (GTK_COMBO_BOX (priv->combo),
			    AREA_CUSTOM);

}


/********************************
 * 	       OTHERS		*
 ********************************/

void
gsas_formats_foreach (gchar *format,
		      GnomeScanAreaSelector *gsas)
{
  GnomeScanAreaSelectorPrivate *priv = GET_PRIVATE (gsas);
  GnomeScanner *scanner;
  GnomeScanGeometry *geometry;
  GtkPaperSize *papersize;
  GtkWidget *combo;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gdouble width, height;


  if (papersize = gtk_paper_size_new (format)) {
    scanner = gnome_scan_context_get_scanner (gsas->context);
    geometry = gnome_scanner_get_geometry (scanner);
    width = gtk_paper_size_get_width (papersize,
				      GTK_UNIT_MM);
    height = gtk_paper_size_get_height (papersize,
					GTK_UNIT_MM);

    /* test wether the scanner can receive such format */
    if ((width <= geometry->width && height <= geometry->height)
	|| (width <= geometry->height && height <= geometry->width)) {

      model = gtk_combo_box_get_model (GTK_COMBO_BOX (priv->combo));
      /* add separator if the first is alone */
      gtk_tree_model_get_iter_first (model, &iter);
      if (!gtk_tree_model_iter_next (model, &iter)) {
	gtk_list_store_append (GTK_LIST_STORE (model), &iter);
	gtk_list_store_set (GTK_LIST_STORE (model), &iter,
			    COLUMN_SEPARATOR, TRUE, -1);
      }

      /* add format */
      gtk_list_store_append (GTK_LIST_STORE (model), &iter);
      gtk_list_store_set (GTK_LIST_STORE (model), &iter,
			  COLUMN_NAME, gtk_paper_size_get_display_name (papersize),
			  COLUMN_PAPERSIZE, papersize,
			  -1);
    }
  }
}

gboolean
gsas_row_separator (GtkTreeModel *model,
		    GtkTreeIter *iter,
		    gpointer data)
{
  gboolean boolean;
  gtk_tree_model_get (model, iter, COLUMN_SEPARATOR, &boolean, -1);
  return boolean;
}

void
gsas_setup_context (GnomeScanAreaSelector *gsas)
{
  GnomeScanAreaSelectorPrivate *priv = GET_PRIVATE (gsas);
  GnomeScanner *scanner;
  GnomeScanGeometry *geometry;
  GnomeScanArea *area, *oriented_area;
  GdkPixbufRotation rotation = GDK_PIXBUF_ROTATE_NONE;
  gdouble tmp, cx, cy, width, height;

  area = gnome_scan_context_get_area (gsas->context);
  oriented_area = g_memdup (area, sizeof (GnomeScanArea));
  oriented_area->custom = (priv->papersize == NULL);

  if (!oriented_area->custom) {
    scanner = gnome_scan_context_get_scanner (gsas->context);
    geometry = gnome_scanner_get_geometry (scanner);
    width = gtk_paper_size_get_width (priv->papersize, GTK_UNIT_MM);
    height = gtk_paper_size_get_height (priv->papersize, GTK_UNIT_MM);

    switch (priv->orientation) {
    case GTK_PAGE_ORIENTATION_PORTRAIT:
      oriented_area->width = width;
      oriented_area->height = height;
      break;
    case GTK_PAGE_ORIENTATION_LANDSCAPE:
      oriented_area->width = height;
      oriented_area->height = width;
      break;
    }

    /* wether the oriented area doesn't fit the scanner geometry */
    if (oriented_area->width > geometry->width
	|| oriented_area->height >= geometry->height) {

      /* rotate one more time (or first time) */
      tmp = oriented_area->width;
      oriented_area->width = height;
      oriented_area->height = tmp;

      /* if orientation is landscape, user usualy rotate the page
	 clockwise, so we have to rotate the final image
	 counterclockwise to get it right  */
      rotation = priv->orientation == GTK_PAGE_ORIENTATION_LANDSCAPE ?
	GDK_PIXBUF_ROTATE_COUNTERCLOCKWISE : GDK_PIXBUF_ROTATE_CLOCKWISE;
    }

    /* keep the area center */
    cx = area->x +  (area->width / 2.);
    cy = area->y +  (area->height / 2.);
    oriented_area->x = cx - (oriented_area->width / 2.);
    oriented_area->y = cy - (oriented_area->width / 2.);

    gnome_scan_context_set_rotation (gsas->context,
				     rotation);
  }

  gnome_scan_context_set_area (gsas->context,
			       oriented_area);
}

