//  Gnomoradio - roboradio/recommendation.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version)
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "roboradio/recommendation.h"
#include "roboradio/song.h"
#include "roboradio/song-rainbow.h"
#include "roboradio/init.h"

#include <libxml++/libxml++.h>
#include <sstream>

using namespace Roboradio;
using namespace Rainbow;
using namespace Glib;
using namespace std;

// FIXME: you've got to watch when modifying the internal rec_songs.  You have to keep upcoming_ref up to date.  Worse yet, state.cc directly modifies this and has to do the same.  This interface could be fixed by deriving this from SongList or making rec_songs itself a SongList.

Roboradio::Recommendation::Recommendation (const ustring &uid, bool dont_fetch)
	: user_id(uid),
	  http(0)
{
	if (!dont_fetch)
		fetch_recommendations_if_necessary();
}

Roboradio::Recommendation::~Recommendation ()
{
	for (deque<SongRef>::iterator s = rec_songs.begin(); s != rec_songs.end(); ++s)
		(*s)->upcoming_unref();
}

SongRef Roboradio::Recommendation::get_next ()
{
	if (rec_songs.begin() == rec_songs.end()) {
		fetch_recommendations_if_necessary(); // it seems to be very necessary
		return SongRef();
	}

	SongRef ret(*rec_songs.begin());
	ret->upcoming_unref();
	rec_songs.pop_front();
	fetch_recommendations_if_necessary();
	return ret;
}

void Roboradio::Recommendation::fetch_recommendations_if_necessary ()
{
	if (rec_songs.size() <= 2
	    && &Roboradio::Init::get_rainbow())
		fetch_recommendations();
}

void Roboradio::Recommendation::fetch_recommendations ()
{
	if (http)
		return; // already fetching

	xmlpp::Document tree;
	tree.create_root_node("recommendation_request");
	xmlpp::Node *root = tree.get_root_node();

	// fixme: send assigned id
	if (0 && user_id.size())
		root->add_child("user")->set_attribute("id", user_id);

	// compile a list of the most recently listened to rainbow songs
	// fixme: for now we are just going to use all the songs
	//        since the songs aren't just sitting around sorted by last_played anywhere
	//        (except a few in the history list maybe)
	xmlpp::Node *songs = root->add_child("songs");
	vector<SongRef> known_songs = Song::get_known_songs();
	int cnt = 0;
	for (vector<SongRef>::iterator i = known_songs.begin(); i != known_songs.end(); ++i) {
		if (dynamic_cast<SongRainbow*>(&**i)) {
			SongRef s(*i);
			xmlpp::Element *song = songs->add_child("song");
			song->set_attribute("url", s->get_url());
			
			ostringstream user_rating, times_played;
			user_rating << s->get_rating();
			times_played << s->times_played();
			song->set_attribute("user_rating", user_rating.str());
			song->set_attribute("times_played", times_played.str());

			++cnt;
		}
	}

	http = new HttpClient("recommend.gnomoradio.org");
	http->signal_request_done.connect(sigc::mem_fun(*this, &Recommendation::on_recommendations_fetched));
	http->post("/recommend.php", tree.write_to_string());
}

void Roboradio::Recommendation::on_recommendations_fetched (bool success)
{
	if (success)
		parse_recommendations();

	delete http;
	http = 0;

	if (success)
		signal_new_recommendations();
}

void Roboradio::Recommendation::parse_recommendations ()
{
	xmlpp::DomParser tree;
	try {
		tree.parse_memory(http->get_buffer());
	} catch (...) {
		return;
	}
	xmlpp::Node *root = tree.get_document()->get_root_node();
	if (root->get_name() != "recommendation")
		return;
	xmlpp::Node::NodeList base = root->get_children();
	for (xmlpp::Node::NodeList::iterator i = base.begin(); i != base.end(); ++i) {
		xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*i);
		if (!el)
			continue;

		if (el->get_name() == "user") {
			if (user_id.size() == 0) {
				xmlpp::Attribute *id = el->get_attribute("id");
				if (id)
					user_id = id->get_value();
			}
		} else if (el->get_name() == "songs") {
			xmlpp::Node::NodeList songs = el->get_children();
			for (xmlpp::Node::NodeList::iterator j = songs.begin(); j != songs.end(); ++j) {
				xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*j);
				if (!el)
					continue;

				xmlpp::Attribute *url = el->get_attribute("url");
				if (!url)
					continue;
				
				SongRef s(url->get_value(), false, true);
				rec_songs.push_back(s);
				s->upcoming_ref();

				xmlpp::Node::NodeList tags = el->get_children();
				for (xmlpp::Node::NodeList::iterator tag = tags.begin(); tag != tags.end(); ++tag) {
					xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*tag);
					if (!el)
						continue;
					
					if (s->get_info(el->get_name()) == "") {
						xmlpp::TextNode *val = el->get_child_text();
						if (val)
							s->set_info(el->get_name(), val->get_content());
					}
				}
			}
		}
	}
}
