// -*- c++ -*-

//  Gnomoradio - roboradio/song-list.h
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __ROBORADIO_SONG_LIST_H
#define __ROBORADIO_SONG_LIST_H

#include <glibmm.h>
#include "roboradio/song.h"

namespace Roboradio
{
	class SongListMutable;

	class SongList : public sigc::trackable
	{
	private:
		class Element;
	public:
		SongList (const Glib::ustring &n);
		virtual ~SongList ();

		void ref () { ++refcnt; }
		void unref () { if (--refcnt == 0) delete this; }

		virtual bool destroyable ();
		void destroy ();

		virtual SongListMutable *get_mutable ();

		const Glib::ustring get_name () const { return name; }
		void set_name (const Glib::ustring &n);

		int size () const { return count; }

		Time calculate_total_time () const;

		class common_iterator
		{
		public:
			common_iterator () {}
			bool operator == (const common_iterator &x) const { return node == x.node; }
			bool operator != (const common_iterator &x) const { return node != x.node; }
			SongRef& operator * () const { return *node; }
			SongRef* operator -> () const { return node; }
		protected:
			friend class SongList;
			common_iterator (Element *el) : node(el) {}
			Element *node;
		};

		class reverse_iterator;

		class iterator : public common_iterator
		{
		public:
			iterator () {}
			iterator& operator -- () { node = node->prev; return *this; }
			iterator& operator ++ () { node = node->next; return *this; }
			void operator = (const reverse_iterator &x) { node = x.node; }
			friend class reverse_iterator;
		private:
			friend class SongList;
			iterator (Element *el) : common_iterator(el) {}
		};
	       
		class reverse_iterator : public common_iterator
		{
		public:
			reverse_iterator () {}
			reverse_iterator& operator -- () { node = node->next; return *this; }
			reverse_iterator& operator ++ () { node = node->prev; return *this; }
			void operator = (const iterator &x) { node = x.node; }
			friend class iterator;
		private:
			friend class SongList;
			reverse_iterator (Element *el) : common_iterator(el) {}
		};
	       
		iterator begin () const { return iterator(list_begin); }
		iterator end () const { return iterator(0); }
		reverse_iterator rbegin () const { return reverse_iterator(list_last); }
		reverse_iterator rend () const { return reverse_iterator(0); }

		iterator get_current_song () { return current_song; }

		void play (iterator start);
		void stop ();
		void transfer_play (ref_ptr<SongList> new_list, // this should be passed by reference, but it kills g++ 3
				    iterator new_song,
				    bool restart_paused_song = true);
		Playback get_playback () const;

		void prev ();
		void next ();

		void set_repeat (bool r);
		bool get_repeat () const { return repeat; }
		bool get_repeatable () const { return repeatable; }
		void set_shuffle (bool s);
		bool get_shuffle () const { return &*shuffle_list; }
		bool get_shufflable () const { return shufflable && (get_name() != ""); }
		ref_ptr<SongList> get_shuffle_list () const { return shuffle_list; }

		sigc::signal<void,iterator> signal_inserted;
		sigc::signal<void,iterator> signal_removed;
		sigc::signal<void,iterator> signal_moved;
		sigc::signal<void,iterator> signal_song_info_changed;
		sigc::signal<void,iterator,unsigned int> signal_song_import_progress;

		sigc::signal<void,bool> signal_repeat_changed;
		sigc::signal<void,bool> signal_shuffle_changed;
		sigc::signal<void,Glib::ustring> signal_name_changed;

		sigc::signal<void> signal_current_song_changed;
		sigc::signal<void> signal_destroyed;
		sigc::signal<void> signal_done;

		static std::vector<ref_ptr<SongList> > get_named_song_lists ();

		static sigc::signal<void,ref_ptr<SongList> > signal_new_named_song_list;
		static sigc::signal<void,ref_ptr<SongList> > signal_named_song_list_destroyed;
		static sigc::signal<void,ref_ptr<SongList> > signal_global_name_changed;

	protected:
		iterator insert (iterator pos, const SongRef &value);
		void move (iterator old_pos, iterator new_pos);
		iterator remove (iterator pos);
		void clear ();
		void clear_except_playing ();
		void pop_front ();
		void pop_back ();
		void push_front (const SongRef &value);
		void push_back (const SongRef &value);

		iterator current_song;

		bool repeatable, shufflable;
		bool do_upcoming_ref;

	private:
		struct Element : public SongRef
		{
			Element *prev, *next;
			bool u;

			Element (const SongRef &ref, bool upcoming_ref) : SongRef(ref), u(upcoming_ref) { if (u) s->upcoming_ref(); }
			~Element () { if (u) s->upcoming_unref(); }
		};

		void on_song_info_changed (SongRef s);
		void on_song_import_progress (SongRef s, unsigned int p);
		
		int refcnt;
		Glib::ustring name;
		Element *list_begin, *list_last;
		int count;

		bool repeat;
		ref_ptr<SongList> shuffle_list;

		sigc::connection song_done_connection;
	};

	class SongListMutable : public SongList
	{
	public:
		SongListMutable (const Glib::ustring &n) : SongList(n) {}
		virtual ~SongListMutable ();

		using SongList::insert;
		using SongList::move;
		using SongList::remove;
		using SongList::pop_front;
		using SongList::pop_back;
		using SongList::push_front;
		using SongList::push_back;

		virtual SongListMutable *get_mutable ();
	};
}

#endif
