/*
 * Copyright (C) 2007  Miriam Ruiz <little_miry@yahoo.es>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "filter.h"
#include "taghandler.h"

#include <string>
#include <ept/debtags/tag.h>

#define FACET_VIOLENCE "rating:violence"
#define FACET_SEX "rating:sex"
#define FACET_LANGUAGE "rating:language"
#define FACET_DISCRIMINATION "rating:discrimination"

#define GREEN_MINIMUM 2

PackageFilter::PackageFilter()
{
	tagdata.SetTag(&green_tags,  "rating:violence::none");
	tagdata.SetTag(&green_tags,  "rating:violence::minor:optional");
	tagdata.SetTag(&yellow_tags, "rating:violence::minor");
	tagdata.SetTag(&yellow_tags, "rating:violence::brief");
	tagdata.SetTag(&red_tags,    "rating:violence::non-realistic:optional");
	tagdata.SetTag(&red_tags,    "rating:violence::non-realistic");
	tagdata.SetTag(&red_tags,    "rating:violence::realistic:optional");
	tagdata.SetTag(&red_tags,    "rating:violence::realistic");

	tagdata.SetTag(&green_tags,  "rating:sex::none");
	tagdata.SetTag(&green_tags,  "rating:sex::nudity:optional");
	tagdata.SetTag(&yellow_tags, "rating:sex::nudity");
	tagdata.SetTag(&yellow_tags, "rating:sex::non-realistic:optional");
	tagdata.SetTag(&red_tags,    "rating:sex::non-realistic");
	tagdata.SetTag(&red_tags,    "rating:sex::realistic:optional");
	tagdata.SetTag(&red_tags,    "rating:sex::realistic");
	tagdata.SetTag(&red_tags,    "rating:sex::hard:optional");
	tagdata.SetTag(&red_tags,    "rating:sex::hard");
	tagdata.SetTag(&red_tags,    "rating:sex::violence:optional");
	tagdata.SetTag(&black_tags,  "rating:sex::violence");

	//tagdata.SetTag(&green_tags,  "rating:language::benign");

	//tagdata.SetTag(&green_tags,  "rating:discrimination::none");

	tagdata.SetTag(&yellow_tags, "rating:theme::death");
	tagdata.SetTag(&yellow_tags, "rating:theme::horror");
	tagdata.SetTag(&red_tags,    "rating:theme::cruelty");

	//std::cerr << "Green Tags: "; tagdata.Print(&green_tags);
	//std::cerr << "Yellow Tags: "; tagdata.Print(&yellow_tags);
	//std::cerr << "Red Tags: "; tagdata.Print(&red_tags);
	//std::cerr << "Black Tags: "; tagdata.Print(&black_tags);
}

PackageFilter::~PackageFilter()
{
}

int PackageFilter::TagValue(const Tag &tag)
{
	//std::string name = tag.fullname();
	//std::string facet_name = tag.facet().name();
	//std::string tag_name = tag.name();

	// The order is important to minimize calculation costs
	if (GreenTag(tag)) return PackageFilter::Green;
	if (BlackTag(tag)) return PackageFilter::Black;
	if (YellowTag(tag)) return PackageFilter::Yellow;
	if (RedTag(tag)) return PackageFilter::Red;
	return PackageFilter::Unknown;
}

bool PackageFilter::GreenTag(const Tag &tag)
{
	std::string name = tag.fullname();
	return tagdata.CheckTag(&green_tags, name);
}

bool PackageFilter::YellowTag(const Tag &tag)
{
	std::string name = tag.fullname();
	return tagdata.CheckTag(&yellow_tags, name);
}

bool PackageFilter::RedTag(const Tag &tag)
{
	std::string name = tag.fullname();
	return tagdata.CheckTag(&red_tags, name);
}

bool PackageFilter::BlackTag(const Tag &tag)
{
	std::string name = tag.fullname();
	return tagdata.CheckTag(&black_tags, name);
}

int PackageFilter::TagsValue(const TagSet &tags)
{
	int black = 0, red = 0, yellow = 0, green = 1 - GREEN_MINIMUM;
	for (TagSet::const_iterator i = tags.begin(); i != tags.end(); ++i)
	{
		switch (TagValue(*i))
		{
			case PackageFilter::Black: black++; break;
			case PackageFilter::Red: red++; break;
			case PackageFilter::Yellow: yellow++; break;
			case PackageFilter::Green: green++; break;
			default: break;
		}
	}

	if (black > 0)  return PackageFilter::Black;
	if (red > 0)    return PackageFilter::Red;
	if (yellow > 0) return PackageFilter::Yellow;
	if (green > 0)  return PackageFilter::Green;

	return PackageFilter::Unknown;
}
