# gozerbot/callbacks.py
#
#

""" callbacks triggered by ircevent CMND """

__copyright__ = 'this file is in the public domain'

from gozerbot.generic import rlog, handle_exception, calledfrom
from gozerbot.ircevent import makeargrest
from gozerbot.dol import Dol
from gozerbot.thr import start_new_thread
import sys, copy

class Callback(object):

    """ class representing a callback """

    def __init__(self, func, prereq, plugname, kwargs, threaded=True):
        # the callback function
        self.func = func
        # pre condition function
        self.prereq = prereq
        # plugin name
        self.plugname = plugname
        # kwargs to pass on to function
        self.kwargs = kwargs
        # flag to see if callback has to run in thread or in the mainloop
        self.threaded = copy.deepcopy(threaded)

class Callbacks(object):

    """ dict of lists containing callbacks """

    def __init__(self):
        # cbs are the callbacks which is a dict of lists .. 1 list per 
        # ievent.CMND
        self.cbs = Dol()

    def size(self):
        """ return nr of callbacks """
        return len(self.cbs)

    def add(self, what, func, prereq=None, kwargs=None, threaded=True, \
nr=False):
        """ add a callback """
        what = what.upper()
        # get the plugin this callback was registered from
        plugname = calledfrom(sys._getframe())
        # see if kwargs is set if not init to {}
        if not kwargs:
            kwargs = {}
        # add callback to the dict of lists
        if nr != False:
            self.cbs.insert(nr, what, Callback(func, prereq, plugname, \
kwargs, threaded))
        else:
            self.cbs.add(what, Callback(func, prereq, plugname, kwargs, \
threaded))
        rlog(-3, 'callbacks', 'added %s (%s)' % (what, plugname))

    def unload(self, plug):
        """ unload plugin """
        unload = []
        # look for all callbacks in a plugin
        for name, cblist in self.cbs.iteritems():
            index = 0
            for item in cblist:
                if item.plugname == plug:
                    unload.append((name, index))
                index += 1
        # delete callbacks
        for callback in unload[::-1]:
            self.cbs.delete(callback[0], callback[1])
            rlog(1, 'callbacks', 'unloaded %s' % callback[0])

    def whereis(self, what):
        """ show where ircevent.CMND callbacks are registered """
        result = []
        what = what.upper()
        for cmnd, callback in self.cbs.iteritems():
            if cmnd == what:
                for item in callback:
                    if not item.plugname in result:
                        result.append(item.plugname)
        return result

class Botcallbacks(Callbacks):

    """ callback on ircevent """

    def check(self, bot, ievent):
        """ check for callbacks to be fired """
        # check for "ALL" callbacks
        if self.cbs.has_key('ALL'):
            for cb in self.cbs['ALL']:
                self.callback(cb, bot, ievent)
        cmnd = ievent.cmnd.upper()
        # check for CMND callbacks
        if self.cbs.has_key(cmnd):
            for cb in self.cbs[cmnd]:
                self.callback(cb, bot, ievent)

    def callback(self, cb, bot, ievent):
        """ callback cb with bot and ievent as arguments """
        if not bot:
            return
        if ievent.nick and ievent.nick == bot.nick:
            return
        try:
            # set ievent bot and socket
            try:
                ievent.bot = bot
                ievent.sock = bot.sock
                makeargrest(ievent)
            except:
                pass
            # see if the callback pre requirement succeeds
            if cb.prereq:
                rlog(-10, 'callback', 'excecuting in loop %s' % \
str(cb.prereq))
                if not cb.prereq(bot, ievent):
                    return
            # check if callback function is there
            if not cb.func:
                return
            # start callback in its own thread
            rlog(-10, 'callback', 'excecuting callback %s' % \
str(cb.func))
            if cb.threaded:
                start_new_thread(cb.func, (bot, ievent), cb.kwargs)
            else:
                cb.func(bot, ievent, **cb.kwargs)
        except Exception, ex:
            handle_exception()

class Jabbercallbacks(Callbacks):

    """ callback on ircevent """

    def check(self, bot, ievent):
        """ check for callbacks to be fired """
        # check for "ALL" callbacks
        if self.cbs.has_key('ALL'):
            for cb in self.cbs['ALL']:
                self.callback(cb, bot, ievent)
        cmnd = ievent.cmnd.upper()
        # check for CMND callbacks
        if self.cbs.has_key(cmnd):
            for cb in self.cbs[cmnd]:
                self.callback(cb, bot, ievent)

    def callback(self, cb, bot, ievent):
        """ callback cb with bot and ievent as arguments """
        if not bot:
            return
        if str(bot.jid) in str(ievent.getFrom()):
            return
        try:
            # see if the callback pre requirement succeeds
            if cb.prereq:
                rlog(-10, 'callback', 'excecuting in loop %s' % \
str(cb.prereq))
                if not cb.prereq(bot, ievent):
                    return
            # check if callback function is there
            if not cb.func:
                return
            # start callback in its own thread
            rlog(-10, 'callback', 'excecuting callback %s' % \
str(cb.func))
            if cb.threaded:
                start_new_thread(cb.func, (bot, ievent), cb.kwargs)
            else:
                cb.func(bot, ievent, **cb.kwargs)
        except Exception, ex:
            handle_exception()

callbacks = Botcallbacks()
jcallbacks = Jabbercallbacks()
