# wrapper for the GNU Privacy Guard
# (c) Wijnand 'tehmaze' Modderman - http://tehmaze.com
# BSD License

from gozerbot.config import config
from gozerbot.datadir import datadir
from gozerbot.generic import gozerpopen, rlog
import os
import re
import tempfile

class PgpNoPubkeyException(Exception):
    pass

class PgpNoFingerprintException(Exception):
    pass

class NoGPGException(Exception):
    pass

class Pgp(object):
    ''' Wrapper for the GNU Privacy Guard. '''   
 
    re_verify = re.compile('^\[GNUPG:\] VALIDSIG ([0-9A-F]+)', re.I | re.M)
    re_import = re.compile('^\[GNUPG:\] IMPORT_OK \d ([0-9A-F]+)', re.I | re.M)
    re_pubkey = re.compile('^\[GNUPG:\] NO_PUBKEY ([0-9A-F]+)', re.I | re.M)

    def __init__(self):
        self.homedir = os.path.join(datadir, 'pgp')
        if not os.path.exists(self.homedir):
            rlog(5, 'pgp', 'creating directory %s' % self.homedir)
            os.mkdir(self.homedir)
        if hasattr(config, 'pgpkeyserver'):
            self.keyserver = config.pgpkeyserver
        else:
            self.keyserver = 'pgp.mit.edu'
        # make sure the pgp dir is safe
        os.chmod(self.homedir, 0700)
        rlog(5, 'pgp', 'will be using public key server %s' % self.keyserver)

    def exists(self, keyid):
        (data, returncode) = self._gpg('--fingerprint', keyid)
        return returncode == 0

    def imports(self, data):
        tmp = self._tmp(data)
        (data, returncode) = self._gpg('--import', tmp)
        os.unlink(tmp)
        test_import = self.re_import.search(data)
        if test_import:
            fingerprint = test_import.group(1)
            return fingerprint
        return None

    def imports_keyserver(self, fingerprint):
        (data, returncode) = self._gpg('--keyserver ', self.keyserver, \
'--recv-keys', fingerprint)
        if returncode == 256:
            raise NoGPGException()
        return returncode == 0

    def remove(self, data):
        if len(data) < 8:
            return False
        (_, returncode) = self._gpg('--yes', '--delete-keys',  data)
        if returncode == 256:
            raise NoGPGException()
        return returncode == 0

    def verify(self, data):
        tmp = self._tmp(data)
        (data, returncode) = self._gpg('--verify', tmp)
        os.unlink(tmp)
        if returncode == 256:
            raise NoGPGException()
        if returncode not in (0, 512):
            return False
        test_pubkey = self.re_pubkey.search(data)
        if test_pubkey:
            raise PgpNoPubkeyException(test_pubkey.group(1))
        test_verify = self.re_verify.search(data)
        if test_verify:
            return test_verify.group(1)

    def verify_signature(self, data, signature):
        tmp1 = self._tmp(data)
        tmp2 = self._tmp(signature)
        (data, returncode) = self._gpg('--verify', tmp2, tmp1)
        os.unlink(tmp2)
        os.unlink(tmp1)
        if returncode == 256:
            raise NoGPGException()
        if returncode not in (0, 512):
            return False
        test_pubkey = self.re_pubkey.search(data)
        if test_pubkey:
            raise PgpNoPubkeyException(test_pubkey.group(1))
        test_verify = self.re_verify.search(data)
        if test_verify:
            return test_verify.group(1)
        return None
        
    def _gpg(self, *extra):
        cmnd = ['gpg', '--homedir', self.homedir, '--batch', '--status-fd', '1']
        cmnd += list(extra)
        rlog(0, 'pgp', 'executing: %s' % ' '.join(cmnd))
        try:
            proces = gozerpopen(cmnd, [])
        except Exception, ex:
            rlog(0, 'pgp', 'error running "%s": %s' % (cmnd, str(ex)))
            return False
        data = proces.fromchild.read()
        returncode = proces.close()
        rlog(0, 'pgp', 'return code: %s, data: %s' % (returncode, data))
        return (data, returncode)

    def _tmp(self, data):
        fdid, tmp = tempfile.mkstemp()
        fd = os.fdopen(fdid, 'w')
        fd.write(data)
        fd.close()
        return tmp

# these are the gozerbot.org plugin repositories
gozerbot_pgp_keys = {
    # gozerbot.org/plugs install key
    '5290D844CF600B6CA6AFF8952D6D437CC5C9C2B2': '''
-----BEGIN PGP PUBLIC KEY BLOCK-----

mQGiBEZM0ZURBADomT07whLs4n/ml84aIl8Ch6ShngfaaOS12ZrBQVQ/VSh7zPOx
IzfhSmwDJAWLZOnnM5zAqWPuNSJa3hQzY/M+X7vv3/p7kkB54/5U6LW+8ODENnMe
yPJhbI7phlfeE+STK9hytC3W5+OSrQknXkwYm1bGOur0iiU4Nr16ePE19wCg4KIc
ecqxm1U2CRtVC6G3qrZpscMD/0loPcv6Yw7Sx+3UwgAFaOJNE73P+h87wz29WkKs
h8Sx/l+Zf2NW/cMUwR0OOQQpzXKtZ8mF/CuPY4YITThKEGh680dUecuGBk5k3LeE
ZuLCFagEwZDWqXq/rR7+v8KCaxHpsaoU9P9p3Z7mruvaTYFp+yNfIYAklYoVI2iO
pB9PA/94xF3Vsdm3rqT6MUCVbBLKRRNHh7RDiFhQb0GfoWQiae0ZdJO2zWKCprKc
cboswKVs0SEUAD30izAaMlfR/p/LrFDHYwr5bBm38xhRMgFxrqdV+5o1+bdYNGA6
dJc6XgDWhDHpErCDibGAugNfVCDPhTE1eKbQQXST1KHnhyOcWbQeQmFydCBUaGF0
ZSA8YmFydEBnb3plcmJvdC5vcmc+iGAEExECACAFAkZM0ZUCGwMGCwkIBwMCBBUC
CAMEFgIDAQIeAQIXgAAKCRAtbUN8xcnCslzoAKCq7QqTlT10I6WcDZddY9GqcKxp
VQCaAgX1r9e39X79HR3NxTU6EHgm1hm5Ag0ERkzRnxAIALEuEASs9rpQNcpn6+UJ
M43J6vu9qYPl6T4ljsTdsi11Lr7r2ltFmvu/RqA9R4M4QwS/JZjm71R3Ci0eJSnR
AlkhejYHn5TKJ5rzRO6EUL0Jd/Rwgk/9qivIyfKAW3A1To2JgRUP9WI19MKYu6e9
K5h3KNi/1FfpeOFptB9mHyXBO2rHR9gaN6Wu9uz+eOGlZCyhNRx1jlmVcKAudT2g
qg/8NsuiEmv9Wf/CMgYawyenqx8S7cYC5EvHEscIOI/8zma0/1JqD8vllbSh64Ih
fsaW3twx/2gAIgtxRSIW7RQuYhs+zBT2C+Sg91rEgGcXjaw+OW2OZJ4kq1Qba82V
UOcAAwUH/21DRtgKgAX6xk97l+6ItP19HrcmydxNh299BboSKJDh5RcO1H4xOYyx
8NCDLhluGNzqeKPbBeW60F3qpkoQMGD7XsVpfKFu4Umn2qtrZAlg6qucgF0476IO
rhvKskAbNMBlocKzxiT+Ruomkt0Bwa3S1owXlMFwImA8coiWCQ8QnbpvjsYW9OH4
AgznK5uXKWgMxlZ9SffU0IHXipl3TENXiIdSwUUc2gV5emV/ROFhASNl7YYxZhY6
ng8YRku3MXM2jN1S/gNX1hvcq20oFjucKx7YkkV1k23FKF6XK776bmD0hYiSH1e0
uSAlBPAgC5HPqMpFhfyeFnDW0N90DR6ISQQYEQIACQUCRkzRnwIbDAAKCRAtbUN8
xcnCsoVMAJ472BdMgh+eUJp0+rGCeQhLELJVlQCcCV1zrNw9yoxIGJYnMZZSn4V5
Yf8=
=UxRq
-----END PGP PUBLIC KEY BLOCK-----
''',
    # tehmaze.com/plugs install key
    'D561C5D7CD86CB3CCA1AB5512A22EC17F9EBC3D8': '''
-----BEGIN PGP PUBLIC KEY BLOCK-----

mQGiBEZMjOARBACYqwLsv7AMOEHL4wspu0DVXwzVm3U3AU2LivHaJjj66zTd16K4
1QQ8CjdWVjMLOSH9tvj0Z7wbdyeQEjAxykgIgIUvQ0zuHgzqVsbpW//W/Noc1Z3K
MIEPBQIDdWM7Ln9+1jZAWIKU6oPU6F9Qt2/8o2NDc0w8O73NKgn/NGWtqwCglitS
SvUvpP3HZZ8aYwqjk51j0V8D/ilgVNkb/7FumD2yF1R48bJmbRaFu58Hu2IplRr0
cGHZ1ijCR8fWeMPGY3CakEOjxQa9IkNtoce/PGTgbIl+TLwXSdiiyu5xkEIt2HHm
F3lzLw2o6T59g2w22KpLeXMO02+LcNdsV/BOrhLiO7E/cDB7wBfd/BG3zNC9C2ln
cmL4BACL5SEaOE4BCRrOfWdDDUF9iiXGxpCErdxlwId9eFM5OMe48ZmB7oGY52dY
rzGK54bqkVxRf5MOcUy3IjulZcy/LcOmm+agNnbQPVTuWNfty8+pSs6cPnF2bBUN
xf+UEZnQRlUHokHBy20DMjV3v8jXMBtZxlB24EmMDE21nAofZrQxdGVobWF6ZS5j
b20gKGdvemVycGx1Z3MpIDxnb3plcnBsdWdzQHRlaG1hemUuY29tPohgBBMRAgAg
BQJGTIzgAhsDBgsJCAcDAgQVAggDBBYCAwECHgECF4AACgkQKiLsF/nrw9gngQCf
esACa4M8ZjvMul8xad4oIUHkMTYAn0+bZBN8ip10/5qIR/CdDvwSL56b
=RTHp
-----END PGP PUBLIC KEY BLOCK-----
'''}

# initiate a Pgp instance and import the distkeys, if needed
pgp = Pgp()
for keyid, keydata in gozerbot_pgp_keys.iteritems():
    if not pgp.exists(keyid):
        pgp.imports(keydata)

