/***************************************************************************
 *            qof-main.c
 *
 *  This is an auto-generated file. Patches are available from
 *  http://qof-gen.sourceforge.net/
 *
 *  Thu Jan 13 10:55:44 2005
 *  Copyright  2005-2006  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/** @addtogroup QOFCLI

Includes common functions for all QOF CLI programs and provides generic
functions to implement command line and interactive shell options.
@{
*/
/** @file qof-main.c
    @brief Common functions for the QOF external framework
    @author Copyright (c) 2005 Neil Williams <linux@codehelp.co.uk>
*/

#define _GNU_SOURCE
#include "config.h"
#include <glib.h>
#include <glib/gi18n.h>
#include <glib/gprintf.h>
#include <qof.h>
#include <stdlib.h>
#include <stdio.h>
#include <regex.h>
#include <time.h>
#include "qof-main.h"

#define MAX_LINE 79

/* the debugging module for this file. */
static QofLogModule log_module = QOF_MAIN_CLI;

void
qof_main_wrap_line (FILE * fp, gint indent, 
					const gchar * template, ...)
{
	gint line_length, msg_length;
	va_list wraps;
	gchar *message;

	line_length = MAX_LINE;
	/* note the modulus. Don't use CLAMP here */
	/* indent != line_length or particularly close to it. */
	indent = indent >= line_length ? indent % line_length : indent;
	indent = indent < 0 ? 0 : indent;
	message = NULL;
	g_return_if_fail (template);
	va_start (wraps, template);
	message = g_strdup_vprintf (template, wraps);
	va_end (wraps);
	g_return_if_fail (message);
	msg_length = strlen (message);
	while (msg_length > line_length)
	{
		gchar *chunk;
		gchar format[16];

		chunk = message + line_length - 1;
		while (chunk > message && !g_ascii_isspace (*chunk))
			chunk--;
		if (chunk == message)
			break;				/* give up */
		while (chunk > (message + 1) && g_ascii_isspace (*chunk))
			chunk--;
		chunk++;
		g_sprintf (format, "%%.%ds\n%%%ds", (gint) (chunk - message),
			indent);
		g_fprintf (fp, format, message, "");
		message = chunk;
		while (g_ascii_isspace (*message) && *message)
			message++;
		msg_length = strlen (message);
		if (line_length == MAX_LINE)
			line_length -= indent;
	}
	if (msg_length)
		g_fprintf (fp, "%s\n", message);
}

gchar *
qof_main_make_utf8 (gchar * string)
{
	gchar *value;

	if (!string)
		return NULL;
	if (g_utf8_validate (string, -1, NULL))
		return string;
	value = g_locale_to_utf8 (string, -1, NULL, NULL, NULL);
	if (!value)
	{
		PWARN (" unable to convert from locale %s", string);
		PINFO ("trying to convert from ISO-8859-15.");
		value = g_convert (string, -1, "UTF-8", "ISO-8859-15",
			NULL, NULL, NULL);
		if (!value)
		{
			PERR (" conversion failed");
			return string;
		}
		return value;
	}
	return value;
}

static void
qof_main_run_sql (QofMainContext * context)
{
	QofSqlQuery *q;
	QofBook *book;
	gchar *sql;

	ENTER (" ");
	q = qof_sql_query_new ();
	sql = g_strdup (context->sql_str);
	book = qof_session_get_book (context->input_session);
	qof_sql_query_set_book (q, book);
	qof_sql_query_run (q, sql);
	context->query = qof_sql_query_get_query (q);
	LEAVE (" ");
}

static void
qof_main_run_query (QofMainContext * context)
{
	QofBook *book;
	GList *results;

	ENTER (" ");
	results = NULL;
	book = qof_session_get_book (context->input_session);
	qof_query_set_book (context->query, book);
	results = qof_query_run (context->query);
	if (results != NULL)
		qof_entity_copy_list (context->export_session, results);
	LEAVE (" ");
}

void
qof_main_free (QofMainContext * context)
{
	g_free (context->filename);
	g_free (context->write_file);
	g_free (context->sql_file);
	g_free (context->database);
	g_free (context->category);
}

static void
find_param_cb (QofParam * param, gpointer user_data)
{
	gchar * tmp;
	QofMainContext *context;
	QofQueryPredData *time_pred_data;

	context = (QofMainContext *) user_data;
	if ((param->param_getfcn == NULL) || 
		(param->param_setfcn == NULL))
		return;
	if (0 == safe_strcmp (context->param_type, param->param_type))
	{
		time_pred_data = qof_query_time_predicate (QOF_COMPARE_GTE,
			QOF_DATE_MATCH_NORMAL, context->min_qt);
		tmp = g_strdup (param->param_name);
		qof_query_add_term (context->query, 
			qof_query_build_param_list (tmp, NULL), time_pred_data, QOF_QUERY_AND);
		time_pred_data = qof_query_time_predicate (QOF_COMPARE_LTE,
			QOF_DATE_MATCH_NORMAL, context->max_qt);
		qof_query_add_term (context->query, 
			qof_query_build_param_list (tmp, NULL), time_pred_data, QOF_QUERY_AND);
		qof_main_run_query (context);
		qof_query_purge_terms (context->query, 
			qof_query_build_param_list (tmp, QOF_ID_BOOK, QOF_TYPE_GUID, NULL));
		PINFO (" param_name=%s added", tmp);
	}
	LEAVE (" ");
}

/* takes one database name and runs -c and -t queries against it. */
static void
build_database_list (QofIdTypeConst obj_type, QofMainContext * context)
{
	if (!obj_type || !context)
		return;
	if (!qof_class_is_registered (obj_type))
		return;
	ENTER (" object_type=%s", obj_type);
	context->query = qof_query_create_for (obj_type);
	if (context->category != NULL)
	{
		QofQueryPredData *category_pred;

		category_pred =
			qof_query_string_predicate (QOF_COMPARE_EQUAL,
			context->category, QOF_STRING_MATCH_CASEINSENSITIVE, 
			FALSE);
		qof_query_add_term (context->query, 
			qof_query_build_param_list (CATEGORY_NAME, NULL),
			category_pred, QOF_QUERY_AND);
	}
	if (context->min_qt)
	{
		PINFO (" Preparing a time based queryset.");
		context->param_type = QOF_TYPE_TIME;
		qof_class_param_foreach (obj_type, find_param_cb, context);
	}
	else
	{
		qof_main_run_query (context);
		if (context->query)
			qof_query_clear (context->query);
	}
	LEAVE (" ");
}

static void
select_cb (QofObject * obj, gpointer data)
{
	QofMainContext *context;

	context = (QofMainContext *) data;
	g_return_if_fail (context);
	if (0 != safe_strcmp (context->exclude, obj->e_type))
		build_database_list (obj->e_type, context);
}

void
qof_main_moderate_query (QofMainContext * context)
{
	GSList *date_param_list, *category_param_list;
	gboolean all;

	ENTER (" ");
	all = TRUE;
	context->query = qof_query_create ();
	date_param_list = NULL;
	category_param_list = NULL;
	while (context->sql_list)
	{
		PINFO ("running sql_list");
		context->sql_str = g_strdup (context->sql_list->data);
		qof_main_run_sql (context);
		qof_main_run_query (context);
		if (context->query)
			qof_query_clear (context->query);
		g_free (context->sql_str);
		context->sql_str = NULL;
		all = FALSE;
		context->sql_list = g_list_next (context->sql_list);
	}
	if (0 < g_list_length (context->sql_list))
	{
		context->sql_str = NULL;
		g_list_free (context->sql_list);
		all = FALSE;
	}
	if (context->sql_str != NULL)
	{
		PINFO ("running sql_str");
		qof_main_run_sql (context);
		qof_main_run_query (context);
		if (context->query)
			qof_query_clear (context->query);
		all = FALSE;
	}
	if ((context->exclude != NULL)
		&& (qof_class_is_registered (context->exclude)))
	{
		qof_object_foreach_type (select_cb, context);
		all = FALSE;
	}
	if ((context->database != NULL)
		&& (qof_class_is_registered (context->database)))
	{
		build_database_list (context->database, context);
		all = FALSE;
	}
	if (all == TRUE)
		qof_object_foreach_type (select_cb, context);
	LEAVE (" ");
}

static void
option_cb (QofBackendOption * option, gpointer data)
{
	QofMainContext *context;

	context = (QofMainContext *) data;
	g_return_if_fail (context);
/* Normally, I'd use GPOINTER_TO_INT but internally, 
the QofBackendOption uses gint64 which gets mangled by
the 32-bit macro. */
	ENTER (" compression=%" G_GINT64_FORMAT " encoding=%s",
		context->gz_level, context->encoding);
	if (0 == safe_strcmp (QSF_COMPRESS, option->option_name))
		option->value = (gpointer) & context->gz_level;
	if (0 == safe_strcmp (QSF_ENCODING, option->option_name))
		option->value = (gpointer) g_strdup(context->encoding);
	if (0 == safe_strcmp (QSF_DATE_CONVERT, option->option_name))
		option->value = (gpointer) & context->convert;
	LEAVE (" ");
}

void
qof_mod_compression (gint64 gz_level, QofMainContext * context)
{
	KvpFrame *be_config;
	QofBook *book;
	QofBackend *be;

	ENTER (" compression=%" G_GINT64_FORMAT, gz_level);
	if ((gz_level > 0) && (gz_level <= 9))
	{
		book = qof_session_get_book (context->export_session);
		be = qof_book_get_backend (book);
		be_config = qof_backend_get_config (be);
		context->gz_level = gz_level;
		qof_backend_option_foreach (be_config, option_cb, context);
		qof_backend_load_config (be, be_config);
	}
	LEAVE (" ");
}

void
qof_mod_encoding (const gchar * encoding, QofMainContext * context)
{
	KvpFrame *be_config;
	QofBook *book;
	QofBackend *be;

	ENTER (" encode to %s", encoding);
	book = qof_session_get_book (context->export_session);
	be = qof_book_get_backend (book);
	be_config = qof_backend_get_config (be);
	context->encoding = encoding;
	qof_backend_option_foreach (be_config, option_cb, context);
	qof_backend_load_config (be, be_config);
	LEAVE (" ");
}

void
qof_mod_convert_deprecated (gint64 convert, QofMainContext * context)
{
	KvpFrame *be_config;
	QofBook *book;
	QofBackend *be;
	gboolean set;

	set = (convert == 0) ? FALSE : TRUE;
	ENTER (" convert deprecated date values? %i No if 0.", set);
	book = qof_session_get_book (context->export_session);
	be = qof_book_get_backend (book);
	be_config = qof_backend_get_config (be);
	context->convert = convert;
	qof_backend_option_foreach (be_config, option_cb, context);
	qof_backend_load_config (be, be_config);
	LEAVE (" ");
}

void
qof_cmd_xmlfile (QofMainContext * context)
{
	QofSession *input_session, *export_session;

	ENTER (" ");
	input_session = context->input_session;
	if (0 == safe_strcmp (context->exclude, context->database)
		&& (context->exclude != NULL))
	{
		qof_main_wrap_line (stderr, ERR_INDENT,
		/* Translators: This line is wrapped by the program - 
		please make sure you do NOT add line endings here. */
			_("%s: Error: Cannot exclude database \"%s\" with option -e "
				"because option -d is set to include the database: \"%s\". "
				"Use the \'-l\' command to see the full list of supported "
				"databases.\n"), PACKAGE, context->exclude,
			context->database);
		qof_session_end (input_session);
		LEAVE (" conflicting options");
		return;
	}
	qof_session_begin (input_session, context->filename, TRUE, FALSE);
	qof_main_show_error (input_session);
	if (0 != safe_strcmp (QOF_STDOUT, context->filename))
		qof_session_load (input_session, NULL);
	qof_main_show_error (input_session);
	export_session = qof_session_new ();
	context->export_session = export_session;
	if (context->write_file)
	{
		qof_session_begin (export_session, context->write_file, TRUE,
			TRUE);
		qof_mod_compression (context->gz_level, context);
	}
	else
		qof_session_begin (export_session, QOF_STDOUT, TRUE, FALSE);
	qof_main_show_error (export_session);
	/* ensure encoding value is set in the new export_session */
	qof_mod_encoding (context->encoding, context);
	qof_main_moderate_query (context);
	qof_session_save (export_session, NULL);
	qof_main_show_error (export_session);
	qof_main_show_error (input_session);
	qof_session_end (input_session);
	qof_session_end (export_session);
	LEAVE (" ");
}

static void
qof_main_list (QofObject * obj, gpointer G_GNUC_UNUSED data)
{
	fprintf (stdout, "%-20s%-20s\n", obj->e_type, obj->type_label);
}

void
qof_main_select (QofMainContext * context)
{
	g_return_if_fail (context);
	qof_object_foreach_type (select_cb, context);
}

void
qof_cmd_list (void)
{
	qof_main_wrap_line (stdout, 0,
	/* Translators: This line is wrapped by the program - 
	please make sure you do NOT add line endings here. */
		_("\n%s: You can use the supported database names with '%s -d' "
			"and in SQL queries (as the table name) with '%s -s|f'. "
			"Descriptions are shown only for readability.\n"),
		PACKAGE, PACKAGE, PACKAGE);
	fprintf (stdout, "%-20s%-20s\n", _("Name"), _("Description"));
	qof_object_foreach_type (qof_main_list, NULL);
	qof_main_wrap_line (stdout, 0,
	/* Translators: This line is wrapped by the program -
	please make sure you do NOT add line endings here. */
		_("\nUse '%s -d <database> --explain' to see the list of fields "
			"within any supported database."), PACKAGE);
	fprintf (stdout, _("\nThank you for using %s\n\n"), PACKAGE);
}

static void
explain_cb (QofParam * param, gpointer G_GNUC_UNUSED user_data)
{
	if (param->param_getfcn && param->param_setfcn)
		fprintf (stdout, _("Type: %s\tName: %s\n"),
			param->param_type, param->param_name);
}

void
qof_cmd_explain (QofMainContext * context)
{
	if (context->error)
		return;
	fprintf (stdout, _("\nParameters of the %s database:\n\n"),
		context->database);
	qof_class_param_foreach (context->database, explain_cb, NULL);
	fprintf (stdout, _("\nThank you for using %s\n\n"), PACKAGE);
}

void
qof_mod_category (const gchar * category, QofMainContext * data)
{
	data->category = g_strdup (category);
}

glong
qof_mod_get_local_offset (void)
{
	glong local_offset;
	struct tm local;
	time_t now;

	local_offset = 0; /* UTC */
	now = time (NULL);
	local = *localtime_r (&now, &local);
	local_offset -= local.tm_gmtoff;
	return local_offset;
}

void
qof_mod_database (const gchar * database, QofMainContext * data)
{
	if (qof_class_is_registered (database))
		data->database = g_strdup (database);
}

void
qof_mod_time (const gchar * date_time, QofMainContext * data)
{
	QofDate *qd;
	gboolean all_year, all_month;
	gint adding_days;
	gchar *info;

	/* incoming date is assumed to be localtime */
	ENTER (" date_time=%s", date_time);
	all_month = all_year = FALSE;
	g_return_if_fail (date_time);
	qd = qof_date_parse (date_time, QOF_DATE_FORMAT_ISO);
	if (!qd)
		qd = qof_date_parse (date_time, QOF_DATE_FORMAT_UTC);
	info = qof_date_print (qd, QOF_DATE_FORMAT_ISO8601);
	PINFO (" parsed start_time=%s", info);
	g_free (info);
	/* set first second of day, UTC */
	qof_date_set_day_start (qd);
	data->min_qt = qof_date_to_qtime (qd);
	/* adjust for incoming localtime */
	qof_time_add_secs (data->min_qt, 
		qof_mod_get_local_offset());
	/* year specified but no month or day, select the entire year */
	if (strlen (date_time) == 4)
	{
		PINFO (" match entire year %s", date_time);
		/* go to end of this year, not first day of next. */
		adding_days = qof_date_isleap(qd->qd_year) ? 365 : 364;
		qof_date_adddays (qd, adding_days);
	}
	/* month specified, but no day, select entire month */
	if (strlen (date_time) == 7)
	{
		PINFO (" match entire month %s", date_time);
		adding_days = qof_date_get_mday (qd->qd_mon, qd->qd_year);
		qof_date_adddays (qd, adding_days - 1);
	}
	/* set last second of day */
	qof_date_set_day_end (qd);
	data->max_qt = qof_date_to_qtime (qd);
	/* adjust for incoming localtime */
	qof_time_add_secs (data->max_qt, 
		qof_mod_get_local_offset());
	LEAVE (" ");
}

void
qof_mod_exclude (const gchar * exclude, QofMainContext * data)
{
	if (qof_class_is_registered (exclude))
		data->exclude = g_strdup (exclude);
}

void
qof_mod_sql (const gchar * sql_query, QofMainContext * data)
{
	data->sql_str = g_strdup (sql_query);
}

void
qof_mod_sql_file (const gchar * sql_file, QofMainContext * data)
{
	FILE *filehandle;
#ifndef HAVE_GETLINE
	gchar lineptr[1024];
#else
	gchar *lineptr;
#endif
	gchar *buf;
	size_t n;
	QofQuery *q;
	regex_t *r;
	gint reg_exp_check;
	const gchar *fmt;
	static gchar *pattern = QOF_SQL_SUPPORTED;

	ENTER (" ");
	data->sql_file = g_strdup (sql_file);
	n = 0;
	q = NULL;
	data->sql_list = NULL;
	filehandle = fopen (sql_file, "r");
	if (!filehandle)
	{
		fmt = _("%s: There was an error reading the file '%s'.\n");
		qof_main_wrap_line (stderr, ERR_INDENT, fmt, PACKAGE, sql_file);
		return;
	}
	r = g_new (regex_t, 1);
#ifndef HAVE_GETLINE
	while (NULL != (fgets (lineptr, sizeof (lineptr), filehandle)))
#else
	lineptr = NULL;
	while (0 < getline (&lineptr, &n, filehandle))
#endif
	{
		reg_exp_check =
			regcomp (r, pattern, REG_ICASE | REG_NOSUB | REG_EXTENDED);
		g_return_if_fail (reg_exp_check == 0);
		if (0 != regexec (r, lineptr, 0, NULL, 0))
			continue;
		buf = g_strdup (g_strchomp (lineptr));
		data->sql_list = g_list_prepend (data->sql_list, buf);
	}
	regfree (r);
	g_free (r);
	fclose (filehandle);
	LEAVE (" sql_list=%d", g_list_length (data->sql_list));
}

void
qof_mod_write (const gchar * write_file, QofMainContext * data)
{
	data->write_file = g_strdup (write_file);
}

void
qof_main_show_error (QofSession * session)
{
	const gchar *fmt;

	if (qof_error_check (session))
	{
		fmt = "%s: %s\n";
		qof_main_wrap_line (stderr, ERR_INDENT, fmt, PACKAGE,
			qof_error_get_message (session));
	}
}

/** @} */

/*==================== END OF FILE ======================*/
