/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*---------------------------------------------------------------------------

   scale - applies spatial and time scale to PIV data

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

------------------------------------------------------------------------*/

#include <stdio.h> 
#include <stdlib.h> 
#include <glib.h>
#include <gpiv.h>

/* #define PARFILE "scale.par"    */  /* Parameter file name */
#define PARFILE "gpivrc"     /* Parameter file name */
#define USAGE "\
Usage: scale [-f filename][ -h | --help] [-p | --print] [-s M | -S M] \n\
             [-t dt | -T dt] [-v | --version] [-z px0 py0] < stdin > stdout \n\
\n\
keys: \n\
-f filename:           files (without .piv extension) instaed of stdin and \n\
                       stdout \n\
-h | --help:           this on-line help \n\
-p | --print:          print parameters to stdout \n\
-s M:                  spatial scaling with magnifation factor M [mm/px]  \n\
-S M:                  inverse spatial scaling \n\
-t dt:                 time scaling with dt time between subsequent \n\
                       recordings [s] \n\
-T dt:                 inverse time scaling \n\
-v | --version:        version number \n\
-z px0 py0:            zero offset of image position \n\
"

#ifdef DEBUG
#define USAGE_DEBUG "\
Developers version also contains:  \n\
               [-p_main] \n\
keys: \n\
-p_'function' N; prints data to be generated in the function; the \n\
                   higher N, the more detailed the output. \n\
                   For N = 10, err_vec will exit at the end of the function"
#endif


#define HELP  "\
scale applies spatial and time scaling to PIV data"

#define RCSID "$Id: scale.c,v 2.9 2006/01/31 14:18:04 gerber Exp $"

/*
 * Global variables
 */
gboolean fname_logic = FALSE;
gboolean print_par = 0;
GpivPivData in_data, out_data;

#ifdef DEBUG
/*
 * Variables for development version 
 */
int print_main=0;

#endif 


static void 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS],
             GpivImagePar * image_par,
             GpivPostPar * post_par
             )
/* ----------------------------------------------------------------------------
 * Command line argument handling 
 */
{
     char c;
     int argc_next;


     while (--argc > 0 && (*++argv)[0] == '-') {
	  argc_next=0;
/*
 * argc_next is set to 1 if the next cmd line argument has to be searched for; 
 * in case that the command line argument concerns more than one char or cmd 
 * line argument needs a parameter
 */
	  while (argc_next==0 && (c = *++argv[0])) {
	       switch (c) {

	       case 'v':
/*
 * Use Revision Control System (RCS) for version
 */
		    printf("%s\n", RCSID); 
		    exit(0);
		    break;

	       case 'f':
		    strcpy(fname,*++argv); 
		    fname_logic = TRUE;
		    argc_next=1;
		    --argc;
		    break;

	       case 'h':
		    printf("%s\n", RCSID); 
		    printf("%s\n",HELP);
		    printf("%s\n",USAGE);
#ifdef DEBUG
		    printf ("\n%s\n",USAGE_DEBUG);
#endif
		    exit(0);
		    break;

/*
 * Spatial scaling
 */
	       case 's':
		    image_par->s_scale = atof(*++argv);
                    image_par->s_scale_logic = TRUE;
                    post_par->scale_type = GPIV_SCALE;
		    argc_next=1;	
		    --argc;
		    break;

/*
 * Inverse spatial scaling
 */
	       case 'S':                 
		    image_par->s_scale = atof(*++argv);
                    image_par->s_scale_logic = TRUE;
                    post_par->scale_type = GPIV_SCALE_INV;
		    argc_next=1;	
		    --argc;
		    break;

/*
 * Time scaling
 */
	       case 't':                 
		    image_par->t_scale = atof(*++argv);
		    image_par->t_scale_logic = TRUE;
                    post_par->scale_type = GPIV_SCALE;
		    argc_next=1;	
		    --argc;
		    break;

/*
 * Inverse time scaling
 */
	       case 'T':                 
		    image_par->t_scale = atof(*++argv);
		    image_par->t_scale_logic= TRUE;
                    post_par->scale_type = GPIV_SCALE_INV;
		    argc_next=1;	
		    --argc;
		    break;

/*
 * Zero-offset x0 y0
 */
	       case 'z':                 
			 image_par->z_off_x = atof(*++argv);
			 image_par->z_off_y = atof(*++argv);
			 image_par->z_off_x_logic = TRUE;
			 image_par->z_off_y_logic = TRUE;
			 argc_next=1;	
			 --argc;
			 --argc;
		    break;

	       case 'p':
#ifdef DEBUG
		    if ((strcmp(*argv,"p_main" ) !=0))
		    {
#endif
			 print_par = TRUE;
#ifdef DEBUG
		    } else if (strcmp("p_main",*argv) == 0) {  
			 print_main=atoi(*++argv);
			 --argc; 
			 argc_next=1;
		    }
#endif /* DEBUG */
		    break;
/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

	       default:
		    fprintf (stderr,USAGE);
#ifdef DEBUG
		    printf ("\n%s",USAGE_DEBUG);
#endif
		    exit(1);
		    break;
	       }
	  }
     }


     if(argc != 0) { 
	  gpiv_error("%s: %s", RCSID, USAGE);
#ifdef DEBUG
	  printf ("\n%s", USAGE_DEBUG);
#endif
	  exit(1); 
     }


}



int 
make_fname(char *fname, 
           char *fname_header, 
           char *fname_parameter, 
           char *fname_in, 
           char *fname_out
           )
/*  ---------------------------------------------------------------------------
 * function to generate filenames
 */
{
     if (fname_logic == FALSE){
	  gpiv_error("%s: Filename has to be set", RCSID);
     }
     
     gpiv_io_make_fname(fname, GPIV_EXT_HEADER, fname_header);
     if (print_par)     printf("# Image header file: %s\n",fname_header);
     
     gpiv_io_make_fname(fname, GPIV_EXT_PAR, fname_parameter);
     if (print_par)     printf("# Parameter file: %s\n",fname_parameter);
     
     gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_in);
     if (print_par)     printf("# Input data file: %s\n",fname_in);
     
     gpiv_io_make_fname(fname, GPIV_EXT_SC, fname_out);
     if (print_par)     printf("# Output file: %s\n",fname_out);
     
     return 0;
}



int
main(int argc, 
     char *argv[]
     )
/* ----------------------------------------------------------------------------
 *  Start of the main program
 */
{
    char * err_msg = NULL;
     FILE  *fp_par_dat;
     char fname[GPIV_MAX_CHARS], fname_out[GPIV_MAX_CHARS], 
         fname_header[GPIV_MAX_CHARS], 
	  fname_parameter[GPIV_MAX_CHARS], fname_in[GPIV_MAX_CHARS];
     char  d_line[GPIV_MAX_LINES][GPIV_MAX_CHARS], 
         c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
     char return_val;
     int nd_lines = 0, nc_lines = 0, var_scale = 0;

     GpivImagePar image_par;
     GpivPostPar post_par, post_par_default;


/*     image_par.s_scale_logic = FALSE; */
/*     image_par.t_scale_logic = FALSE;  */
/*     image_par.z_off_logic = FALSE; */
    gpiv_img_parameters_logic(&image_par, FALSE);
    gpiv_post_parameters_logic(&post_par, FALSE);
    gpiv_post_default_parameters(&post_par_default, TRUE);
     

     command_args(argc, argv, fname, &image_par, &post_par);
    if (print_par) {
      printf("# %s\n# Command line options:\n", RCSID);
      gpiv_img_print_parameters(image_par);
      gpiv_post_print_parameters(post_par);
    }


     if(fname_logic == TRUE) {
/*
 * Generating proper filenames
 */
	  if ((return_val = make_fname(fname, fname_header, fname_parameter, fname_in, 
				       fname_out)) != 0) {
	       gpiv_error("%s: Failure calling make_fname", RCSID);
	  }
	  

/*
 * Prints command line parameters to par-file 
 */
	  if ((fp_par_dat = fopen (fname_parameter, "a")) == NULL) {
	       gpiv_error("\n%s: failure opening %s for input",
			  RCSID, fname_parameter);
	  }
       fprintf(fp_par_dat, "\n\n# %s\n# Command line options:\n", RCSID); 
       gpiv_img_fprint_parameters(fp_par_dat, image_par);


/*
 * Reading parametes from image header and PARFILE (and writing to data 
 * par-file)
 */
        gpiv_scan_parameter("", fname_header, &image_par, print_par);
        gpiv_scan_parameter(GPIV_IMAGE_PAR_KEY, PARFILE, &image_par, 
                            print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_IMAGE_PAR_KEY, &image_par, 
                                     print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
        gpiv_img_fprint_parameters(fp_par_dat, image_par);

        gpiv_scan_parameter(GPIV_POST_PAR_KEY, PARFILE, &post_par, print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_POST_PAR_KEY, &post_par, print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
        gpiv_post_fprint_parameters(fp_par_dat, post_par);

	fclose(fp_par_dat);


     } else {
        gpiv_scan_parameter(GPIV_IMAGE_PAR_KEY, PARFILE, &image_par, 
                            print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_IMAGE_PAR_KEY, &image_par, 
                                     print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

        gpiv_scan_parameter(GPIV_POST_PAR_KEY, PARFILE, &post_par, 
                            print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_IMAGE_PAR_KEY, &post_par, 
                                     print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
     }

    if ((err_msg = 
         gpiv_img_check_header_read(image_par))
        != NULL) gpiv_error ("%s: %s", RCSID, err_msg);

    if ((err_msg = 
         gpiv_post_check_parameters_read(&post_par, post_par_default))
        != NULL) gpiv_error ("%s: %s", RCSID, err_msg);

/*
 * Check parameters on correct values and adjust belonging 
 * variables.
 * As in_data.nx or in_data.ny are not known, the input data file will first 
 * be read
 */

     if(fname_logic == TRUE) {
	  if ((var_scale = gpiv_fcount_pivdata(fname_in, &in_data)) == -1) {
	       gpiv_error("%s: Failure calling gpiv_count_pivdata", RCSID);
	  }
     } else {
	  if ((var_scale = gpiv_count_pivdata(&in_data, d_line, &nd_lines)) == -1) {
	       gpiv_error("%s: Failure calling gpiv_count_pivdatn", RCSID);
	  }
     }


/* 
 * Now the parameters are known, data memory can be allocated
 */
     gpiv_alloc_pivdata(&in_data);
     out_data.nx = in_data.nx;
     out_data.ny = in_data.ny;
     gpiv_alloc_pivdata(&out_data);


     if(fname_logic == TRUE) {
         if ((err_msg =
              gpiv_fread_pivdata(fname_in, &in_data, c_line, &nc_lines)) 
             != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
     } else {
         gpiv_read_pivdata(&in_data, d_line, nd_lines, c_line, &nc_lines);
     }



/*
 * Here the function calls of the post-processing; scaling of PIV data
 */
     if (print_par == TRUE) printf("\n");

/* t_scale=1.0; */
     if (post_par.scale_type == GPIV_SCALE) {
         if ((var_scale = gpiv_post_scale (in_data, &out_data, image_par)) 
             != 1) {
             gpiv_error("%s: Failure calling gpiv_post_scale", RCSID);
         }
     } else if (post_par.scale_type == GPIV_SCALE_INV) {
         if ((var_scale = gpiv_post_inverse_scale (in_data, &out_data, 
                                                   image_par)) 
             != 1) {
             gpiv_error("%s: Failure calling gpiv_post_inverse_scale", RCSID);
         }
     } else {
         gpiv_error("scale:: used unexisting image_par.s_scale_logic");
     }

/*
 *  And writing to output
 */
     if (fname_logic == TRUE) {
        if ((err_msg =
             gpiv_fwrite_pivdata (fname_out, &out_data, c_line,
                                  nc_lines, var_scale, RCSID))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
     } else {
	  gpiv_write_pivdata(&out_data, c_line, nc_lines, var_scale, RCSID);
     }


/*
 * Freeing allocated memory
 */
     gpiv_free_pivdata(&in_data);
     gpiv_free_pivdata(&out_data);


     if (print_par == TRUE) printf("\n");
     exit(0);
}








   




