#include <gtk/gtk.h>
#include <cairo.h>
#include <math.h>
#include <string.h>
#include <sys/time.h>

#include "ubuntulooks_style.h"
#include "ubuntulooks_rc_style.h"
#include "ubuntulooks_draw.h"
#include "support.h"

/* #define DEBUG 1 */

#define SCALE_SIZE 5

#define DETAIL(xx)   ((detail) && (!strcmp(xx, detail)))
#define COMPARE_COLORS(a,b) (a.red == b.red && a.green == b.green && a.blue == b.blue)

#define DRAW_ARGS    GtkStyle       *style, \
                     GdkWindow      *window, \
                     GtkStateType    state_type, \
                     GtkShadowType   shadow_type, \
                     GdkRectangle   *area, \
                     GtkWidget      *widget, \
                     const gchar    *detail, \
                     gint            x, \
                     gint            y, \
                     gint            width, \
                     gint            height

#ifdef HAVE_ANIMATION
#include "animation.h"
#endif

static GtkStyleClass *parent_class;

static cairo_t *
ubuntulooks_begin_paint (GdkDrawable  *window, GdkRectangle *area)
{
    cairo_t *cr;

    cr = (cairo_t*)gdk_cairo_create (window);

    if (area) {
        cairo_rectangle (cr, area->x, area->y, area->width, area->height);
        cairo_clip (cr);
        cairo_new_path (cr);
    }

    return cr;
}

static void
ubuntulooks_set_widget_parameters (const GtkWidget      *widget,
                                  const GtkStyle       *style,
                                  GtkStateType          state_type,
                                  WidgetParameters     *params)
{
	if (widget && GTK_IS_ENTRY (widget))
		state_type = GTK_WIDGET_STATE (widget);

	params->active      = (state_type == GTK_STATE_ACTIVE);
	params->prelight    = (state_type == GTK_STATE_PRELIGHT);
	params->disabled    = (state_type == GTK_STATE_INSENSITIVE);			
	params->state_type  = (UbuntulooksStateType)state_type;
	params->corners     = CL_CORNER_ALL;
	params->enable_glow = FALSE;
		
	params->focus      = widget && GTK_WIDGET_HAS_FOCUS (widget);
	params->is_default = widget && GTK_WIDGET_HAS_DEFAULT (widget);
	params->rtl        = (cl_get_parent_direction (widget) == GTK_TEXT_DIR_RTL);
	
	if (!params->active && widget && GTK_IS_TOGGLE_BUTTON (widget))
		params->active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
		
	params->xthickness = style->xthickness;
	params->ythickness = style->ythickness;
		
	/* I want to avoid to have to do this. I need it for GtkEntry, unless I
	   find out why it doesn't behave the way I expect it to. */
	if (widget)
		ubuntulooks_get_parent_bg (widget, &params->parentbg);	
}

static void
draw_flat_box (DRAW_ARGS)
{
	UbuntulooksStyle  *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	UbuntulooksColors *colors = &ubuntulooks_style->colors;
	
	sanitize_size (window, &width, &height);

	if (DETAIL("tooltip"))
	{
		cairo_t           *cr = ubuntulooks_begin_paint (window, area);
		WidgetParameters   params;

		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		ubuntulooks_draw_tooltip (cr, colors, &params,
		                          x, y, width, height);

		cairo_destroy (cr);
	}
	else if (ubuntulooks_style->gdm && DETAIL("base"))
	{
		cairo_t           *cr = ubuntulooks_begin_paint (window, area);
		WidgetParameters   params;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		
		ubuntulooks_draw_gdm_window (cr, colors, &params,
		                             x, y, width, height);
		
		cairo_destroy (cr);
	}
	else if ((DETAIL("checkbutton") || DETAIL("radiobutton")) && state_type == CL_STATE_PRELIGHT)
	{
		/*cairo_t           *cr = ubuntulooks_begin_paint (window, area);
		WidgetParameters   params;
	
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		ubuntulooks_draw_prelightbox (cr, colors, &params,
		                              x, y, width, height);

		cairo_destroy (cr);*/
	}
	else if (detail && 	
	    state_type == GTK_STATE_SELECTED && (
	    !strncmp ("cell_even", detail, 9) ||
	    !strncmp ("cell_odd", detail, 8)))
	{
		cairo_t           *cr = ubuntulooks_begin_paint (window, area);
		WidgetParameters   params;

		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
	
		ubuntulooks_draw_list_selection (cr, colors, &params,
		                                 x, y, width, height);

		cairo_destroy (cr);
	}
	else
	{
		//printf("draw_flat_box: %s %s\n", detail, widget? G_OBJECT_TYPE_NAME (widget) : "null");
		parent_class->draw_flat_box (style, window, state_type,
		                             shadow_type,
		                             area, widget, detail,
		                             x, y, width, height);
	}
}

static void
draw_shadow (DRAW_ARGS)
{
	UbuntulooksStyle  *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	UbuntulooksColors *colors = &ubuntulooks_style->colors;
	cairo_t          *cr     = ubuntulooks_begin_paint (window, area);

	sanitize_size (window, &width, &height);

	if (DETAIL ("entry") && !(widget && widget->parent && GTK_IS_TREE_VIEW (widget->parent)))
	{
		WidgetParameters params;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		if (params.xthickness ==  1 && params.ythickness == 1)
			return draw_shadow (style, window, state_type, shadow_type, area, widget, "frame", x, y, width, height);

		if (widget && (GTK_IS_COMBO (widget->parent) ||
		               GTK_IS_COMBO_BOX_ENTRY(widget->parent) ||
		               GTK_IS_SPIN_BUTTON (widget)))
		{
			if (params.rtl)
			{
				x -= style->xthickness-1;
				width += style->xthickness-1;
				params.corners = CL_CORNER_TOPRIGHT | CL_CORNER_BOTTOMRIGHT;				
			}
			else
			{
				width += style->xthickness-1;
				params.corners = CL_CORNER_TOPLEFT | CL_CORNER_BOTTOMLEFT;
			}
		}
		
		ubuntulooks_draw_entry (cr, &ubuntulooks_style->colors, &params,
		                       x, y, width, height);
	}
	else if (DETAIL ("frame") && widget && GTK_IS_STATUSBAR (widget->parent))
	{
		WidgetParameters params;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		gtk_style_apply_default_background (style, window, TRUE, state_type,
		                                    area, x, y, width, height);
		
		ubuntulooks_draw_statusbar (cr, colors, &params,
		                           x, y, width, height);
	}
	else if (DETAIL ("frame"))
	{
		WidgetParameters params;
		FrameParameters  frame;
		frame.shadow  = shadow_type;
		frame.gap_x   = -1;                 /* No gap will be drawn */
		frame.border  = &colors->shade[4];
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		params.corners = CL_CORNER_NONE;
	
		// check if widget == null
		if (!widget || !g_str_equal ("XfcePanelWindow", gtk_widget_get_name (gtk_widget_get_toplevel (widget))))
			ubuntulooks_draw_frame (cr, colors, &params, &frame,
			                       x, y, width, height);
	}
	else if (DETAIL ("scrolled_window") || DETAIL ("viewport") || detail == NULL)
	{
		CairoColor *border = (CairoColor*)&colors->shade[5];
		cairo_rectangle (cr, x+0.5, y+0.5, width-1, height-1);
		cairo_set_source_rgb (cr, border->r, border->g, border->b);
		cairo_set_line_width (cr, 1);
		cairo_stroke (cr);
	}
	else
	{
		WidgetParameters params;
		FrameParameters frame;

		frame.shadow = shadow_type;//CL_SHADOW_IN;
		frame.gap_x  = -1;
		frame.border = &colors->shade[5];
		//printf("draw_shadow: %s %s\n", detail, widget? G_OBJECT_TYPE_NAME (widget) : "null");
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		params.corners = CL_CORNER_NONE;
		
		ubuntulooks_draw_frame (cr, colors, &params, &frame, x, y, width, height);
	}
	
	cairo_destroy (cr);
}

static void 
draw_box_gap (DRAW_ARGS,
	          GtkPositionType gap_side,
	          gint            gap_x,
	          gint            gap_width)
{
	UbuntulooksStyle  *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	UbuntulooksColors *colors = &ubuntulooks_style->colors;
	cairo_t          *cr;
	
	cr = ubuntulooks_begin_paint (window, area);

	if (DETAIL ("notebook"))
	{
		WidgetParameters params;
		FrameParameters  frame;
		
		frame.shadow    = shadow_type;
		frame.gap_side  = gap_side;
		frame.gap_x     = gap_x;
		frame.gap_width = gap_width;
		frame.border    = &colors->shade[6];
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		params.corners = CL_CORNER_NONE;
		
		ubuntulooks_draw_frame (cr, colors, &params, &frame,
		                       x, y, width, height);
	}
	else
	{
		if (widget)
			printf("box_gap: %s %s\n", detail, G_OBJECT_TYPE_NAME (widget));
		parent_class->draw_box_gap (style, window, state_type, shadow_type,
									   area, widget, detail,
									   x, y, width, height,
									   gap_side, gap_x, gap_width);
	}
	
	cairo_destroy (cr);	
}

static void
draw_extension (DRAW_ARGS, GtkPositionType gap_side)
{
	UbuntulooksStyle  *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	UbuntulooksColors *colors = &ubuntulooks_style->colors;
	cairo_t          *cr;

	cr = ubuntulooks_begin_paint (window, area);
	
	if (DETAIL ("tab"))
	{
		WidgetParameters params;
		TabParameters    tab;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		
		tab.gap_side = (UbuntulooksGapSide)gap_side;
		
		if (gap_side == GTK_POS_BOTTOM)
			params.corners = CL_CORNER_TOPLEFT | CL_CORNER_TOPRIGHT;
		else if (gap_side == GTK_POS_TOP)
			params.corners = CL_CORNER_BOTTOMLEFT | CL_CORNER_BOTTOMRIGHT;
		
		ubuntulooks_draw_tab (cr, colors, &params, &tab,
		                     x, y, width, height);
	}
	else
	{
		printf("draw_extension: %s\n", detail);
		parent_class->draw_extension (style, window, state_type, shadow_type, area,
		                              widget, detail, x, y, width, height,
		                              gap_side);

	}
	
	cairo_destroy (cr);
}

static void 
draw_handle (DRAW_ARGS, GtkOrientation orientation)
{
	UbuntulooksStyle  *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	UbuntulooksColors *colors = &ubuntulooks_style->colors;
	cairo_t          *cr;
	gboolean         is_horizontal;
	
	sanitize_size (window, &width, &height);

	cr = ubuntulooks_begin_paint (window, area);
	
	// Evil hack to work around broken orientation for toolbars
	is_horizontal = (width > height);
	
	if (DETAIL ("handlebox"))
	{
		WidgetParameters params;
		HandleParameters handle;

		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		handle.type = CL_HANDLE_TOOLBAR;
		handle.horizontal = is_horizontal;
		
		if (GTK_IS_TOOLBAR (widget) && shadow_type != GTK_SHADOW_NONE)
		{
			cairo_save (cr);
			ubuntulooks_draw_toolbar (cr, colors, &params, x, y, width, height);
			cairo_restore (cr);
		}

		ubuntulooks_draw_handle (cr, colors, &params, &handle,
		                        x, y, width, height);
	}
	else if (DETAIL ("paned"))
	{
		WidgetParameters params;
		HandleParameters handle;

		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		handle.type = CL_HANDLE_SPLITTER;
		handle.horizontal = orientation == GTK_ORIENTATION_HORIZONTAL;
			
		ubuntulooks_draw_handle (cr, colors, &params, &handle,
		                        x, y, width, height);
	}
	else
	{
		WidgetParameters params;
		HandleParameters handle;

		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		handle.type = CL_HANDLE_TOOLBAR;
		handle.horizontal = is_horizontal;
		
		if (GTK_IS_TOOLBAR (widget) && shadow_type != GTK_SHADOW_NONE)
		{
			cairo_save (cr);
			ubuntulooks_draw_toolbar (cr, colors, &params, x, y, width, height);
			cairo_restore (cr);
		}
		
		ubuntulooks_draw_handle (cr, colors, &params, &handle,
		                        x, y, width, height);
	}

	cairo_destroy (cr);
}

static int
combo_box_get_seperator_pos (GtkWidget *widget)
{
	GList *children, *children_start;
	int pos = 0;
	
	if (widget && GTK_IS_COMBO_BOX (widget->parent))
	{
		children = children_start = gtk_container_get_children (GTK_CONTAINER (widget));
		
		if (children && GTK_IS_HBOX (children->data))
			children = gtk_container_get_children (GTK_CONTAINER (children->data));

		if (children)
		{
			do
			{
				if (GTK_IS_ARROW (children->data) || GTK_IS_VSEPARATOR (children->data))
					pos += GTK_WIDGET (children->data)->allocation.width;
				
			} while ((children = g_list_next (children)));
		}
	}
	
	pos += 2; /* don't ask me why.... widget->style->xthickness - 1 ? */
	
	g_list_free (children_start);	

	return pos;
}

static void
draw_box (DRAW_ARGS)
{
	UbuntulooksStyle *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	const UbuntulooksColors *colors;
	cairo_t *cr;

	cr     = ubuntulooks_begin_paint (window, area);
	colors = &ubuntulooks_style->colors;

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)   
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	if (DETAIL ("menubar") && 
	    !(widget && (cl_is_panel_widget (widget->parent))))
	{
		WidgetParameters params;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		ubuntulooks_draw_menubar (cr, colors, &params,
		                         x, y, width, height);
	}
	else if (DETAIL ("button") && widget && widget->parent &&
                 (GTK_IS_TREE_VIEW(widget->parent) ||
                  GTK_IS_CLIST (widget->parent)))
	{
		WidgetParameters params;
		ListViewHeaderParameters header;
		
		gint columns, column_index;
		gboolean resizable = TRUE;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		
		params.corners = CL_CORNER_NONE;
		
		if (GTK_IS_TREE_VIEW (widget->parent))
		{
			gtk_treeview_get_header_index (GTK_TREE_VIEW(widget->parent),
										   widget, &column_index, &columns,
										   &resizable);
		}
		else if (GTK_IS_CLIST (widget->parent))
		{
			gtk_clist_get_header_index (GTK_CLIST(widget->parent),
										widget, &column_index, &columns);
		}
		
		header.resizable = resizable;
		
		if (column_index == 0)
			header.order = CL_ORDER_FIRST;
		else if (column_index == columns-1)
			header.order = CL_ORDER_LAST;
		else
			header.order = CL_ORDER_MIDDLE;
		
		if (params.rtl && header.order == CL_ORDER_FIRST)
			header.order = CL_ORDER_LAST;
		else if (params.rtl && header.order == CL_ORDER_LAST)
			header.order = CL_ORDER_FIRST;
		
		gtk_style_apply_default_background (style, window, FALSE, state_type, area, x, y, width, height);
		
		ubuntulooks_draw_list_view_header (cr, colors, &params, &header,
		                                  x, y, width, height);
	}
	else if (DETAIL ("button") || DETAIL ("buttondefault"))
	{
		WidgetParameters params;
		ShadowParameters shadow = { CL_CORNER_ALL, CL_SHADOW_NONE } ;
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		if ((widget && (GTK_IS_COMBO_BOX_ENTRY (widget->parent) || GTK_IS_COMBO (widget->parent))))
		{
			if (params.rtl)
				params.corners = CL_CORNER_TOPLEFT | CL_CORNER_BOTTOMLEFT;
			else
				params.corners = CL_CORNER_TOPRIGHT | CL_CORNER_BOTTOMRIGHT;
			
			shadow.shadow = CL_SHADOW_IN;

			if (params.xthickness > 2)
			{
				if (!params.rtl)
					x--;
				width++;
			}			
		}
		else
		{
			params.corners     = CL_CORNER_ALL;
			params.enable_glow = TRUE;
		}
	
		if (GTK_IS_TOGGLE_BUTTON (widget) &&
		    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
			params.active = TRUE;
		
		// Fix some firefox crap.
		if (GTK_IS_BUTTON (widget) && GTK_IS_FIXED (widget->parent) && widget->allocation.x == -1 &&  widget->allocation.y == -1)
		{
			gtk_style_apply_default_background (widget->parent->style, window, TRUE, GTK_STATE_NORMAL,
			                                    area, x, y, width, height);
		}
		
		shadow.corners = params.corners;
		
		ubuntulooks_draw_button (cr, &ubuntulooks_style->colors, &params, &shadow,
		                        x, y, width, height);
	}
	else if (DETAIL ("spinbutton_up"))
	{
		if (state_type == GTK_STATE_ACTIVE)
		{
			WidgetParameters params;
			
			params.state_type = (UbuntulooksStateType)state_type;
			params.rtl        = (cl_get_parent_direction (widget) == GTK_TEXT_DIR_RTL);
			params.corners    = params.rtl ? CL_CORNER_TOPLEFT : CL_CORNER_TOPRIGHT;
			
			if (style->xthickness == 3)
			{
				if (params.rtl)
					width+=2;
				else
				{
					x--;
					width++;
				}
			}
			
			height+=2;
			
			
			ubuntulooks_draw_spinbutton_down (cr, &ubuntulooks_style->colors, &params,
											 x, y, width, height);
		}
	}
	else if (DETAIL ("spinbutton_down"))
	{
		if (state_type == GTK_STATE_ACTIVE)
		{
			WidgetParameters params;
			
			params.state_type = (UbuntulooksStateType)state_type;
			params.rtl        = (cl_get_parent_direction (widget) == GTK_TEXT_DIR_RTL);
			params.corners    = params.rtl ? CL_CORNER_BOTTOMLEFT : CL_CORNER_BOTTOMRIGHT;
			
			if (style->xthickness ==3)
			{
				if (params.rtl)
					width += 2;
				else
				{
					x--;
					width++;
				}
			}
			ubuntulooks_draw_spinbutton_down (cr, &ubuntulooks_style->colors, &params,
											 x, y, width, height);
		}
	}
	else if (DETAIL ("spinbutton"))
	{
		WidgetParameters params;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		
		if (params.rtl)
			params.corners = CL_CORNER_TOPLEFT | CL_CORNER_BOTTOMLEFT;
		else
			params.corners = CL_CORNER_TOPRIGHT | CL_CORNER_BOTTOMRIGHT;
		
		if (style->xthickness == 3)
		{
			if (params.rtl)
			{
				width++;
			}
			else
			{
				x--;
				width++;
			}
		}
		
		ubuntulooks_draw_spinbutton (cr, &ubuntulooks_style->colors, &params,
		                            x, y, width, height);
	}
	else if (DETAIL ("trough") && widget && GTK_IS_SCALE (widget))
	{
		GtkAdjustment *adjustment = gtk_range_get_adjustment (GTK_RANGE (widget));	
		WidgetParameters params;
		SliderParameters slider;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		
		params.corners    = CL_CORNER_NONE;
		
		slider.inverted   = gtk_range_get_inverted (GTK_RANGE (widget));
		
		slider.horizontal = (GTK_RANGE (widget)->orientation == GTK_ORIENTATION_HORIZONTAL);
		slider.fill_size  = (slider.horizontal ? width: height) * (1 / ((adjustment->upper - adjustment->lower) / (adjustment->value - adjustment->lower)));	

		if (slider.horizontal && params.rtl)
			slider.inverted = !slider.inverted;
		
		ubuntulooks_draw_scale_trough (cr, &ubuntulooks_style->colors,
		                              &params, &slider,
		                              x, y, width, height);
	}
	else if (DETAIL ("trough") && widget && GTK_IS_PROGRESS_BAR (widget))
	{
		WidgetParameters params;
		ProgressBarParameters progressbar;

		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);		
		
		if (widget && GTK_IS_PROGRESS_BAR (widget))
			progressbar.orientation = gtk_progress_bar_get_orientation (GTK_PROGRESS_BAR (widget));
		else
			progressbar.orientation = CL_ORIENTATION_LEFT_TO_RIGHT;

		ubuntulooks_draw_progressbar_trough (cr, colors, &params, &progressbar, 
		                                    x, y, width, height);
	}
	else if (DETAIL ("trough") && widget && (GTK_IS_VSCROLLBAR (widget) || GTK_IS_HSCROLLBAR (widget)))
	{
		WidgetParameters params;
		ScrollBarParameters scrollbar;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		params.corners = CL_CORNER_NONE;
		
		scrollbar.horizontal = TRUE;
		scrollbar.junction   = scrollbar_get_junction (widget);
		scrollbar.steppers   = scrollbar_visible_steppers (widget);
		
		if (GTK_IS_RANGE (widget))
			scrollbar.horizontal = GTK_RANGE (widget)->orientation == GTK_ORIENTATION_HORIZONTAL;
		
		if (scrollbar.horizontal)
		{
			x += 2;
			width -= 4;
		}
		else
		{
			y += 2;
			height -= 4;
		}
		
		ubuntulooks_draw_scrollbar_trough (cr, colors, &params, &scrollbar,
		                                  x, y, width, height);
	}
	else if (DETAIL ("bar"))
	{
		WidgetParameters      params;
		ProgressBarParameters progressbar;
		gdouble               elapsed = 0.0;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		if (widget && GTK_IS_PROGRESS_BAR (widget))
			progressbar.orientation = gtk_progress_bar_get_orientation (GTK_PROGRESS_BAR (widget));
		else
			progressbar.orientation = CL_ORIENTATION_LEFT_TO_RIGHT;
		
		ubuntulooks_draw_progressbar_fill (cr, colors, &params, &progressbar,
		                                  x, y, width, height,
		                                  10 - (int)(elapsed * 10.0) % 10);
	}
	else if (DETAIL ("hscale") || DETAIL ("vscale"))
	{
		WidgetParameters params;
		SliderParameters slider;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		
		slider.horizontal = DETAIL ("hscale");
		
		ubuntulooks_draw_slider_button (cr, &ubuntulooks_style->colors,
		                               &params, &slider,
		                               x, y, width, height);
	}
	else if (DETAIL ("optionmenu"))
	{
		WidgetParameters params;
		OptionMenuParameters optionmenu;
		
		GtkRequisition indicator_size;
		GtkBorder indicator_spacing;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		option_menu_get_props (widget, &indicator_size, &indicator_spacing);
		
		params.enable_glow = TRUE;
		
		if (widget && get_direction (widget) == GTK_TEXT_DIR_RTL)
			optionmenu.linepos = (indicator_size.width + indicator_spacing.left + indicator_spacing.right) + style->xthickness;
		else
			optionmenu.linepos = width - (indicator_size.width + indicator_spacing.left + indicator_spacing.right) - style->xthickness;

		ubuntulooks_draw_optionmenu (cr, colors, &params, &optionmenu,
		                            x, y, width, height);
	}
	else if (DETAIL ("menuitem"))
	{
		WidgetParameters params;
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		
		if (widget && GTK_IS_MENU_BAR (widget->parent))
		{
			/*params.corners = CL_CORNER_NONE;
			params.active = TRUE;
			params.state_type = CL_STATE_ACTIVE;
			params.xthickness = 2;
			params.ythickness = 2;
			
			ubuntulooks_draw_button (cr, colors, &params, x, y, width, height+1);*/
			
			params.corners = CL_CORNER_TOPLEFT | CL_CORNER_TOPRIGHT;
			height += 1;
		}
		else
		{	
			params.corners = CL_CORNER_ALL;
		}
		
		if (ubuntulooks_style->gdm)
		{
			cairo_translate (cr, x, y);
			cairo_rectangle (cr, 0, 0, width, height);
		
			cairo_set_source_rgba (cr, 1.0, 1.0, 1.0, 0.4);
			cairo_fill (cr);
		}
		else
		{
			ubuntulooks_draw_menuitem (cr, colors, &params, x, y, width, height);
		}
	}
	else if (DETAIL ("hscrollbar") || DETAIL ("vscrollbar") || DETAIL ("slider") || DETAIL ("stepper"))
	{
		WidgetParameters    params;
		ScrollBarParameters scrollbar;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		params.corners = CL_CORNER_NONE;
		
		scrollbar.has_color  = FALSE;
		scrollbar.horizontal = TRUE;
		scrollbar.junction   = scrollbar_get_junction (widget);
		scrollbar.steppers   = scrollbar_visible_steppers (widget);
		
		if (GTK_IS_RANGE (widget))
			scrollbar.horizontal = GTK_RANGE (widget)->orientation == GTK_ORIENTATION_HORIZONTAL;
		
		if (ubuntulooks_style->has_scrollbar_color)
		{
			ubuntulooks_gdk_color_to_rgb (&ubuntulooks_style->scrollbar_color, &scrollbar.color.r, &scrollbar.color.g, &scrollbar.color.b);
			scrollbar.has_color = TRUE;
		}
		
		if (DETAIL ("slider"))
		{
			ubuntulooks_draw_scrollbar_slider (cr, colors, &params, &scrollbar,
			                                  x, y, width, height);
		}
		else
		{
			ScrollBarStepperParameters stepper;
			GdkRectangle this_rectangle = { x, y, width, height };

			stepper.stepper = scrollbar_get_stepper (widget, &this_rectangle);

			ubuntulooks_draw_scrollbar_stepper (cr, colors, &params, &scrollbar, &stepper,
			                                   x, y, width, height);
		}
	}
	else if (DETAIL ("toolbar") || DETAIL ("handlebox_bin") || DETAIL ("dockitem_bin"))
	{
		// Only draw the shadows on horizontal toolbars
		if (shadow_type != GTK_SHADOW_NONE && height < 2*width )
			ubuntulooks_draw_toolbar (cr, colors, NULL, x, y, width, height);
	}
	else if (DETAIL ("trough"))
	{
		
	}
	else if (DETAIL ("menu"))
	{
		WidgetParameters params;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		
/*		if (ubuntulooks_style->gdm)
		{
			ubuntulooks_draw_gdm_menu_frame (cr, colors, &params, x, y, width, height);
		}
		else*/
		{
			CairoColor *bg = (CairoColor*)&colors->bg[state_type];
			cairo_set_source_rgb (cr, bg->r, bg->g, bg->b);
			cairo_rectangle (cr, x, y, width, height);
			cairo_fill (cr);
			
			ubuntulooks_draw_menu_frame (cr, colors, &params, x, y, width, height);
		}
	}
	else
	{
		//printf("draw_box: %s\n", detail);
		parent_class->draw_box (style, window, state_type, shadow_type, area,
		                        widget, detail, x, y, width, height);
	}
	
	cairo_destroy (cr);
}

static void
draw_slider (DRAW_ARGS, GtkOrientation orientation)
{
	if (DETAIL ("hscale") || DETAIL ("vscale"))
	{
		draw_box (style, window, state_type, shadow_type, area,
		          widget, detail, x, y, width, height);
	}
	else
		parent_class->draw_slider (style, window, state_type, shadow_type, area,
		                           widget, detail, x, y, width, height, orientation);
}

static void
draw_option (DRAW_ARGS)
{
	UbuntulooksStyle *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	const UbuntulooksColors *colors;
	cairo_t *cr = ubuntulooks_begin_paint (window, area);
	colors = &ubuntulooks_style->colors;
	
	WidgetParameters params;
	CheckboxParameters checkbox;
	
	checkbox.shadow_type = shadow_type;
	checkbox.in_menu = (widget && GTK_IS_MENU(widget->parent));
		
	ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
	
	ubuntulooks_draw_radiobutton (cr, colors, &params, &checkbox, x, y, width, height);

	cairo_destroy (cr);
}

static void
draw_check (DRAW_ARGS)
{
	UbuntulooksStyle *ubuntulooks_style = UBUNTULOOKS_STYLE (style);

	cairo_t *cr = ubuntulooks_begin_paint (window, area);
	
	WidgetParameters params;
	CheckboxParameters checkbox;
		
	ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
	
	params.corners = CL_CORNER_ALL;
	
	checkbox.shadow_type = shadow_type;
	checkbox.in_cell = DETAIL("cellcheck");

	checkbox.in_menu = (widget && widget->parent && GTK_IS_MENU(widget->parent));

	ubuntulooks_draw_checkbox (cr, &ubuntulooks_style->colors, &params, &checkbox,
	                          x, y, width, height);
	
	cairo_destroy (cr);
}

static void
draw_tab (DRAW_ARGS)
{
	UbuntulooksStyle *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	UbuntulooksColors *colors = &ubuntulooks_style->colors;	
	WidgetParameters params;
	ArrowParameters  arrow;
	
	cairo_t *cr = ubuntulooks_begin_paint (window, area);
	
	ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
	arrow.type      = CL_ARROW_COMBO;
	arrow.direction = CL_DIRECTION_DOWN;
	
	ubuntulooks_draw_arrow (cr, colors, &params, &arrow, x, y, width, height);

	cairo_destroy (cr);
}

static void
draw_vline                      (GtkStyle               *style,
                                 GdkWindow              *window,
                                 GtkStateType            state_type,
                                 GdkRectangle           *area,
                                 GtkWidget              *widget,
                                 const gchar            *detail,
                                 gint                    y1,
                                 gint                    y2,
                                 gint                    x)
{
	SeparatorParameters separator = { FALSE };
	cairo_t *cr;

	cr = ubuntulooks_begin_paint (window, area);
	
	ubuntulooks_draw_separator (cr, NULL, NULL, &separator,
	                           x, y1, 2, y2-y1);
	
	cairo_destroy (cr);
}

static void
draw_hline                      (GtkStyle               *style,
                                 GdkWindow              *window,
                                 GtkStateType            state_type,
                                 GdkRectangle           *area,
                                 GtkWidget              *widget,
                                 const gchar            *detail,
                                 gint                    x1,
                                 gint                    x2,
                                 gint                    y)
{
	cairo_t *cr;

	cr = ubuntulooks_begin_paint (window, area);
	
	if (DETAIL ("label")) /* wtf? */
	{
		printf("draw_vline: label. ermm....?\n");
	}
	else
	{
		SeparatorParameters separator;
		
		separator.horizontal = TRUE;
		
		ubuntulooks_draw_separator (cr, NULL, NULL, &separator,
		                           x1, y, x2-x1, 2);
	}
	
	cairo_destroy (cr);
}

static void 
draw_shadow_gap (DRAW_ARGS,
                 GtkPositionType gap_side,
                 gint            gap_x,
                 gint            gap_width)
{
	UbuntulooksStyle *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	const UbuntulooksColors *colors;
	cairo_t *cr;

	cr     = ubuntulooks_begin_paint (window, area);
	colors = &ubuntulooks_style->colors;
	
	if (DETAIL ("frame"))
	{
		WidgetParameters params;
		FrameParameters  frame;
		
		frame.shadow    = shadow_type;
		frame.gap_side  = gap_side;
		frame.gap_x     = gap_x;
		frame.gap_width = gap_width;
		frame.border    = (CairoColor*)&colors->shade[5];
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);

		params.corners = CL_CORNER_NONE;
		
		ubuntulooks_draw_frame (cr, colors, &params, &frame,
		                       x, y, width, height);
	}
	else
	{
		parent_class->draw_shadow_gap (style, window, state_type, shadow_type, area,
									   widget, detail, x, y, width, height,
									   gap_side, gap_x, gap_width);
	}
	
	cairo_destroy (cr);
}

static void
draw_resize_grip (GtkStyle       *style,
                  GdkWindow      *window,
                  GtkStateType    state_type,
                  GdkRectangle   *area,
                  GtkWidget      *widget,
                  const gchar    *detail,
                  GdkWindowEdge   edge,
                  gint            x,
                  gint            y,
                  gint            width,
                  gint            height)
{
	UbuntulooksStyle *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	UbuntulooksColors *colors = &ubuntulooks_style->colors;

	cairo_t *cr = ubuntulooks_begin_paint (window, area);

	WidgetParameters params;
	ResizeGripParameters grip;
	grip.edge = (UbuntulooksWindowEdge)edge;

	int lx, ly;

	g_return_if_fail (window != NULL);

	if (edge != GDK_WINDOW_EDGE_SOUTH_EAST)
		return; // sorry... need to work on this :P

	ubuntulooks_set_widget_parameters (widget, style, state_type, &params);	

	ubuntulooks_draw_resize_grip (cr, colors, &params, &grip,
	                             x, y, width, height);

	cairo_destroy (cr);

	return;
}

static void
draw_arrow (GtkStyle  *style,
                       GdkWindow     *window,
                       GtkStateType   state_type,
                       GtkShadowType  shadow,
                       GdkRectangle  *area,
                       GtkWidget     *widget,
                       const gchar   *detail,
                       GtkArrowType   arrow_type,
                       gboolean       fill,
                       gint           x,
                       gint           y,
                       gint           width,
                       gint           height)
{
	UbuntulooksStyle  *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	UbuntulooksColors *colors = &ubuntulooks_style->colors;
	cairo_t *cr = ubuntulooks_begin_paint (window, area);
		
	sanitize_size (window, &width, &height);

	if (DETAIL ("arrow"))
	{
		WidgetParameters params;
		ArrowParameters  arrow;
		
		ubuntulooks_set_widget_parameters (widget, style, state_type, &params);
		arrow.type = CL_ARROW_NORMAL;
		arrow.direction = (UbuntulooksDirection)arrow_type;
		
/*		cairo_rectangle (cr, x, y, width, height);
		cairo_set_source_rgb (cr, 1, 0, 0);
		cairo_fill (cr);
	*/	
		if (cl_is_combo_box (widget))
		{
			arrow.type = CL_ARROW_COMBO;
			y -= 2;
			height += 4;
			x += 1;
		}

		ubuntulooks_draw_arrow (cr, colors, &params, &arrow,
		                       x, y, width, height);
	}
	else
	{
		//printf("draw_arrow: %s %s\n", detail, widget ? G_OBJECT_TYPE_NAME (widget) : "null");

		parent_class->draw_arrow (style, window, state_type, shadow, area,
		                          widget, detail, arrow_type, fill,
		                          x, y, width, height);
	}
	
	cairo_destroy (cr);
}

static void
ubuntulooks_style_init_from_rc (GtkStyle * style,
			       GtkRcStyle * rc_style)
{
	UbuntulooksStyle *ubuntulooks_style = UBUNTULOOKS_STYLE (style);
	double shades[] = {1.15, 0.95, 0.896, 0.82, 0.7, 0.665, 0.5, 0.45, 0.4};
	CairoColor spot_color;
	CairoColor bg_normal;
	double contrast;
	int i;
	
	parent_class->init_from_rc (style, rc_style);

	bg_normal.r = style->bg[0].red   / 65535.0;
	bg_normal.g = style->bg[0].green / 65535.0;
	bg_normal.b = style->bg[0].blue  / 65535.0;

	contrast = UBUNTULOOKS_RC_STYLE (rc_style)->contrast;
	
	ubuntulooks_style->progressbarstyle  = UBUNTULOOKS_RC_STYLE (rc_style)->progressbarstyle;
	ubuntulooks_style->menubarstyle      = UBUNTULOOKS_RC_STYLE (rc_style)->menubarstyle;
	ubuntulooks_style->menuitemstyle     = UBUNTULOOKS_RC_STYLE (rc_style)->menuitemstyle;
	ubuntulooks_style->listviewitemstyle = UBUNTULOOKS_RC_STYLE (rc_style)->listviewitemstyle;
	ubuntulooks_style->has_scrollbar_color = UBUNTULOOKS_RC_STYLE (rc_style)->has_scrollbar_color;
	ubuntulooks_style->animation         = UBUNTULOOKS_RC_STYLE (rc_style)->animation;
	ubuntulooks_style->gdm               = UBUNTULOOKS_RC_STYLE (rc_style)->gdm;
	
	if (ubuntulooks_style->has_scrollbar_color)
		ubuntulooks_style->scrollbar_color = UBUNTULOOKS_RC_STYLE (rc_style)->scrollbar_color;
	
	/* Lighter to darker */
	for (i = 0; i < 9; i++)
	{
		ul_shade (&bg_normal, &ubuntulooks_style->colors.shade[i],
		       (shades[i]-0.7) * contrast + 0.7);
	}
		
	spot_color.r = style->bg[GTK_STATE_SELECTED].red   / 65535.0;
	spot_color.g = style->bg[GTK_STATE_SELECTED].green / 65535.0;
	spot_color.b = style->bg[GTK_STATE_SELECTED].blue  / 65535.0;
	
	ul_shade (&spot_color, &ubuntulooks_style->colors.spot[0], 1.42);
	ul_shade (&spot_color, &ubuntulooks_style->colors.spot[1], 1.05);
	ul_shade (&spot_color, &ubuntulooks_style->colors.spot[2], 0.65);
	
	for (i=0; i<5; i++)
	{
		ubuntulooks_gdk_color_to_rgb (&style->bg[i],
		                             &ubuntulooks_style->colors.bg[i].r,
		                             &ubuntulooks_style->colors.bg[i].g,
		                             &ubuntulooks_style->colors.bg[i].b);

		ubuntulooks_gdk_color_to_rgb (&style->base[i],
		                             &ubuntulooks_style->colors.base[i].r,
		                             &ubuntulooks_style->colors.base[i].g,
		                             &ubuntulooks_style->colors.base[i].b);

		ubuntulooks_gdk_color_to_rgb (&style->text[i],
		                             &ubuntulooks_style->colors.text[i].r,
		                             &ubuntulooks_style->colors.text[i].g,
		                             &ubuntulooks_style->colors.text[i].b);
	}
}

static void
gdk_cairo_set_source_color_alpha (cairo_t  *cr,
                GdkColor *color, float alpha)
{
  g_return_if_fail (cr != NULL);
  g_return_if_fail (color != NULL);

  cairo_set_source_rgba (cr,
            color->red / 65535.,
            color->green / 65535.,
            color->blue / 65535.,
	        alpha);
}

static void
draw_focus (GtkStyle *style, GdkWindow *window, GtkStateType state_type,
            GdkRectangle *area, GtkWidget *widget, const gchar *detail,
            gint x, gint y, gint width, gint height)
{
	cairo_t *cr;
	gboolean free_dash_list = FALSE;
	gint line_width = 1;
	gint8 *dash_list = "\1\1";

	if (widget)
	{
		gtk_widget_style_get (widget,
				      "focus-line-width", &line_width,
				      "focus-line-pattern",
				      (gchar *) & dash_list, NULL);

		free_dash_list = TRUE;
	}

	if (detail && !strcmp (detail, "add-mode"))
	{
		if (free_dash_list)
			g_free (dash_list);

		dash_list = "\4\4";
		free_dash_list = FALSE;
	}

	sanitize_size (window, &width, &height);

	cr = gdk_cairo_create (window);

	if (detail && !strcmp (detail, "colorwheel_light"))
		cairo_set_source_rgb (cr, 0., 0., 0.);
	else if (detail && !strcmp (detail, "colorwheel_dark"))
		cairo_set_source_rgb (cr, 1., 1., 1.);
	else
		gdk_cairo_set_source_color_alpha (cr, &style->fg[state_type],
						  0.7);

	cairo_set_line_width (cr, line_width);

	if (dash_list[0])
	{
		gint n_dashes = strlen (dash_list);
		gdouble *dashes = g_new (gdouble, n_dashes);
		gdouble total_length = 0;
		gdouble dash_offset;
		gint i;

		for (i = 0; i < n_dashes; i++)
		{
			dashes[i] = dash_list[i];
			total_length += dash_list[i];
		}

		/* The dash offset here aligns the pattern to integer pixels
		 * by starting the dash at the right side of the left border
		 * Negative dash offsets in cairo don't work
		 * (https://bugs.freedesktop.org/show_bug.cgi?id=2729)
		 */
		dash_offset = -line_width / 2.;
		while (dash_offset < 0)
			dash_offset += total_length;

		cairo_set_dash (cr, dashes, n_dashes, dash_offset);
		g_free (dashes);
	}

	if (area)
	{
		gdk_cairo_rectangle (cr, area);
		cairo_clip (cr);
	}

	cairo_rectangle (cr,
			 x + line_width / 2.,
			 y + line_width / 2.,
			 width - line_width, height - line_width);
	cairo_stroke (cr);
	cairo_destroy (cr);

	if (free_dash_list)
		g_free (dash_list);
}

static void
ubuntulooks_style_copy (GtkStyle * style, GtkStyle * src)
{
	UbuntulooksStyle * cl_style = UBUNTULOOKS_STYLE (style);
	UbuntulooksStyle * cl_src = UBUNTULOOKS_STYLE (src);
	
	cl_style->colors              = cl_src->colors;
	cl_style->progressbarstyle    = cl_src->progressbarstyle;
	cl_style->menubarstyle        = cl_style->menubarstyle;
	cl_style->menuitemstyle       = cl_style->menuitemstyle;
	cl_style->listviewitemstyle   = cl_style->listviewitemstyle;
	cl_style->scrollbar_color     = cl_style->scrollbar_color;
	cl_style->has_scrollbar_color = cl_style->has_scrollbar_color;
	cl_style->animation           = cl_style->animation;
	cl_style->gdm                 = cl_style->gdm;
	
	parent_class->copy (style, src);
}

static void
ubuntulooks_style_unrealize (GtkStyle * style)
{
	parent_class->unrealize (style);
}

static GdkPixbuf *
set_transparency (const GdkPixbuf *pixbuf, gdouble alpha_percent)
{
	GdkPixbuf *target;
	guchar *data, *current;
	guint x, y, rowstride, height, width;

	g_return_val_if_fail (pixbuf != NULL, NULL);
	g_return_val_if_fail (GDK_IS_PIXBUF (pixbuf), NULL);

	/* Returns a copy of pixbuf with it's non-completely-transparent pixels to
	   have an alpha level "alpha_percent" of their original value. */

	target = gdk_pixbuf_add_alpha (pixbuf, FALSE, 0, 0, 0);

	if (alpha_percent == 1.0)
		return target;
	width = gdk_pixbuf_get_width (target);
	height = gdk_pixbuf_get_height (target);
	rowstride = gdk_pixbuf_get_rowstride (target);
	data = gdk_pixbuf_get_pixels (target);

	for (y = 0; y < height; y++) {
		for (x = 0; x < width; x++) {
			/* The "4" is the number of chars per pixel, in this case, RGBA,
			   the 3 means "skip to the alpha" */
			current = data + (y * rowstride) + (x * 4) + 3; 
			*(current) = (guchar) (*(current) * alpha_percent);
		}
	}

	return target;
}

static GdkPixbuf*
scale_or_ref (GdkPixbuf *src,
              int width,
              int height)
{
	if (width == gdk_pixbuf_get_width (src) &&
	    height == gdk_pixbuf_get_height (src)) {
		return g_object_ref (src);
	} else {
		return gdk_pixbuf_scale_simple (src,
                                        width, height,
                                        GDK_INTERP_BILINEAR);
	}
}

static GdkPixbuf *
render_icon (GtkStyle            *style,
             const GtkIconSource *source,
             GtkTextDirection     direction,
             GtkStateType         state,
             GtkIconSize          size,
             GtkWidget           *widget,
             const char          *detail)
{
	int width = 1;
	int height = 1;
	GdkPixbuf *scaled;
	GdkPixbuf *stated;
	GdkPixbuf *base_pixbuf;
	GdkScreen *screen;
	GtkSettings *settings;
	
	/* Oddly, style can be NULL in this function, because
	 * GtkIconSet can be used without a style and if so
	 * it uses this function.
	 */
	
	base_pixbuf = gtk_icon_source_get_pixbuf (source);
	
	g_return_val_if_fail (base_pixbuf != NULL, NULL);
	
	if (widget && gtk_widget_has_screen (widget)) {
		screen = gtk_widget_get_screen (widget);
		settings = gtk_settings_get_for_screen (screen);
	} else if (style->colormap) {
		screen = gdk_colormap_get_screen (style->colormap);
		settings = gtk_settings_get_for_screen (screen);
	} else {
		settings = gtk_settings_get_default ();
		GTK_NOTE (MULTIHEAD,
			  g_warning ("Using the default screen for gtk_default_render_icon()"));
	}
	
  
	if (size != (GtkIconSize) -1 && !gtk_icon_size_lookup_for_settings (settings, size, &width, &height)) {
		g_warning (G_STRLOC ": invalid icon size '%d'", size);
		return NULL;
	}

	/* If the size was wildcarded, and we're allowed to scale, then scale; otherwise,
	 * leave it alone.
	 */
	if (size != (GtkIconSize)-1 && gtk_icon_source_get_size_wildcarded (source))
		scaled = scale_or_ref (base_pixbuf, width, height);
	else
		scaled = g_object_ref (base_pixbuf);
	
	/* If the state was wildcarded, then generate a state. */
	if (gtk_icon_source_get_state_wildcarded (source)) {
		if (state == GTK_STATE_INSENSITIVE) {
			stated = set_transparency (scaled, 0.3);
#if 0
			stated =
				gdk_pixbuf_composite_color_simple (scaled,
								   gdk_pixbuf_get_width (scaled),
								   gdk_pixbuf_get_height (scaled),
								   GDK_INTERP_BILINEAR, 128,
								   gdk_pixbuf_get_width (scaled),
								   style->bg[state].pixel,
								   style->bg[state].pixel);
#endif
			gdk_pixbuf_saturate_and_pixelate (stated, stated,
							  0.1, FALSE);
			
			g_object_unref (scaled);
		} else if (state == GTK_STATE_PRELIGHT) {
			stated = gdk_pixbuf_copy (scaled);      
			
			gdk_pixbuf_saturate_and_pixelate (scaled, stated,
							  1.2, FALSE);
			
			g_object_unref (scaled);
		} else {
			stated = scaled;
		}
	}
	else
		stated = scaled;
  
  return stated;
}

static void
ubuntulooks_style_init (UbuntulooksStyle * style)
{
}

static void
ubuntulooks_style_class_init (UbuntulooksStyleClass * klass)
{
	GtkStyleClass *style_class = GTK_STYLE_CLASS (klass);
	
	parent_class = g_type_class_peek_parent (klass);

	style_class->copy             = ubuntulooks_style_copy;
	style_class->unrealize        = ubuntulooks_style_unrealize;
	style_class->init_from_rc     = ubuntulooks_style_init_from_rc;
	style_class->draw_handle      = draw_handle;
	style_class->draw_slider      = draw_slider;
	style_class->draw_shadow_gap  = draw_shadow_gap;
	style_class->draw_focus       = draw_focus;
	style_class->draw_box         = draw_box;
	style_class->draw_shadow      = draw_shadow;
	style_class->draw_box_gap     = draw_box_gap;
	style_class->draw_extension   = draw_extension;
	style_class->draw_option      = draw_option;
	style_class->draw_check       = draw_check;
	style_class->draw_flat_box    = draw_flat_box;
	style_class->draw_tab         = draw_tab;
	style_class->draw_vline       = draw_vline;
	style_class->draw_hline       = draw_hline;
	style_class->draw_resize_grip = draw_resize_grip;
	style_class->draw_arrow       = draw_arrow;
	style_class->render_icon      = render_icon;
}

GType ubuntulooks_type_style = 0;

void
ubuntulooks_style_register_type (GTypeModule * module)
{
	static const GTypeInfo object_info =
	{
		sizeof (UbuntulooksStyleClass),
		(GBaseInitFunc) NULL,
		(GBaseFinalizeFunc) NULL,
		(GClassInitFunc) ubuntulooks_style_class_init,
		NULL,         /* class_finalize */
		NULL,         /* class_data */
		sizeof (UbuntulooksStyle),
		0,            /* n_preallocs */
		(GInstanceInitFunc) ubuntulooks_style_init,
		NULL
	};

	ubuntulooks_type_style = g_type_module_register_type (module,
	                                                     GTK_TYPE_STYLE,
	                                                     "UbuntulooksStyle",
	                                                     &object_info, 0);
}

