/* relay.c
 * - Relay functions *
 * Copyright (c) 1999 Jack Moffitt, Barath Raghavan, and Alexander Havng
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#ifdef _WIN32
#include <win32config.h>
#else
#include <config.h>
#endif
#endif

#include "definitions.h"

#include <stdio.h>
#include <errno.h>
# ifndef __USE_BSD
#  define __USE_BSD
# endif
#ifndef __EXTENSIONS__
#define __EXTENSIONS__
#endif

#include <string.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <time.h>
#include <fcntl.h>
#include <ctype.h>

#ifndef _WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#else
#include <winsock.h>
#endif

#include "avl.h"
#include "threads.h"
#include "icetypes.h"
#include "icecast.h"
#include "utility.h"
#include "ice_string.h"
#include "connection.h"
#include "relay.h"
#include "threads.h"
#include "client.h"
#include "sock.h"
#include "ice_resolv.h"
#include "source.h"
#include "log.h"
#include "memory.h"
#include "commands.h"
#include "vars.h"
#include "logtime.h"
#include "pool.h"

extern int running;
extern server_info_t info;
extern mutex_t library_mutex;
extern mutex_t authentication_mutex;

/* 
 * Add a pulling relay to the list of relays automagically connected now and then.
 * Possible error codes:
 * ICE_ERROR_INVALID_SYNTAX    - Invalid syntax or too few arguments 
 * ICE_ERROR_NO_SUCH_MOUNT     - Trying to push a nonexistant mount
 * ICE_ERROR_ARGUMENT_REQUIRED - The given option requires an argument
 * Assert Class: 3
 */
int
relay_add_pull_to_list (char *arg)
{
	relay_t *new;
	char localmount[BUFSIZE];
	request_t req;

	if (!arg || !arg[0])
		return ICE_ERROR_INVALID_SYNTAX;
	
	xa_debug (2, "DEBUG: Adding [%s] to list of pulling relays", arg);

	localmount[0] = '\0';

	if (arg[0] == '-' && arg[1] == 'm') {
		localmount[0] = '\0';
		if (splitc (localmount, arg, ' ') == NULL) {
			return ICE_ERROR_ARGUMENT_REQUIRED;
		}
		
		generate_http_request (arg, &req);
		
		if (!req.path[0]) {
			return ICE_ERROR_INVALID_SYNTAX;
		}
	}
	
	new = relay_create ();

	generate_http_request (arg, &(new->req));

	if (localmount[0])
		strncpy(new->localmount, localmount, BUFSIZE);
	else
		snprintf(new->localmount, BUFSIZE, "%s:%d%s", new->req.host, new->req.port, new->req.path);

	relay_insert (new);

	xa_debug (3, "DEBUG: Insertion of pulling relay succeeded", arg);

	return OK;
}

/* 
 * Add a pushing relay to the list of relays automagically connected now and then.
 * User Syntax: relay push <sourceid/localmount> <host[:port]/mount> [options] 
 * Possible error codes:
 * ICE_ERROR_INVALID_SYNTAX    - Invalid syntax or too few arguments 
 * ICE_ERROR_NO_SUCH_MOUNT     - Trying to push a nonexistant mount
 * ICE_ERROR_ARGUMENT_REQUIRED - The given option requires an argument
 * Assert Class: 3
 */
int
relay_add_push_to_list (char *arg)
{
	char id_or_localmount[BUFSIZE], password[BUFSIZE], url[BUFSIZE], description[BUFSIZE], name[BUFSIZE], 
		genre[BUFSIZE], mimetype[BUFSIZE], buf[BUFSIZE], carg[BUFSIZE];
	request_t req;
	int bitrate, public, go_on = 0;
	relay_t *relay = NULL;
	connection_t *con = NULL;
	source_t *source;

	if (!arg || !arg[0] || (splitc (id_or_localmount, arg, ' ') == NULL)) {
		return ICE_ERROR_INVALID_SYNTAX;
	}

	xa_debug (2, "DEBUG: Adding [%s %s] to list of pushing relays", id_or_localmount, arg);

	generate_http_request (arg, &req);

	if (!is_valid_http_request (&req)) {
		return ICE_ERROR_INVALID_SYNTAX;
	}
	
	if (isdigit ((int)id_or_localmount[0]))
		con = find_source_with_id (atoi (id_or_localmount));
	else
		con = find_source_with_mount (id_or_localmount);

	if (!con)
		return ICE_ERROR_NO_SUCH_MOUNT;

	source = con->food.source;

	relay = relay_create ();

	strncpy(relay->localmount, con->food.source->audiocast.mount, BUFSIZE);
	generate_http_request (arg, &relay->req);
	
	strncpy (password, info.encoder_pass, BUFSIZE);
	strncpy (url, source->audiocast.url, BUFSIZE);
	strncpy (description, source->audiocast.description, BUFSIZE);
	strncpy (name, source->audiocast.name, BUFSIZE);
	strncpy (genre, source->audiocast.genre, BUFSIZE);
	strncpy (mimetype, source->audiocast.streammimetype, BUFSIZE);

	bitrate = source->audiocast.bitrate;
	public = source->audiocast.public;
	
	do
	{
		if (splitc (buf, arg, ' ') == NULL)
		{
			go_on = 0;
			strncpy(buf, arg, BUFSIZE);
		} else {
			if (buf[0] == '-' && buf[1] != 'i' && splitc (carg, arg, ' ') != NULL)
			{
				if (carg[0] == '-' || (ice_strcmp (carg, buf) == 0))
					return ICE_ERROR_ARGUMENT_REQUIRED;
			} else {
				strncpy(carg, arg, BUFSIZE);
			}
		}
		
		if (buf[0] == '-')
		{
			xa_debug (2, "DEBUG: Relay push arguments, buf: [%s] arg: [%s] carg: [%s]", buf, arg, carg);
			
			switch (buf[1])
			{
				case 'p':
					strncpy (password, carg, BUFSIZE);
					break;
				case 'u':
					strncpy (url, carg, BUFSIZE);
					break;
				case 'n':
					strncpy (name, carg, BUFSIZE);
					break;
				case 'g':
					strncpy (genre, carg, BUFSIZE);
					break;
				case 'P':
					public = atoi (carg);
					break;
				case 'b':
					bitrate = atoi (carg);
					break;
				case 'd':
					strncpy (description, carg, BUFSIZE);
					break;
				case 'i':
					relay->protocol = icy_e;
					break;
				case 'm':
					strncpy (mimetype, carg, BUFSIZE);
					break;
			}
		}
	} while (go_on);

	relay->password = nstrdup (password);
	relay->audiocast.url = nstrdup (url);
	relay->audiocast.name = nstrdup (name);
	relay->audiocast.genre = nstrdup (genre);
	relay->audiocast.public = public;
	relay->audiocast.bitrate = bitrate;
	relay->audiocast.description = description;
	relay->audiocast.streammimetype = mimetype;
	relay->audiocast.contentid = nstrdup (con->food.source->audiocast.contentid);

	relay->type = relay_push_e;

	relay_insert (relay);

	xa_debug (3, "DEBUG: Insertion of pushing relay succeeded");

	return OK;
}

/*
 * Remove the relay from list of relays
 * Possible error codes:
 * ICE_ERROR_NOT_FOUND - No such relay in list
 * ICE_ERROR_NULL - Argument was NULL
 * Assert Class: 1
 */
int
relay_remove (relay_t *relay)
{
	relay_t *out;

	if (!relay) {
		write_log (LOG_DEFAULT, "WARNING: Trying to remove a NULL relay from tree");
		return ICE_ERROR_NULL;
	}

	thread_mutex_lock (&info.relay_mutex);
	out = avl_delete (info.relays, relay);
	thread_mutex_unlock (&info.relay_mutex);

	if (out) {
		xa_debug (2, "DEBUG: Removed relay [%s:%d%s]", out->req.host, out->req.port, out->req.path);
		relay_dispose (out);
		return OK;
	}
	
	return ICE_ERROR_NOT_FOUND;
}

int
relay_remove_with_con (connection_t *con)
{
	relay_t *rel = relay_find_with_con (con);

	if (rel)
		return relay_remove (rel);
	return ICE_ERROR_NOT_FOUND;
}

int
relay_remove_with_req (request_t *req)
{
	relay_t *rel = relay_find_with_req (req);
	
	if (rel)
		return relay_remove (rel);
	return ICE_ERROR_NOT_FOUND;
}

/*
 * Find a relay given a request struct 
 * Assert Class: 2
 */
relay_t *
relay_find_with_req (request_t *req)
{
	relay_t rel, *out = NULL;

	strncpy(rel.req.host, req->host, BUFSIZE);
	rel.req.port = req->port;
	strncpy(rel.req.path, req->path, BUFSIZE);
	
	thread_mutex_lock (&info.relay_mutex);
	out = avl_find (info.relays, &rel);
	thread_mutex_unlock (&info.relay_mutex);
	
	return out;
}

/*
 * Find a relay given a connection struct
 * Assert Class: 2
 */
relay_t *
relay_find_with_con (connection_t *con)
{
	avl_traverser trav = {0};
	relay_t *rel;

	if (!con)
		return NULL;

	while ((rel = avl_traverse (info.relays, &trav))) {
		if (rel->con == con)
			break;
	}

	if (rel && rel->con == con)
		return rel;
	return NULL;
}


/*
 * Insert a relay struct into the list of relays
 * Assert Class: 0
 */
void
relay_insert (relay_t *relay)
{
	relay_t *out;
	
	if (!relay)
	{
		write_log (LOG_DEFAULT, "WARNING: Trying to insert NULL relay into relay tree");
		return;
	}

	relay->reconnect_now = 1;

	thread_mutex_lock (&info.relay_mutex);
	out = avl_replace (info.relays, relay);
	thread_mutex_unlock (&info.relay_mutex);
	
	if (out)
		relay_dispose (out);
}

/*
 * Free dynamic memory in a relay struct
 * Assert Class: 1
 */
void
relay_dispose (relay_t *relay)
{
	if (relay->con)
		relay->con = NULL; /* Paranoia */
	if (relay->password)
		nfree (relay->password);
	dispose_audiocast (&relay->audiocast);
	nfree (relay);
}

/* 
 * Dynamically create a relay struct
 * Assert Class: 1
 */
relay_t *
relay_create ()
{
	relay_t *relay = (relay_t *) nmalloc (sizeof (relay_t));
	relay->con = NULL;
	relay->localmount[0] = '\0';
	relay->reconnects = 0;
	relay->last_reconnect = (time_t) 0;
	relay->password = NULL;
	relay->type = relay_pull_e;
	relay->protocol = xaudiocast_e;
	zero_audiocast (&relay->audiocast);

	return relay;
}

/*
 * Run through the list of relays, and connect the not connected ones
 * according to relay type.
 * Assert Class: 2
 */
void 
relay_connect_all_relays ()
{
	avl_traverser trav = {0};
	relay_t *rel = NULL;
	time_t now;

	xa_debug (4, "DEBUG: Reconnecting unconnected relays....");
	
	thread_mutex_lock (&info.double_mutex);
	thread_mutex_lock (&info.relay_mutex);

	if (!info.relays) {
		write_log (LOG_DEFAULT, "WARNING: info.relays is NULL, weeird!");
	}

	while ((rel = avl_traverse (info.relays, &trav)))
	{
		now = get_time ();
		if (!relay_connected (rel) && rel->reconnect_now) {
			xa_debug (3, "DEBUG: Immediately connecting relay");
			rel->reconnects++;
			rel->last_reconnect = now;
			relay_connect_list_item (rel);
			rel->reconnect_now = 0;
		} else if ((!relay_connected (rel)) && 
			   ((rel->last_reconnect == 0) || (rel->last_reconnect + info.relay_reconnect_time) < now) &&
			   ((info.relay_reconnect_tries == -1) || (rel->reconnects < info.relay_reconnect_tries)))
		{
			rel->reconnects++;
			rel->last_reconnect = now;
			relay_connect_list_item (rel);
			rel->reconnect_now = 0;
		}
	}

	thread_mutex_unlock (&info.relay_mutex);
	thread_mutex_unlock (&info.double_mutex);
	
	xa_debug (4, "DEBUG: Done reconnecting relays.");
}

/*
 * Is relay connected?
 * Assert Class: 0
 */
int
relay_connected (relay_t *rel)
{
	if (!rel) {
		write_log (LOG_DEFAULT, "WARNING: relay_connected(): Relay is NULL!");
		return 0;
	}
	
	if (rel->con)
		return 1;
	else
		return 0;
}

/* 
 * Connect a single relay from the list, according to type
 * Assert Class: 1
 */
void
relay_connect_list_item (relay_t *rel)
{
	int err;

	xa_debug (2, "DEBUG: Reconnecting relay [%s:%d%s]", rel->req.host, rel->req.port, rel->req.path);
	
	if (rel->type == relay_pull_e)
		rel->con = relay_connect_pull (rel, &err);
	else
		rel->con = relay_connect_push (rel, &err);
	
	if (rel->con) {
		rel->reconnects = 0;
		rel->last_reconnect = 0;
	} else if ((rel->reconnects > info.relay_reconnect_tries) && (info.relay_reconnect_tries != -1)) {
		write_log (LOG_DEFAULT, "Relay for [%s:%d%s] has exceeded the maximum number of reconnections.", rel->req.host, 
			   rel->req.port, rel->req.path);
	}
		
	xa_debug (3, "DEBUG: Reconnect %s (%d)", rel->con ? "succeeded" : "failed", err);
}

/* 
 * Given a relay struct, connect this, login, and add as a client
 * Error codes in err variable, one of following:
 * ICE_ERROR_CONNECT - Connection failed
 * ICE_ERROR_TRANSMISSION - Connection ok, but failed to read data
 * ICE_ERROR_INVALID_PASSWORD
 * Returns connection_t pointer to new connection, or NULL if failed.
 * Assert Class: 4
 */
connection_t *
relay_connect_push (relay_t *relay, int *err)
{
	SOCKET sockfd;
	connection_t *relaycon;
	connection_t *con;
	char buf[BUFSIZE];

	*err = OK;
	
	xa_debug (2, "Connecting source %s to [%s:%d%s]", relay->localmount, relay->req.host, relay->req.port, relay->req.path);
	
	if ((sockfd = sock_connect_wto (relay->req.host, relay->req.port, 15)) == -1) {
		*err = ICE_ERROR_CONNECT;
		return NULL;
	}
	
	if (relay->protocol == icy_e)
	{
		sock_write_line (sockfd, "%s", relay->password);
		
		if (recv (sockfd, buf, 100, 0) < 0)
		{
			sock_close(sockfd);
			*err = ICE_ERROR_TRANSMISSION;
			return NULL;
		}
		
		if (buf[0] != 'O' && buf[0] != 'o')
		{
			sock_close(sockfd);
			*err = ICE_ERROR_INVALID_PASSWORD;
			return NULL;
		}
		
		sock_write_line (sockfd, "icy-name:%s", relay->audiocast.name);
		sock_write_line (sockfd, "icy-genre:%s", relay->audiocast.genre);
		sock_write_line (sockfd, "icy-url:%s", relay->audiocast.url);
		sock_write_line (sockfd, "icy-pub:%d", relay->audiocast.public);
		sock_write_line (sockfd, "icy-br:%d", relay->audiocast.bitrate);
	} else {
		sock_write_line (sockfd, "SOURCE %s %s\r\n", relay->password, relay->req.path);
		sock_write_line (sockfd, "x-audiocast-name:%s", relay->audiocast.name);
		sock_write_line (sockfd, "x-audiocast-genre:%s", relay->audiocast.genre);
		sock_write_line (sockfd, "x-audiocast-url:%s", relay->audiocast.url);
		sock_write_line (sockfd, "x-audiocast-public:%d", relay->audiocast.public);
		sock_write_line (sockfd, "x-audiocast-bitrate:%d", relay->audiocast.bitrate);
		sock_write_line (sockfd, "x-audiocast-description:%s", relay->audiocast.description);
		sock_write_line (sockfd, "x-audiocast-mimetype:%s", relay->audiocast.streammimetype);
		sock_write_line (sockfd, "x-audiocast-contentid:%s", relay->audiocast.contentid);
		sock_write_line (sockfd, "x-audiocast-type:relay\r\n");
		
		if (recv (sockfd, buf, 100, 0) < 0) {
			sock_close (sockfd);
			*err = ICE_ERROR_TRANSMISSION;
			return NULL;
		}
		
		/* HTTP/1.0 200 OK || ICY 200 OK */
		if (buf[0] != 'O' && buf[0] != 'o') {
			sock_close(sockfd);
			*err = ICE_ERROR_INVALID_PASSWORD;
			return NULL;
		}
	}
	
	relaycon = create_connection ();
	relaycon->sock = sockfd;
	sock_set_blocking(relaycon->sock, SOCK_BLOCK);
	
	relaycon->host = nstrdup (relay->req.host);
	if (info.reverse_lookups)
		relaycon->hostname = reverse (relaycon->host);

	relaycon->id = new_id ();

	relaycon->connect_time = time (NULL);
	put_client (relaycon);
	relaycon->food.client->virgin = 1;
	relaycon->food.client->write_bytes = 0;
	relaycon->food.client->type = pusher_e;
	relaycon->headervars = create_header_vars ();
	add_varpair2 (relaycon->headervars, nstrdup ("x-audiocast-source-password"), nstrdup (relay->password));
	add_varpair2 (relaycon->headervars, nstrdup ("x-audiocast-port"), ice_itoa (relay->req.port));
	add_varpair2 (relaycon->headervars, nstrdup ("x-audiocast-mount"), nstrdup (relay->req.path));

	con = find_source_with_mount (relay->localmount);

	if (!con)
	{
		write_log (LOG_DEFAULT, "WARNING: Trying to insert pushing relay for nonexitant source");
		return relaycon;
	}

	relaycon->food.client->source = con->food.source;
	pool_add (relaycon);
	
	util_increase_total_clients ();
	
	return relaycon;
}

/* 
 * Called when spawning an alias or in transparent proxy mode 
 * Errors in err, forwarded from relay_connect_pull()
 * Assert Class: 2
 */
connection_t *
relay_pull_stream (request_t *req, int *err)
{
	connection_t *con;
	relay_t rel;
	xa_debug (2, "DEBUG: relay_pull_stream called with [%s:%d%s]", req->host, req->port, req->path);
	
	rel.req = *req;

	if (!(con = relay_connect_pull (&rel, err))) {
		return NULL;
	}

	xa_debug (3, "DEBUG: relay_pull_stream() succeeded");

	return con;
}

/* 
 * Initiate a pulling relay connection, login and add to list of sources
 * Errors in err, one of following:
 * ICE_ERROR_INIT_FAILED - Failed to initialize relay resources
 * ICE_ERROR_INSERT_FAILED - Failed to insert new source in sourcetree (login failure...)
 * Assert Class: 4
 */
connection_t *
relay_connect_pull (relay_t *relay, int *err)
{
	connection_t *newcon = NULL, *con = NULL;
	int id;
	char buf[BUFSIZE];

	*err = OK;

	/* Setup the connection with sockets and stuff */
	if (!relay_setup_connection (&newcon, &relay->req)) {
		xa_debug (4, "WARNING: relay_setup_connection() failed");
		*err = ICE_ERROR_INIT_FAILED;
		return NULL;
	}

	id = newcon->id;

	/* Make the connection, login with headers, parse the headers */
	*err = login_as_client_on_server (newcon, &relay->req, buf);
	if (*err < 0) {
		xa_debug (4, "WARNING: login_as_client_on_server() failed (%d)", *err);
		return NULL;
	}
	
	/* Initialize the connection as a source on our icecast server */
	source_login (newcon, buf);

	/* Make sure it got in */
	if ((con = find_source_with_id (id))) {
		return con;
	}
	
	xa_debug (4, "WARNING: find_source_with_id() didn't find new source");

	*err = ICE_ERROR_INSERT_FAILED;
	return NULL;
}

/*
 * Login as a client on a given server
 * Returns OK or one of the following errors:
 * ICE_ERROR_CONNECT - Connection failed 
 * ICE_ERROR_TRANSMISSION - Connect worked, but failed to read data
 * ICE_ERROR_HEADER - Invalid headers received from server
 * Assert Class: 3
 */
int
login_as_client_on_server (connection_t *con, request_t *req, char *buf)
{
	SOCKET sockfd;

	if ((sockfd = sock_connect_wto (req->host, req->port, 15)) == -1) {
		xa_debug (4, "WARNING: sock_connect_wto() to [%s:%d] failed", req->host, req->port);
		return ICE_ERROR_CONNECT;
	}

	con->sock = sockfd;
	con->connect_time = get_time ();

	snprintf(buf, BUFSIZE, "%s:%d%s", req->host, req->port, req->path);

	con->food.source->audiocast.mount = nstrdup (buf);
	
	/* This comes after, so we don't get kicked out */
	sock_write_line (sockfd, "GET %s HTTP/1.0\r\nHOST: %s", req->path, req->host);
	sock_write_line (sockfd, "User-Agent: icecast/%s\r\nReferer: RELAY", VERSION);
	sock_write_line (sockfd, "x-audiocast-port: %d", info.port[0]);
	sock_write_line (sockfd, "x-audiocast-source-password: %s\r\n", info.encoder_pass);

	/* Here the remote server prints audiocast headers or
	   some kind of error */
	if (!sock_read_lines (sockfd, buf, BUFSIZE))
	{
		kick_connection (con, "Error in read");
		return ICE_ERROR_TRANSMISSION;
	}

	/* If there was an error, we need to deal with it here */
	{
		char *sptr = strchr (buf, ' ');
		if (!sptr)
		{
			kick_connection (con, "Erroneous header");
			return ICE_ERROR_HEADER;
		}
		if ((ice_strncmp (buf, "HTTP/1.0 200", 12) != 0) && (ice_strncmp (buf, "ICY 200", 7) != 0))
		{
			kick_connection (con, "Error in request, relay refused entrance");
			return ICE_ERROR_HEADER;
		}
	}

	if (ice_strncmp (buf, "ICY", 3) == 0)
	{
		con->food.source->protocol = icy_e;
	} else {
		char *ptr = strstr (buf, "Server:");
		if (ptr)
		{
			if ((ice_strncmp (ptr + 1, " icecast/1.1", 12) == 0) ||
			    (ice_strncmp (ptr + 1, " icecast/1.2", 12) == 0))
				con->food.source->protocol = icy_e;
			else
				con->food.source->protocol = xaudiocast_e;
		} else {
			con->food.source->protocol = icy_e;
		}
	}

	return OK;
}

/*
 * Allocate and innitiate a relay connection and request struct
 * Assert Class: 2
 */
int
relay_setup_connection (connection_t **con, request_t *req)
{
	*con = create_connection ();
	(*con)->type = source_e;
	(*con)->host = forward (req->host, (char *) nmalloc (20));
	if ((*con)->host == NULL)
		(*con)->host = nstrdup (req->host);
	(*con)->id = new_id ();
	if (info.reverse_lookups)
		(*con)->hostname = reverse ((*con)->host);
	
	put_source (*con);
	(*con)->food.source->type = puller_e;
	return OK;
}













