/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iclipplane.h"


#include "iactor.h"
#include "ierror.h"
#include "itransform.h"
#include "iviewmodule.h"

#include <vtkCellArray.h>
#include <vtkMath.h>
#include <vtkMatrix4x4.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>
#include <vtkProperty.h>


//
//  Main class
//
iClipPlane* iClipPlane::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return new iClipPlane(vm);
}


iClipPlane::iClipPlane(iViewModule *vm) : mViewModule(vm)
{
	int i, j;
	//
	// Set plane parameters
	//	
	vtkPolyData *pd = vtkPolyData::New(); IERROR_ASSERT(pd);
	mClipPlaneActor = iActor::New(); IERROR_ASSERT(mClipPlaneActor);
	
	static float x1[8][3] = { {1.0,0.0,0.0}, {0.7071,0.7071,0.0}, {0.0,1.0,0.0}, {-0.7071,0.7071,0.0}, 
	{-1.0,0.0,0.0}, {-0.7071,-0.7071,0.0}, {0.0,-1.0,0.0}, {0.7071,-0.7071,0.0} };
	static vtkIdType pts1[8] = { 0, 1, 2, 3, 4, 5, 6, 7 };
	
	for(i=0; i<8; i++) for(j=0; j<3; j++) x1[i][j] *= 2.5;
	
	vtkPoints *points = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT(points);
	for(i=0; i<8; i++) points->InsertPoint(i,x1[i]);
	pd->SetPoints(points);
	points->Delete();
	
	vtkCellArray *polys = vtkCellArray::New(); IERROR_ASSERT(polys);
	polys->InsertNextCell(8,pts1);
	pd->SetPolys(polys);
	polys->Delete();
	
	pd->Update();
	mClipPlaneActor->SetInput(pd);	
	pd->Delete();

	mClipPlaneActor->VisibilityOff();
	mClipPlaneActor->PickableOff();
	
	mClipPlaneActor->GetProperty()->SetOpacity(0.5);
	mClipPlaneActor->GetProperty()->SetAmbient(1.0);
	mClipPlaneActor->GetProperty()->SetDiffuse(1.0);
	mClipPlaneActor->GetProperty()->SetSpecular(0.7);
	mClipPlaneActor->GetProperty()->SetSpecularPower(50.0);
	mClipPlaneActor->GetProperty()->SetColor(0.6,0.7,0.7);

	mTransform = iTransform::New(); IERROR_ASSERT(mTransform);
	mClipPlaneActor->SetUserTransform(mTransform);

	this->GetViewModule()->AddObject(mClipPlaneActor);

	mDistance = 0.0f;
	float dir[3] = { 1.0, 0.0, 0.0 };
	this->SetDirection(dir);
}


iClipPlane::~iClipPlane()
{
	this->GetViewModule()->RemoveObject(mClipPlaneActor);
	mClipPlaneActor->Delete();
	mTransform->Delete();
}


void iClipPlane::SetGlassPlaneVisible(bool s)
{
	mClipPlaneActor->SetVisibility(s?1:0);
}


bool iClipPlane::GetGlassPlaneVisible() const
{
	return mClipPlaneActor->GetVisibility() != 0;
}


void iClipPlane::SetDirection(const float *dir)
{
	int i;
	double x[3];
	double n[3];

	for(i=0; i<3; i++) mClipPlaneDirection[i] = dir[i];
	vtkMath::Normalize(mClipPlaneDirection);
	for(i=0; i<3; i++) n[i] = -mClipPlaneDirection[i];
	this->SetNormal(n);
	for(i=0; i<3; i++) x[i] = -mDistance*n[i];
	this->SetOrigin(x);
	//
	//  Move the glass plane
	//
	mTransform->Identity();
	mTransform->SetDirection(mClipPlaneDirection);
	mTransform->Translate(0.0,0.0,mDistance);
}


void iClipPlane::SetDistance(float l)
{
	int i;
	double x[3];
	double n[3];
	this->GetNormal(n);

	mDistance = l;

	for(i=0; i<3; i++) x[i] = -l*n[i];
	this->SetOrigin(x);
	mTransform->Identity();
	mTransform->SetDirection(mClipPlaneDirection);
	mTransform->Translate(0.0,0.0,mDistance);
}
