/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IDATALIMITS_H
#define IDATALIMITS_H


#include <vtkObjectBase.h>


#include "iarray.h"
#include "imath.h"
#include "iparameter.h"
#include "istring.h"

#include <vtkSetGet.h>

class iDataSubject;
class iDataType;


class iDataLimits : public vtkObjectBase
{

	friend class iDataSubject;

public:
	
	vtkTypeMacro(iDataLimits,vtkObjectBase);
	static iDataLimits* New(const iDataSubject *subject = 0, int num = 0, const iString &name = "");

	int GetNumVars() const;
	int GetStretch(int n) const;
	float GetMax(int n) const;
	float GetMin(int n) const;
	float GetUpperLimit(int n) const;
	float GetLowerLimit(int n) const;
	const iString& GetName(int n) const;
	const iString& GetUnit(int n) const;
	bool GetFixedStretch(int n) const;
	int GetClass(int n) const;
	const iString& GetNameForClass(int n) const; // GetClassName conflicts with something under VC++

	void SetMax(int n, float v);
	void SetMin(int n, float v);
	void SetName(int n, const iString &v);
	void SetStretch(int n, int v);
	void SetUpperLimit(int n, float v);
	void SetLowerLimit(int n, float v);

	void ResetVars();
	void AddVar(int n);
	void AssignVars(int n = -1); // add n first listed Vars

	void BackupVars();
	void RestoreVars();

	inline bool AreNamesFixed() const { return mAreNamesFixed; }

	const iDataType& GetDataType() const;

	//
	//  Helper functions for using non-limits Lower & Upper Limits
	//
	void UpdateLowerLimit(float v, int n, float &ll, float &ul);
	void UpdateUpperLimit(float v, int n, float &ll, float &ul);

	//
	//  Boundary conditions are set by the Subject
	//
	void GetPeriodicities(bool per[3]) const;
	bool IsBoxPeriodic() const;

	void BlockNotifications(bool b);

	//
	//  Expandability
	//
	void Resize(int n);
	inline int GetSize() const { return mListedVars.Size(); }

protected:
	
	iDataLimits(const iDataSubject *subject, int num, const iString &name, bool resizable, bool namesfixed);
	virtual ~iDataLimits();

	struct Var
	{
		bool Included, FixedStretch;
		float Min, Max;
		float LowerLimit, UpperLimit;
		int Stretch, Class;
		iString Name, Unit, ClassName;
	};

	virtual void NotifyDependencies(int var);

	void Configure(int num);
	void Initialize(Var &v);
	void AdjustMinMax(Var &v);

	iString mNamePrefix;
	const iDataSubject *mSubject;
	iSearchableArray<Var*> mVars, mVarsBackup;
	iArray<Var> mListedVars; 

	bool mBlockNotifications;
	const bool mIsResizable, mAreNamesFixed;
	const int mMinNumListed;

private:

	//
	//  For internal testing
	//
	int Invalid(int) const;
	float Invalid(float) const;
	const iString& Invalid(const iString &) const;
	bool mBlockChecking;
};


inline int iDataLimits::GetNumVars() const 
{ 
	return mVars.Size();
}


inline int iDataLimits::GetStretch(int n) const
{
	if(n>=0 && n<mVars.Size()) return mVars[n]->Stretch; else return this->Invalid(0);
}


inline float iDataLimits::GetMax(int n) const
{
	if(n>=0 && n<mVars.Size()) return mVars[n]->Max; else return this->Invalid(iMath::_LargeFloat);
}


inline float iDataLimits::GetMin(int n) const
{
	if(n>=0 && n<mVars.Size()) return mVars[n]->Min; else return this->Invalid(-iMath::_LargeFloat);
}


inline float iDataLimits::GetUpperLimit(int n) const
{
	if(n>=0 && n<mVars.Size()) return mVars[n]->UpperLimit; else return this->Invalid(iMath::_LargeFloat);
}


inline float iDataLimits::GetLowerLimit(int n) const
{
	if(n>=0 && n<mVars.Size()) return mVars[n]->LowerLimit; else return this->Invalid(-iMath::_LargeFloat);
}


inline const iString& iDataLimits::GetName(int n) const
{
	static const iString null;
	if(n>=0 && n<mVars.Size()) return mVars[n]->Name; else return this->Invalid(null);
}


inline const iString& iDataLimits::GetUnit(int n) const
{
	static const iString null;
	if(n>=0 && n<mVars.Size()) return mVars[n]->Unit; else return this->Invalid(null);
}


inline bool iDataLimits::GetFixedStretch(int n) const
{
	if(n>=0 && n<mVars.Size()) return mVars[n]->FixedStretch; else return (this->Invalid(0) == 0);
}


inline int iDataLimits::GetClass(int n) const
{
	if(n>=0 && n<mVars.Size()) return mVars[n]->Class; else return this->Invalid(-1);
}


inline const iString& iDataLimits::GetNameForClass(int n) const
{
	static const iString null;
	if(n>=0 && n<mVars.Size()) return mVars[n]->ClassName; else return this->Invalid(null);
}


inline void iDataLimits::BlockNotifications(bool b)
{ 
	mBlockNotifications = b;
}

#endif // IDATALIMITS_H

