/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "irendertoolbackground.h"


#include "ierror.h"

#include <vtkActor2D.h>
#include <vtkImageData.h>
#include <vtkImageMapper.h>
#include <vtkImageResample.h>
#include <vtkProperty2D.h>
#include <vtkRenderer.h>


namespace iRenderToolBackground_Private
{
	class ImageMapper : public vtkImageMapper
	{

	public:

		static ImageMapper* New()
		{
			return new ImageMapper;
		}

		virtual void RenderOverlay(vtkViewport *viewport, vtkActor2D *mActor)
		{
			this->RenderStart(viewport,mActor);
		}

		virtual void RenderData(vtkViewport *viewport, vtkImageData *data, vtkActor2D *mActor)
		{
			int dims[3], *size = viewport->GetSize();
			data->GetDimensions(dims);

			mFilter->SetInput(this->GetInput());

			mFilter->SetAxisMagnificationFactor(0,float(size[0])/dims[0]);
			mFilter->SetAxisMagnificationFactor(1,float(size[1])/dims[1]);

			mFilter->Update();
			mWorker->RenderOverlay(viewport,mActor);
		}

	protected:
		
		ImageMapper()
		{
			mFilter = vtkImageResample::New(); IERROR_ASSERT(mFilter);
			mWorker = vtkImageMapper::New(); IERROR_ASSERT(mWorker);

			mWorker->SetColorWindow(255.0);
			mWorker->SetColorLevel(127.5);
			mWorker->RenderToRectangleOn();
			mWorker->UseCustomExtentsOff();
	
			mFilter->SetDimensionality(2);
			mWorker->SetInput(mFilter->GetOutput());
		}

		~ImageMapper()
		{
			mFilter->Delete();
			mWorker->Delete();
		}

		vtkImageResample *mFilter;
		vtkImageMapper *mWorker;
	};
};


using namespace iRenderToolBackground_Private;


//
//  Main class
//
iRenderToolBackground* iRenderToolBackground::New()
{
	return new iRenderToolBackground;
}


iRenderToolBackground::iRenderToolBackground()
{
	mRenderer = vtkRenderer::New(); IERROR_ASSERT(mRenderer);

	mActor = vtkActor2D::New(); IERROR_ASSERT(mActor);
	mMapper = ImageMapper::New(); IERROR_ASSERT(mMapper);

	mMapper->SetColorWindow(255.0);
	mMapper->SetColorLevel(127.5);
	mMapper->RenderToRectangleOn();
	mMapper->UseCustomExtentsOff();

	mActor->SetMapper(mMapper);
	mMapper->Delete();
	mActor->GetPositionCoordinate()->SetCoordinateSystemToNormalizedDisplay();
	mActor->SetPosition(0.0,0.0);
	mActor->GetPosition2Coordinate()->SetCoordinateSystemToNormalizedDisplay();
	mActor->SetPosition2(1.0,1.0);
	mActor->VisibilityOff();
	mActor->PickableOff();

	mActor->GetProperty()->SetDisplayLocationToBackground();

	mRenderer->AddActor2D(mActor);
	mRenderer->SetLayer(IRENDERTOOLBACKGROUND_BACK_LAYER);
	mRenderer->SetInteractive(0);
}


iRenderToolBackground::~iRenderToolBackground()
{
	mRenderer->RemoveActor2D(mActor);
	mRenderer->Delete();

	mActor->Delete();
	mMapper->Delete();
}


void iRenderToolBackground::SetColor(const iColor &color)
{
	if(color != mColor)
	{
		mColor = color;
		mRenderer->SetBackground(color.ToVTK());
	}
}


void iRenderToolBackground::SetImage(const iImage &image)
{
	if(image != mImage)
	{
		if(image.IsEmpty())
		{
			mImage.Clear();
			mActor->VisibilityOff();
		}
		else
		{
			mImage = image;
			mMapper->SetInput(mImage.DataObject());
			mActor->VisibilityOn();
		}
	}
}

