/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iscriptkit.h"


#include "ierror.h"
#include "iscript.h"


//
//  Templates
//
#include "iarraytemplate.h"
#include "icalculatortemplate.h"
#include "iscriptkittemplate.h"


namespace iScriptKit
{
#ifdef I_DEBUG
	int Value::mIdCounter = 0;
#endif

	//
	//  Calculator
	//
	Calculator::Calculator(iScript *script) : iCalculator<double>(script==0?0:script->GetCalculator()), mScript(script)
	{
		IERROR_ASSERT(mScript);
		this->SetTrapFPE(true);
	}


	bool Calculator::Verify(const iString &expr, int dim, int use, const iString &err)
	{
		const char *s[2] = { "numeric (0)", "boolean (1)" };

		if(!this->Compile(expr) || !mScript->SatisfyRequestForVariables(this->ListUsedVariables()) || !this->Link()) return false;

		if(use > 1) use = 0; // just in case...

		if(use>-1 && this->GetResult()->Use()!=use)
		{
			this->SetError("The usage pattern of the "+err+" expression must be "+s[use]+" instead of "+s[this->GetResult()->Use()]+".",-1);
			return false;
		}
		if(dim>-1 && this->GetResult()->Dim()!=dim)
		{
			this->SetError("The dimension of the "+err+" expression must be "+iString::FromNumber(dim)+" instead of "+iString::FromNumber(this->GetResult()->Dim())+".",-1);
			return false;
		}
		return true;
	}


	void Calculator::AddVariable(const result_t *var)
	{
		this->iCalculator<double>::AddVariable(var);
	}


	//
	//  Base class for values (variables and parameters)
	//
	Value::Value(iScript *script, const iString &name, int dim, int use) : mScript(script)
	{
		IERROR_ASSERT(mScript);

		mIsReleased = false;

		if(dim < 1)
		{
			//
			//  For non-positive dimension we create a morphable scalar
			//
			dim = 1;
			mValue = new iCalculatorKit::var_<Calculator::number_t,true>(dim,use,name); IERROR_ASSERT(mValue);
		}
		else
		{
			mValue = new iCalculatorKit::var_<Calculator::number_t,false>(dim,use,name); IERROR_ASSERT(mValue);
		}
		if(mScript->GetCalculator() != 0) mScript->GetCalculator()->AddVariable(mValue);

#ifdef I_DEBUG
		mId = ++mIdCounter;
		iConsole::PrintDebugMessage("iScriptKit::Value: creating #"+iString::FromNumber(mId)+" ("+name+")");
#endif
	}


	Value::~Value()
	{
#ifdef I_DEBUG
		mIdCounter--;
		iConsole::PrintDebugMessage("iScriptKit::Value: deleting #"+iString::FromNumber(mId)+" ("+this->Name()+"), "+iString::FromNumber(mIdCounter)+" left.");
#endif
		if(mScript->GetCalculator() != 0) mScript->GetCalculator()->RemoveVariable(mValue->Name());
		delete mValue;
	}


	bool Value::Morph(int dim)
	{
		return mValue->Morph(dim,mValue->Use());
	}


	const iString& Value::Name() const
	{
		return mValue->Name();
	}

	
	int Value::Dim() const
	{
		return mValue->Dim();
	}


	int Value::Use() const
	{
		return mValue->Use();
	}


	void Value::ReportError(const iString &str)
	{
		mScript->ReportError(str,-1,-1);
	}


	//
	//  Base class for prototypes.
	//
	Prototype::Prototype(iScript *script, const iString &command) : mCommand(command), mScript(script)
	{
		IERROR_ASSERT(mScript);
		mCommand.ReduceWhiteSpace();
	}

	
	Prototype::~Prototype()
	{
	}


	bool Prototype::Compile(const iString &arg, int line, int pos)
	{
		//
		//  Create worker
		//
		Statement *worker = this->CreateWorker();
		if(worker == 0)
		{
			mScript->ReportError("There is not enough free memory to proceed.",-1,line);
			return false;
		}

		//
		//  Look for the index
		//
		int last = -1;
		int index = worker->ParseOperandForIndex(arg,0,last);
		if(index == -2) return false;
		if(index>-1 && !worker->mAcceptsArrays)
		{
			mScript->ReportError("This statement does not accept an index qualifier for the body of the command.",pos,line);
			return false;
		}

		//
		//  Configure the worker
		//
		worker->mIndex = index;
		worker->mEntryInCode = mScript->mCode.Size();
		worker->mArgument = arg;
		worker->mOperand = arg.Part(last+1);
		worker->mLineInScript = line;
		worker->ShiftOperandWhiteSpace();

		if(index == 0)
		{
			//
			//  Index is an expression and needs to be recomputed at run-time, so save it.
			//
			worker->mIndexExpression = arg.Part(1,last-1);
		}

		mScript->mCode.Add(worker);
		if(mScript->mStack.Size() > 0) worker->mOwnsCalculator = true;

		return worker->CompileBody();
	}


	//
	//  Base class for statements.
	//
	Statement::Statement(iScript *script, const iString &command) : mCommand(command), mScript(script)
	{
		IERROR_ASSERT(mScript);

		mAcceptsArrays = mOwnsCalculator = false;
		mIndex = mEntryInCode = mLineInScript = -1;
	}

	
	Statement::~Statement()
	{
	}

	
	bool Statement::Execute()
	{
		//
		//  Call actual execute
		//
#ifdef I_DEBUG
		iConsole::Display(iConsole::_Info,("Executing: {"+mCommand+" "+mArgument+"}...").ToCharPointer());
#endif
		return this->ExecuteBody();
	}


	void Statement::ShiftOperandWhiteSpace()
	{
		int i = 0;
		while(mOperand.IsWhiteSpace(i)) i++; 
		mOperand = mOperand.Part(i);
	}


	int Statement::ParseOperandForIndex(const iString &str, int pos, int &last) const
	{
		if(pos<0 || pos>str.Length()-3) return -1;

		int iBeg, iEnd;
		str.FindTokenMatch(mScript->GetCalculator()->SymbolOpeningBraket(),iBeg,mScript->GetCalculator()->SymbolClosingBraket(),iEnd,pos,true);
		if(iBeg != pos) return -1;
		if(iEnd ==-1)
		{
			this->ReportError("Missing closing bracket.");
			return -2;
		}
		if(iEnd < iBeg+2)
		{
			this->ReportError("Empty index expression.");
			return -2;
		}

		bool ok;
		int res = str.Part(iBeg+1,iEnd-iBeg-1).ToInt(ok);
		if(ok)
		{
			//
			//  Index is an integer number, but is it positive?
			//
			if(res > 0)
			{
				last = iEnd;
				return res;
			}
			else
			{
				this->ReportError("Index must be a positive integer number.",str.Part(iBeg+1,iEnd-iBeg-1));
				return -1;
			}
		}
		else
		{
			//
			//  Index is not a number - may be an expression, and so should be left for the run-time evaluation.
			//  We indicate it by assigning it a zero value.
			//
			last = iEnd;
			return 0;
		}
	}


	Value* Statement::FindVariable(const iString &name) const
	{
		return mScript->FindVariable(name);
	}

	
	Calculator* Statement::CreateCalculator() const
	{
		return mScript->CreateCalculator();
	}


	void Statement::ReportError(const iString &message, const iString &context) const
	{
		int i = mArgument.FindIgnoringWhiteSpace(context);
		mScript->ReportError(message,mCommand.Length()+(i>0?i:0),-1);
	}


	void Statement::ReportError(const Calculator *c) const
	{
		if(c!=0 && !c->GetErrorMessage().IsEmpty())
		{
			int i = mArgument.FindIgnoringWhiteSpace(mOperand);
			mScript->ReportError(c->GetErrorMessage(),mCommand.Length()+(i>-1?i+c->GetErrorPosition():0),-1);
		}
	}


	//
	//  A simple, argument-less function call statement
	//
	SimpleStatement::SimpleStatement(iScript *script, const iString &command, FunctionType f) : Statement(script,command), mFun(f)
	{
	}


	bool SimpleStatement::CompileBody()
	{
		if(mFun != 0)
		{
			mOperand.ReduceWhiteSpace();
			if(!mOperand.IsEmpty())
			{
				this->ReportError("iScriptKit::SimpleStatement cannot have an argument.");
				return false;
			}
		}
		else
		{
			this->ReportError("iScriptKit::SimpleStatement is configured incorrectly.");
			return false;
		}
		return true;
	}


	bool SimpleStatement::ExecuteBody()
	{
		return this->mFun(mScript);
	}


	//
	//  A deprecated statement
	//
	DeprecatedStatement::DeprecatedStatement(iScript *script, const iString &command, const iString &message) : Statement(script,command), mMessage(message)
	{
	}


	bool DeprecatedStatement::CompileBody()
	{
		if(mMessage.IsEmpty())
		{
			this->ReportError("This statement is deprecated.");
		}
		else
		{
			this->ReportError("This statement is deprecated. "+mMessage);
		}
		return false;
	}


	bool DeprecatedStatement::ExecuteBody()
	{
		return true;
	}


	//
	//  A programmable function call statement (can be specialized to implement any command)
	//
	FunctionCallStatement::FunctionCallStatement(iScript *script, const iString &command, FunctionType1 f) : Statement(script,command), mFun1(f), mFun2(0)
	{
		mAcceptsArrays = false;
		mIndexCalculator = 0;
	}


	FunctionCallStatement::FunctionCallStatement(iScript *script, const iString &command, FunctionType2 f) : Statement(script,command), mFun1(0), mFun2(f)
	{
		mAcceptsArrays = true;
		mIndexCalculator = 0;
	}


	bool FunctionCallStatement::CompileBody()
	{
		if(mFun1==0 && mFun2==0)
		{
			this->ReportError("iScriptKit::FunctionCallStatement is configured incorrectly.");
			return false;
		}

		if(mOwnsCalculator && mIndex==0) mIndexCalculator = this->CreateCalculator();

		if(mIndex == 0)
		{
			if(!this->GetIndexCalculator()->Verify(mIndexExpression,1,0,"index"))
			{
				this->ReportError(this->GetIndexCalculator());
				return false;
			}
		}
		return true;
	}

	
	bool FunctionCallStatement::ExecuteBody()
	{
		int index = mIndex;
		if(index == 0)
		{
			//
			//  Need to recalculate
			//
			if(!this->GetIndexCalculator()->Evaluate(mIndexExpression))
			{
				this->ReportError(this->GetIndexCalculator());
				return false;
			}
			index = Constant<int>::ConvertToNative(this->GetIndexCalculator()->GetResultData()[0]);
			if(index < 1)
			{
				this->ReportError("Index must be a positive integer number.",0);
				return false;
			}
		}

		//
		//  Scalar/full vector assignment
		//
		if(index == -1)
		{
			if(mFun1 != 0)
			{
				return this->mFun1(mScript,mOperand);
			}
			else
			{
				this->ReportError("Internal error: null pointer for the called function (should be caught at compile-time).");
				return false;
			}
		}
		else
		{
			if(index > 0) index--; // convert to a C-style index
			if(mFun2 != 0)
			{
				return this->mFun2(mScript,mOperand,index);
			}
			else
			{
				this->ReportError("Internal error: null pointer for the called function (should be caught at compile-time).");
				return false;
			}
		}
	}


	Calculator* FunctionCallStatement::GetIndexCalculator() const
	{
		if(mIndexCalculator != 0) return mIndexCalculator; else return mScript->GetCalculator();
	}


	//
	//  Generic assignment statement
	//
	AssignmentStatement::AssignmentStatement(iScript *script, const iString &command, short mask, int dim, int use) : Statement(script,command), mAssignmentMask(mask), mDim(dim), mUse(use)
	{
		mCalculator = 0;
		mIndexCalculator = 0;
	}


	AssignmentStatement::~AssignmentStatement()
	{
		if(mCalculator != 0) delete mCalculator;
		if(mIndexCalculator != 0) delete mIndexCalculator;
	}


	bool AssignmentStatement::CompileBody()
	{
		if(!this->ParseOperandForAssignmentType()) return false;

		if(mOwnsCalculator)
		{
			mCalculator = this->CreateCalculator();  //  if new fails, the pointer will be zero, which is ok
			if(mIndex == 0) mIndexCalculator = this->CreateCalculator();
		}

		if(mIndex == 0)
		{
			if(!this->GetCalculator()->Verify(mOperand,1,mUse,"right-hand-side"))
			{
				this->ReportError(this->GetCalculator());
				return false;
			}
			if(!this->GetIndexCalculator()->Verify(mIndexExpression,1,0,"index"))
			{
				this->ReportError(this->GetIndexCalculator());
				return false;
			}
		}
		else
		{
			if(!this->GetCalculator()->Verify(mOperand,(mIndex>0?1:mDim),mUse,"right-hand-side"))
			{
				this->ReportError(this->GetCalculator());
				return false;
			}
		}

		return true;
	}


	bool AssignmentStatement::ExecuteBody()
	{
		int index = mIndex;
		if(index == 0)
		{
			//
			//  Need to recalculate
			//
			if(!this->GetIndexCalculator()->Evaluate(mIndexExpression))
			{
				this->ReportError(this->GetIndexCalculator());
				return false;
			}
			index = Constant<int>::ConvertToNative(this->GetIndexCalculator()->GetResultData()[0]);
			if(index < 1)
			{
				this->ReportError("Index must be a positive integer number.",0);
				return false;
			}
		}

		//
		//  Special case: variable to variable assignment (including vector variables)
		//
		if(index == -1)
		{
			iString w = this->mOperand;
			w.ReduceWhiteSpace();
			const Value *v = this->FindVariable(w);
			if(v!=0 && (mDim<0 || v->Dim()==mDim) && (mUse<0 || v->Use()==mUse))
			{
				return this->SetValue(v->CalculatorValue(),-1);
			}
		}

		//
		//  Execute our calculator
		//
		if(!this->GetCalculator()->Evaluate(this->mOperand))
		{
			this->ReportError(this->GetCalculator());
			return false;
		}

		if(index > 0) index--; // convert to a C-style index
		return this->SetValue(this->GetCalculator()->GetResult(),index);
	}



	Calculator* AssignmentStatement::GetCalculator() const
	{
		if(mCalculator != 0) return mCalculator; else return mScript->GetCalculator();
	}


	Calculator* AssignmentStatement::GetIndexCalculator() const
	{
		if(mIndexCalculator != 0) return mIndexCalculator; else return mScript->GetCalculator();
	}


	bool AssignmentStatement::ParseOperandForAssignmentType()
	{
		//
		//  Find the assignment type
		//
		mAssignmentType = _Direct;
		iString s(mOperand.Part(0,2));

		if(s == "+=")
		{
			if((mAssignmentMask & _IncAdd) == 0)
			{
				this->ReportError("Incremental assigment += is not supported for this command.");
				return false;
			}
			else
			{
				mAssignmentType = _IncAdd;
				mOperand = mOperand.Part(2);
			}
		}
		else if(s == "-=")
		{
			if((mAssignmentMask & _IncSub) == 0)
			{
				this->ReportError("Incremental assigment -= is not supported for this command.");
				return false;
			}
			else
			{
				mAssignmentType = _IncSub;
				mOperand = mOperand.Part(2);
			}
		}
		else if(s == "*=")
		{
			if((mAssignmentMask & _IncMul) == 0)
			{
				this->ReportError("Incremental assigment *= is not supported for this command.");
				return false;
			}
			else
			{
				mAssignmentType = _IncMul;
				mOperand = mOperand.Part(2);
			}
		}
		else if(s == "/=")
		{
			if((mAssignmentMask & _IncDiv) == 0)
			{
				this->ReportError("Incremental assigment /= is not supported for this command.");
				return false;
			}
			else
			{
				mAssignmentType = _IncDiv;
				mOperand = mOperand.Part(2);
			}
		}

		if(mAssignmentType==_Direct && mOperand[0]=='=')
		{
			mOperand = mOperand.Part(1);
		}

		this->ShiftOperandWhiteSpace();

		return true;
	}


	//
	//  Generic flow control statement
	//
	GenericFlowControlStatement::GenericFlowControlStatement(iScript *script, const iString &command, FlowControlType type) : Statement(script,command), mType(type)
	{
		mState = _Empty; // this a flow control statement
		mJumpPoint = 0;
	}

	
	bool GenericFlowControlStatement::CompileBody()
	{
		if(this->IsOfType(_Exit))
		{
			//
			//  Is there an opening statement?
			//
			if(mScript->mStack.Size() == 0)
			{
				this->ReportError("This closing flow control statement does not close a valid opening statement.");
				return false;
			}
			mScript->mStack.Last()->mJumpPoint = this;
			mJumpPoint = mScript->mStack.Last();
			mScript->mStack.RemoveLast();
		}

		if(this->IsOfType(_Entry))
		{
			mScript->mStack.Add(this);
		}

		return this->ChildCompileBody();
	}


	bool GenericFlowControlStatement::ExecuteBody()
	{
		if(mJumpPoint == 0)
		{
			this->ReportError("Internal error: missing jump point for a flow control statement.");
			return false;
		}

		if(this->ChildExecuteBody())
		{
			if(this->IsOfType(_Exit)) mScript->mStack.RemoveLast();
			if(this->IsOfType(_Entry)) mScript->mStack.Add(this);
			return true;
		}
		else return false;
	}


	void GenericFlowControlStatement::Jump()
	{
		if(mJumpPoint == 0)
		{
			this->ReportError("Internal error: missing jump point for a flow control statement.");
		}
		else
		{
			mScript->JumpToEntry(mJumpPoint->EntryInCode());
		}
	}


	//
	//  Closing flow control statement (like enddo or endif)
	//
	ClosingFlowControlStatement::ClosingFlowControlStatement(iScript *script, const iString &command) : GenericFlowControlStatement(script,command,_Exit)
	{
	}


	bool ClosingFlowControlStatement::ChildCompileBody()
	{
		//
		//  Nothing to do here
		//
		return true;
	}


	bool ClosingFlowControlStatement::ChildExecuteBody()
	{
		if(mJumpPoint->TestState(_Returning))
		{
			this->Jump();
		}
		return true;
	}


	//
	//  Swapping flow control statement (like else)
	//
	SwappingFlowControlStatement::SwappingFlowControlStatement(iScript *script, const iString &command) : GenericFlowControlStatement(script,command,FlowControlType(_Entry|_Exit))
	{
		mAllowInLoops = false;
		mArrivalPoint = 0;
	}


	bool SwappingFlowControlStatement::ChildCompileBody()
	{
		mArrivalPoint = mJumpPoint;
		return true;
	}


	bool SwappingFlowControlStatement::ChildExecuteBody()
	{
		//
		//  Is this allowed?
		//
		if(!mAllowInLoops && mArrivalPoint->TestState(_Returning))
		{
			this->ReportError("A swapping flow control statement is not permitted in loops.");
			return false;
		}

		//
		//  If we jumped here, continue; if we come here sequentially, jump
		//
		if(!mArrivalPoint->TestState(_Jumped)) this->Jump();

		return true;
	}


	//
	//  Conditional flow control statement (like if)
	//
	ConditionalFlowControlStatement::ConditionalFlowControlStatement(iScript *script, const iString &command) : GenericFlowControlStatement(script,command,_Entry)
	{
		mCalculator = 0;
	}


	ConditionalFlowControlStatement::~ConditionalFlowControlStatement()
	{
		if(mCalculator != 0) delete mCalculator;
	}


	Calculator* ConditionalFlowControlStatement::GetCalculator() const
	{
		if(mCalculator != 0) return mCalculator; else return mScript->GetCalculator();
	}


	bool ConditionalFlowControlStatement::ChildCompileBody()
	{
		if(!this->ParseOperandForCondition()) return false;

		if(mOwnsCalculator)
		{
			mCalculator = this->CreateCalculator();
		}

		if(!this->GetCalculator()->Verify(mOperand,1,1,"condition"))
		{
			this->ReportError(this->GetCalculator());
			return false;
		}

		return true;
	}


	bool ConditionalFlowControlStatement::ChildExecuteBody()
	{
		if(this->GetCalculator()->Evaluate(mOperand))
		{
			bool result = Constant<bool>::ConvertToNative(this->GetCalculator()->GetResultData()[0]);
			if(result) mState = _Empty; else
			{
				mState = _Jumped;
				this->Jump();
			}
			return true;
		}
		else
		{
			this->ReportError(this->GetCalculator());
			return false;
		}
	}


	//
	//  Loop flow control statement (like do or for)
	//
	LoopFlowControlStatement::LoopFlowControlStatement(iScript *script, const iString &command, bool usesSingleCalculator) : GenericFlowControlStatement(script,command,_Entry), mUsesSingleCalculator(usesSingleCalculator)
	{
		mIndex = iMath::_IntMax;
		mCount = 0;
		mStep = 1;

		int i;
		for(i=0; i<3; i++) mCalculator[i] = 0;
	}


	LoopFlowControlStatement::~LoopFlowControlStatement()
	{
		int i;
		for(i=0; i<3; i++) if(mCalculator[i] != 0) delete mCalculator[i];
	}


	Calculator* LoopFlowControlStatement::GetCalculator(int i) const
	{
		if(i>=0 && i<3)
		{
			if(mCalculator[i] != 0) return mCalculator[i]; else return mScript->GetCalculator();
		}
		else return mScript->GetCalculator();
	}


	bool LoopFlowControlStatement::ChildCompileBody()
	{
		if(!this->ParseOperandForCount()) return false;

		if(mOwnsCalculator)
		{
			mCalculator[0] = this->CreateCalculator();
			if(!mUsesSingleCalculator)
			{
				if(!mFirstExpression.IsEmpty()) mCalculator[1] = this->CreateCalculator();
				if(!mStepExpression.IsEmpty()) mCalculator[2] = this->CreateCalculator();
			}
		}

		if(!this->GetCalculator(0)->Verify(mOperand,1,0,"loop count"))
		{
			this->ReportError(this->GetCalculator(0));
			return false;
		}
		if(!mFirstExpression.IsEmpty())
		{
			if(!this->GetCalculator(1)->Verify(mFirstExpression,1,0,"loop first"))
			{
				this->ReportError(this->GetCalculator(1));
				return false;
			}
		}
		if(!mStepExpression.IsEmpty())
		{
			if(!this->GetCalculator(2)->Verify(mStepExpression,1,0,"loop step"))
			{
				this->ReportError(this->GetCalculator(2));
				return false;
			}
		}

		return true;
	}


	bool LoopFlowControlStatement::ChildExecuteBody()
	{
		if(this->GetCalculator(0)->Evaluate(mOperand))
		{
			mCount = Constant<int>::ConvertToNative(this->GetCalculator(0)->GetResultData()[0]);
		}
		else
		{
			this->ReportError(this->GetCalculator(0));
			return false;
		}

		if(mIndex == iMath::_IntMax)
		{
			if(mFirstExpression.IsEmpty())
			{
				mIndex = 1;
			}
			else
			{
				if(this->GetCalculator(1)->Evaluate(mFirstExpression))
				{
					mIndex = Constant<int>::ConvertToNative(this->GetCalculator(1)->GetResultData()[0]);
				}
				else
				{
					this->ReportError(this->GetCalculator(1));
					return false;
				}
			}
		}

		if(mStepExpression.IsEmpty())
		{
			mStep = 1;
		}
		else
		{
			if(this->GetCalculator(2)->Evaluate(mStepExpression))
			{
				mStep = Constant<int>::ConvertToNative(this->GetCalculator(2)->GetResultData()[0]);
				if(mStep == 0)
				{
					this->ReportError("Zero step in the loop statement is not permitted.");
					return false;
				}
			}
			else
			{
				this->ReportError(this->GetCalculator(2));
				return false;
			}
		}

		if((mStep>0 && mIndex<mCount) || (mStep<0 && mIndex>mCount))
		{
			mState = _Returning;
		}
		else if(mIndex == mCount)
		{
			mState = _Empty;
		}
		else
		{
			mState = _Jumped;
			this->Jump();
		}

		mScript->SetLoopParameters(mIndex,mCount);
		this->OnExecute();

		if(mState != _Returning) mIndex = iMath::_IntMax; else mIndex += mStep;

		return true;
	}


	//
	//  Variable declaration statement
	//
	VariableDeclarationStatement::VariableDeclarationStatement(iScript *script, const iString &command) : Statement(script,command)
	{
#ifdef ISCRIPT_BACKWARD_COMPATIBLE
		mAcceptsArrays = true;
#endif
	}

	
	bool VariableDeclarationStatement::CompileBody()
	{
		iString w, s(mOperand);

		//
		//  Remove commas if they are present - they are just for aestetics
		//
		s.Replace(","," ");
		s.ReduceWhiteSpace();

		if(s.IsEmpty())
		{
			this->ReportError("Variable name must be specified.");
			return false;
		}

		int i, n, last, k = 0;
		w = s.Section(" ",k,k);
		while(!w.IsEmpty())
		{
			//
			// Check that the name is alpha-numeric
			//
			i = w.Find(mScript->GetCalculator()->SymbolOpeningBraket()); if(i == -1) i = w.Length();
			if(!mScript->IsValidVariableName(w.Part(0,i)))
			{
				this->ReportError("Variable name must be alpha-numeric.",w);
				return false;
			}

			if(i == w.Length())
			{
				//
				//  No dimension argument
				//
				if(mIndex < 0)
				{
					//
					//  No dimension argument at all, a scalar
					//
					n = 1;
				}
				else if(mIndex > 1)
				{
					n = mIndex;
				}
				else
				{
					this->ReportError("Array dimension must be an integer positive constant.");
					return false;
				}
			}
			else
			{
				last = i;
				n = this->ParseOperandForIndex(w,i,last);
				if(n == 0)
				{
					this->ReportError("Array dimension must be an integer positive constant.",w.Part(i));
					return false;
				}
				if(n<0 || last!=w.Length()-1)
				{
					this->ReportError("Syntax error: foreign characters after the variable name.",w.Part(last));
					return false;
				}
			}

			w = w.Part(0,i);

			//
			//  Is it unique and non-reserved?
			//
			if(mScript->mDummyWords.Find(w) > -1)
			{
				this->ReportError("This name is reserved for a dummy command word.",w);
				return false;
			}

			iArray<const iScriptKit::Value*> &c(mScript->mConstants);
			for(i=0; i<c.Size(); i++) if(c[i]->Name() == w)
			{
				this->ReportError("This name is reserved for a global constant.",w);
				return false;
			}

			iArray<iScriptKit::Prototype*> &ps(mScript->mPrototypes);
			for(i=0; i<mScript->mNumStatementPrototypes; i++) if(ps[i]->Command() == w)
			{
				this->ReportError("This name is reserved for a command word.",w);
				return false;
			}
			for(i=mScript->mNumStatementPrototypes; i<ps.Size(); i++) if(ps[i]->Command() == w)
			{
				this->ReportError("Variable with this name has already been declared.",w);
				return false;
			}

			Prototype *p = this->CreatePrototype(w,n);
			if(p == 0)
			{
				this->ReportError("There is not enough free memory to proceed.");
				return false;
			}

			ps.Add(p);
			mVarNames.Add(w);

			k++;
			w = s.Section(" ",k,k);
		}

		return true;
	}


	bool VariableDeclarationStatement::ExecuteBody()
	{
		//
		//  Release the variables
		//
		int i;
		Value *val;
		for(i=0; i<mVarNames.Size(); i++)
		{
			val = mScript->FindVariable(mVarNames[i]);
			if(val == 0)
			{
				this->ReportError("Internal error: cannnot find a declared variable.");
				return false;
			}
			val->mIsReleased = true;
		}

		return true;
	}
};

