/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: itkIsolatedConnectedImageFilter.h,v $
  Language:  C++
  Date:      $Date: 2006/03/29 14:53:40 $
  Version:   $Revision: 1.15 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkIsolatedConnectedImageFilter_h
#define __itkIsolatedConnectedImageFilter_h

#include "itkImage.h"
#include "itkImageToImageFilter.h"

namespace itk{

/** /class IsolatedConnectedImageFilter
 * \brief Label pixels that are connected to one set of seeds but not
 * another.
 *
 * IsolatedConnectedImageFilter finds the optimal threshold to
 * separate two regions.  It has two modes, one to separate dark
 * regions surrounded by bright regions by automatically finding a
 * minimum isolating upper threshold, and another to separate bright
 * regions surrounded by dark regions by automatically finding a
 * maximum lower isolating threshold.  The mode can be chosen by
 * setting FindUpperThresholdOn()/Off().  In both cases, the isolating
 * threshold is retrieved with GetIsolatedValue().
 *
 * The algorithm labels pixels with ReplaceValue that are connected to
 * Seeds1 AND NOT connected to Seeds2.  When finding the threshold to
 * separate two dark regions surrounded by bright regions, given a
 * fixed lower threshold, the filter adjusts the upper threshold until
 * the two sets of seeds are not connected. The algorithm uses a
 * binary search to adjust the upper threshold, starting at Upper. The
 * reverse is true for finding the threshold to separate two bright
 * regions.  Lower defaults to the smallest possible value for the
 * InputImagePixelType, and Upper defaults to the largest possible
 * value for the InputImagePixelType.
 *
 * The user can also supply the Lower and Upper values to restrict the
 * search.  However, if the range is too restrictive, it could happen
 * that no isolating threshold can be found between the user specified
 * Lower and Upper values.  Therefore, unless the user is sure of the
 * bounds to set, it is recommended that the user set these values to
 * the lowest and highest intensity values in the image, respectively.
 *
 * The user can specify more than one seed for both regions to
 * separate.  The algorithm will try find the threshold that ensures
 * that all of the first seeds are contained in the resulting
 * segmentation and all of the second seeds are not contained in the
 * segmentation.
 *
 * It is possible that the algorithm may not be able to find the
 * isolating threshold because no such threshold exists.  The user can
 * check for this by querying the GetThresholdingFailed() flag.
 *
 *
 * \ingroup RegionGrowingSegmentation 
 */
template <class TInputImage, class TOutputImage>
class ITK_EXPORT IsolatedConnectedImageFilter:
    public ImageToImageFilter<TInputImage,TOutputImage>
{
public:
  /** Standard class typedefs. */
  typedef IsolatedConnectedImageFilter Self;
  typedef ImageToImageFilter<TInputImage,TOutputImage> Superclass;
  typedef SmartPointer<Self> Pointer;
  typedef SmartPointer<const Self> ConstPointer;

  /** Method for creation through the object factory. */
  itkNewMacro(Self);

  /** Run-time type information (and related methods).  */
  itkTypeMacro(IsolatedConnectedImageFilter,
               ImageToImageFilter);

  typedef TInputImage InputImageType;
  typedef typename InputImageType::Pointer InputImagePointer;
  typedef typename InputImageType::ConstPointer InputImageConstPointer;
  typedef typename InputImageType::RegionType InputImageRegionType; 
  typedef typename InputImageType::PixelType InputImagePixelType; 
  typedef typename InputImageType::IndexType IndexType;
  typedef typename InputImageType::SizeType SizeType;
  
  typedef TOutputImage OutputImageType;
  typedef typename OutputImageType::Pointer OutputImagePointer;
  typedef typename OutputImageType::RegionType OutputImageRegionType; 
  typedef typename OutputImageType::PixelType OutputImagePixelType; 

  typedef std::vector< IndexType > SeedsContainerType;

  typedef typename NumericTraits< 
                InputImagePixelType >::RealType InputRealType;
  

  void PrintSelf ( std::ostream& os, Indent indent ) const;

  /** Set seed point 1. This seed will be isolated from Seed2 (if possible).
   *  All pixels connected to this seed will be replaced with ReplaceValue.
   *  This method is deprecated, please use AddSeed() */
  void SetSeed1(const IndexType & seed)
  {
    this->ClearSeeds1();
    this->AddSeed1( seed );
  };

  
  /** Clear all the seeds1. */
  void ClearSeeds1()
  {
    m_Seeds1.clear();
  };


  /** Add seed point 1. */
  void AddSeed1(const IndexType & seed)
  {
    m_Seeds1.push_back( seed );
    this->Modified();
  };

  /** Set seed point 2. This seed will be isolated from Seed1 (if possible).
   *  This method is deprecated, please use AddSeed() */
  void SetSeed2(const IndexType & seed)
  {
    this->ClearSeeds2();
    this->AddSeed2( seed );
  };

  
  /** Clear all the seeds2. */
  void ClearSeeds2()
  {
    m_Seeds2.clear();
  };


  /** Add seed point 2. */
  void AddSeed2(const IndexType & seed)
  {
    m_Seeds2.push_back( seed );
    this->Modified();
  };

  /** Set/Get the limit on the lower threshold value. The default is the NonpositiveMin() for the InputPixelType. */
  itkSetMacro(Lower, InputImagePixelType);
  itkGetConstReferenceMacro(Lower, InputImagePixelType);

  /** Set/Get the limit on the upper threshold value. The default is the max() for the InputPixelType. */
  itkSetMacro(Upper, InputImagePixelType);
  itkGetConstReferenceMacro(Upper, InputImagePixelType);

  /** Set/Get the limit on the upper threshold value. The default is
      the max() for the InputPixelType.  These methods have been
      deprecated.  Please use Set/Get Upper instead. */
  void SetUpperValueLimit( InputImagePixelType upperValue)
    {
      this->SetUpper( upperValue );
    };
  InputImagePixelType GetUpperValueLimit()
    {
      return this->GetUpper();
    };

  /** Set/Get the precision required for the intensity threshold value. The default is 1. */
  itkSetMacro(IsolatedValueTolerance, InputImagePixelType);
  itkGetConstReferenceMacro(IsolatedValueTolerance, InputImagePixelType);

  /** Set/Get value to replace thresholded pixels. Pixels that lie
   *  within the thresholds will be replaced with this value. The
   *  default is 1. */
  itkSetMacro(ReplaceValue, OutputImagePixelType);
  itkGetConstReferenceMacro(ReplaceValue, OutputImagePixelType);

  /** Get value that isolates the two seeds. */
  itkGetConstReferenceMacro(IsolatedValue, InputImagePixelType);

  /** Set/Get whether to find an upper threshold (separating two dark
   * regions) or a lower threshold (separating two bright regions). */
  itkSetMacro(FindUpperThreshold,bool);
  itkBooleanMacro(FindUpperThreshold);
  itkGetConstReferenceMacro(FindUpperThreshold,bool);

  /** Get the flag that tells whether the algorithm failed to find a
   * threshold. */
  itkGetConstReferenceMacro(ThresholdingFailed, bool);

#ifdef ITK_USE_CONCEPT_CHECKING
  /** Begin concept checking */
  itkConceptMacro(InputHasNumericTraitsCheck,
                  (Concept::HasNumericTraits<InputImagePixelType>));
  /** End concept checking */
#endif

protected:
  IsolatedConnectedImageFilter();
  ~IsolatedConnectedImageFilter(){};
  SeedsContainerType m_Seeds1;
  SeedsContainerType m_Seeds2;
  InputImagePixelType m_Lower;
  InputImagePixelType m_Upper;
  OutputImagePixelType m_ReplaceValue;
  InputImagePixelType m_IsolatedValue;
  InputImagePixelType m_IsolatedValueTolerance;
  bool m_FindUpperThreshold;
  bool m_ThresholdingFailed;
  
  // Override since the filter needs all the data for the algorithm
  void GenerateInputRequestedRegion();

  // Override since the filter produces the entire dataset
  void EnlargeOutputRequestedRegion(DataObject *output);

  void GenerateData();
  
private:
  IsolatedConnectedImageFilter(const Self&); //purposely not implemented
  void operator=(const Self&); //purposely not implemented

};

} // end namespace itk

#ifndef ITK_MANUAL_INSTANTIATION
#include "itkIsolatedConnectedImageFilter.txx"
#endif

#endif
