/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: itkMembershipSampleGenerator.h,v $
  Language:  C++
  Date:      $Date: 2003/12/15 01:00:46 $
  Version:   $Revision: 1.11 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkMembershipSampleGenerator_h
#define __itkMembershipSampleGenerator_h


#include "itkObject.h"
#include "itkMembershipSample.h"

namespace itk{
  namespace Statistics{

/** \class MembershipSampleGenerator
 * \brief MembershipSampleGenerator generates a MembershipSample object
 * using a class mask sample.
 *
 * You can see it as a filter that takes two samples as input and generates 
 * a MembershipSample. There are two class template arguments first one 
 * is the type of basic sample that has measurement vectors in it, and
 * the second one is the type of sample that its measurement vectors has 
 * only one component - class label. What it does is that it merges 
 * measurement vectors and class labels in a single MembershipSample
 * object.
 *
 * To get the MembershipSample object you should plug in the input sample
 * and the class mask sample using SetInput and SetClassMask methods 
 * respectively.
 *
 * NOTE: This generator expects the class labels in the class mask sample
 * to be integer value where the value starts 0 and increases without
 * any missing value between 0 to the last class label value.
 *
 * \sa MembershipSample
 */

template< class TInputSample, class TClassMaskSample >
class ITK_EXPORT MembershipSampleGenerator : 
      public Object
{
public:
  /** Standard class typedefs. */
  typedef MembershipSampleGenerator Self;
  typedef Object Superclass ;
  typedef SmartPointer<Self> Pointer;
  typedef SmartPointer<const Self> ConstPointer;

  /** Standard Macros */
  itkTypeMacro(MembershipSampleGenerator, Object);
  itkNewMacro(Self) ;

  /** Input sample types alias */
  typedef typename TInputSample::MeasurementVectorType MeasurementVectorType ;

  /** Output: MembershipSample types */
  typedef MembershipSample< TInputSample > OutputType ;
  typedef typename OutputType::Pointer OutputPointer ;

  /** Stores the input sample */
  void SetInput(const TInputSample* sample) ;
  
  /** Returns the input sample pointer */
  const TInputSample* GetInput() const;

  /** Stores the class mask sample */
  void SetClassMask(const TClassMaskSample* classMask) ;

  /** Returns the class mask sample pointer*/
  const TClassMaskSample* GetClassMask() const;

  void SetNumberOfClasses(int numberOfClasses) ;

  int GetNumberOfClasses() ;

  /** Returns the pointer to the MembershipSample object
   * that has been generated by this */
  const OutputType * GetOutput() const;

  /** The actual output generation procedure. */
  void GenerateData() ;

protected:
  MembershipSampleGenerator() ;
  virtual ~MembershipSampleGenerator() {}
  void PrintSelf(std::ostream& os, Indent indent) const;

private:
  typename TInputSample::ConstPointer m_Input ;
  typename TClassMaskSample::ConstPointer m_ClassMask ;
  int m_NumberOfClasses ;
  OutputPointer m_Output ;
} ; // end of class

  } // end namespace Statistics
} // end namespace itk

#ifndef ITK_MANUAL_INSTANTIATION
#include "itkMembershipSampleGenerator.txx"
#endif

#endif
