/***************************************************************************
 *   Copyright (C) 2004 by Michael Schulze                                 *
 *   mike.s@genion.de                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "directorymodel.h"

#include <qfileinfo.h>
#include <qstringlist.h>

const QString DirectoryModel::categoryNames[] = { QString( CATNAME_ARTISTS), QString( CATNAME_PLAYLISTS), QString( CATNAME_UTILS), QString( CATNAME_TRANSFER ) };

DirectoryModel::DirectoryModel()
{
    type = ROOT;
}

DirectoryModel::DirectoryModel( const KURL& kurl) {
    parseUrl( kurl);
}

DirectoryModel::~DirectoryModel()
{
}

const QString& DirectoryModel::getCategoryName( Category category) {
    if( category >= 0 && category < NumCategories)
        return categoryNames[ category];
    else
        return QString::null;
}

DirectoryModel::Category DirectoryModel::getCategory() const {
    return category;
}

const QString& DirectoryModel::getArtist() const {
    if( category == Artists && dircomponents.count() > ARTIST_IDX)
        return dircomponents[ ARTIST_IDX];
    else
        return QString::null;
}

const QString& DirectoryModel::getPlaylist() const {
    if( category == Playlists && dircomponents.count() > PLAYLIST_IDX)
        return dircomponents[ PLAYLIST_IDX];
    else
        return QString::null;
}

const QString& DirectoryModel::getUtilityName() const {
    if( type == UTILITY && category == Utilites && dircomponents.count() > UTILITY_IDX)
        return dircomponents[ UTILITY_IDX];
    else
        return QString::null;
}

const QString& DirectoryModel::getAlbum() const {
    if( category == Artists && dircomponents.count() > ALBUM_IDX)
        return dircomponents[ ALBUM_IDX];
    else
        return QString::null;
}

const QString& DirectoryModel::getIPodName() const {
    if ( type >= IPOD ) {
        return dircomponents[ IPOD_IDX ];
    } else {
        return QString::null;
    }
}

const QString& DirectoryModel::getTrack() const {
    if( type != TRACK)
        return QString::null;
    
    if( category == Artists && dircomponents.count() > ALBUM_IDX+ 1) {
        return dircomponents[ ALBUM_IDX+ 1];
    } else if( category == Playlists && dircomponents.count() > PLAYLIST_IDX+ 1) {
        return dircomponents[ PLAYLIST_IDX+ 1];
    } else {
        return QString::null;
    }
}

bool DirectoryModel::isDeleteAllowed() const {
    return type != UNKNOWN && ( type >= PLAYLIST || type >= ARTIST ) ;
}

bool DirectoryModel::isMkDirAllowed() const {
    return type != UNKNOWN && ( type == PLAYLIST || type >= ARTIST );
}

bool DirectoryModel::isRenameAllowed() const {
    return type != UNKNOWN && ( type >= PLAYLIST || type >= ARTIST ) ;
}

bool DirectoryModel::isCopyAllowed() const {
    return type == TRACK ;
}

bool DirectoryModel::ignoreMkDir() const {
    return category == Transfer;   // ignore mkdir;
}

bool DirectoryModel::isFileExtSupported() const {
    QString fileExtension = getFileExtension();
    return fileExtension.isEmpty() || fileExtension.startsWith(".mp", false);
}

/*!
    \fn kio_ipodslaveProtocol::parseUrl( KURL& url, URLinfo& buffer)
 */
void DirectoryModel::parseUrl( const KURL& url)
{
    type = ROOT;
    isFile = false;

    if ( url.path().isEmpty() ) {
        type = UNKNOWN;
        return;
    }

    QStringList componentslist= QStringList::split( "/", url.path(), TRUE);
    for( QStringList::iterator components = componentslist.begin(); components != componentslist.end(); ++components) {
        switch( type ) {
        case ROOT:
            if ( !(*components).isEmpty() ) {
                dircomponents.push_back( (*components) );
                type = IPOD;
                isFile = false;
            }
            break;
        case IPOD:
            if( !(*components).isEmpty()) {
                type= CATEGORY;
                dircomponents.push_back( (*components));
                if( (*components) == categoryNames[ Artists]) {
                    category= Artists;
                    isFile= false;
                } else if( (*components) == categoryNames[ Playlists]) {
                    category= Playlists;
                    isFile= false;
                } else if( (*components) == categoryNames[ Utilites]) {
                    category= Utilites;
                    isFile= false;
                } else if ( (*components) == categoryNames[ Transfer ] ) {
                    category = Transfer;
                    isFile = false;
                } else {
                    category = NumCategories;
                    type= UNKNOWN;
                }
            }
            break;
        case CATEGORY:
            if( !(*components).isEmpty()) {
                dircomponents.push_back( (*components));
                switch( category) {
                case Artists:
                    type= ARTIST;
                    isFile= false;
                    break;
                case Playlists:
                    type= PLAYLIST;
                    isFile= false;
                    break;
                case Utilites:
                    type= UTILITY;
                    isFile= true;    // for now
                    break;
                case Transfer:
                    isFile = true;
                    break;
                default:
                    type= UNKNOWN;
                }
            }
            break;
        case ARTIST:
            if( !(*components).isEmpty()) {
                dircomponents.push_back( (*components));
                type= ALBUM;
                isFile= false;
            }
            break;
        case ALBUM:
        case PLAYLIST:
            if( !(*components).isEmpty()) {
                dircomponents.push_back( (*components));
                type= TRACK;
                isFile= true;
            }
            break;
        case TRACK:
        case UTILITY:
            // something weird about the URL ...
            type= UNKNOWN;
            break;
        case UNKNOWN:
            // doesn't happen
            break;
        }
        if( type == UNKNOWN)
            break;
    }
}

const QString& DirectoryModel::getCurrentDirectory() const {
    int idx= isFile ? dircomponents.count()- 2 : dircomponents.count()- 1;
    if( idx >= 0)
        return dircomponents[ idx];
    else
        return QString::null;
}

const QString& DirectoryModel::getFilename() const {
    if( dircomponents.count() > 0)
        return dircomponents[ dircomponents.count()- 1];
    else
        return QString::null;
}

QString DirectoryModel::getFileExtension() const {
    const QString & filename = getFilename();
    if ( !filename.isEmpty() ) {
        return filename.section('.', -1, -1, QString::SectionIncludeLeadingSep);
    }
    return filename;
}
