////
///  genfisuffix - a Finnish affix table generator for ispell
//   Copyright (C) 2000  Pauli Virtanen <pauli.virtanen@saunalahti.fi>
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//   The program's output is also subject to this license, and considered
//   program source code.
//
//////////////////////////////////////////////////////////////////////////////
//
//
#include <vector>
#include <string>
#include <algorithm>

#include <stdarg.h>
#include <ctype.h>
#include <locale.h>

#include "genfisuffix.hh"

//////////////////////////////////////////////////////////////////////////////
///
/// Helper functions
///
string ucase(const string &str)
{
        string new_str(str);
        string::iterator i = new_str.begin();
        for (; i != new_str.end(); ++i)
                *i = toupper(*i);
        return new_str;
}

string lcase(const string &str)
{
        string new_str(str);
        string::iterator i = new_str.begin();
        for (; i != new_str.end(); ++i)
                *i = tolower(*i);
        return new_str;
}

static void optimize_suffix(const string &replaced_pre,
                            const string &replacement_pre,
                            string &replaced,
                            string &replacement)
{
        string::const_iterator r = replaced_pre.begin();
        string::const_iterator m = replacement_pre.begin();

        while (r != replaced_pre.end() && m != replacement_pre.end() &&
               tolower(*m) == tolower(*r))
        {
                ++r; ++m;
        }

        replaced.assign(r, replaced_pre.end());
        replacement.assign(m, replacement_pre.end());
}

void output_suffix(const string &pattern_pre,
                   const string &replaced_pre,
                   const string &replacement_pre)
{
        string pattern(pattern_pre);
        string replaced;
        string replacement;
        optimize_suffix(replaced_pre, replacement_pre,
                        replaced, replacement);

        if (pattern.empty()) pattern = ".";
        if (!replaced.empty() && !replacement.empty())
        {
                cout << "   " << ucase(pattern)
                     << " >\t-" << ucase(replaced) << ","
                     << ucase(replacement)
                     << endl;
        }
        else if (!replaced.empty()) // replacement.empty()
        {
                cout << "   " << ucase(pattern)
                     << " >\t-" << ucase(replaced) << ",-"
                     << endl;
        }
        else if (!replacement.empty()) // replaced.empty()
        {
                cout << "   " << ucase(pattern)
                     << " >\t" << ucase(replacement)
                     << endl;
        }
}
void output_suffix_group(const string &pattern, const string &replaced,
                         const vector<string> &replacements)
{
        vector<string>::const_iterator i = replacements.begin();
        for (; i != replacements.end(); ++i)
                output_suffix(pattern, replaced, *i);
}

vector<string> suffix_strings(const vector<string> &strings,
                              const vector<string> &suffixes)
{
        vector<string> new_strings;
        
        vector<string>::const_iterator i = strings.begin();
        for (; i != strings.end(); ++i)
        {
                vector<string>::const_iterator j = suffixes.begin();
                for (; j != suffixes.end(); ++j)
                {
                        new_strings.push_back(*i + *j);
                }
        }

        return new_strings;
}
vector<string> suffix_strings(const vector<string> &strings,
                              const string &suffix)
{
        vector<string> new_strings;
        
        vector<string>::const_iterator i = strings.begin();
        for (; i != strings.end(); ++i)
        {
                new_strings.push_back(*i + suffix);
        }
        return new_strings;
}

vector<string> prefix_strings(const vector<string> &strings,
                              const vector<string> &prefixes)
{
        return suffix_strings(prefixes, strings);
}
vector<string> prefix_strings(const vector<string> &strings,
                              const string &prefix)
{
        vector<string> new_strings;
        
        vector<string>::const_iterator i = strings.begin();
        for (; i != strings.end(); ++i)
        {
                new_strings.push_back(prefix + *i);
        }
        return new_strings;
}

string do_tr(const string &str, const char *from, const char *from_end,
             const char *to, const char *to_end)
{
        string new_str = str;
        
        string::iterator i = new_str.begin();
        for (; i != new_str.end(); ++i)
        {
                const char *found = find(from, from_end, *i);
                if (found != from_end)
                {
                        *i = *(to + (found - from));
                }
        }

        return new_str;
}


static const string back_vowels("aouAOU");
static const string front_vowels("yY");

string to_front_vowel(const string &str)
{
        return do_tr(str,
                     back_vowels.begin(), back_vowels.end(),
                     front_vowels.begin(), front_vowels.end());
}
string to_back_vowel(const string &str)
{
        return do_tr(str,
                     front_vowels.begin(), front_vowels.end(),
                     back_vowels.begin(), back_vowels.end());
}
vector<string> to_front_vowel(const vector<string> &strs)
{
        vector<string> new_strings;
        
        vector<string>::const_iterator i = strs.begin();
        for (; i != strs.end(); ++i)
                new_strings.push_back(
                        do_tr(*i,
                              back_vowels.begin(), back_vowels.end(),
                              front_vowels.begin(), front_vowels.end()));
        return new_strings;
}
vector<string> to_back_vowel(const vector<string> &strs)
{
        vector<string> new_strings;
        
        vector<string>::const_iterator i = strs.begin();
        for (; i != strs.end(); ++i)
                new_strings.push_back(
                        do_tr(*i,
                              front_vowels.begin(), front_vowels.end(),
                              back_vowels.begin(), back_vowels.end()));
        return new_strings;
}

void add_to_vector(vector<string> &vect, const vector<string> &to_add)
{
        vect.insert(vect.end(), to_add.begin(), to_add.end());
}

void add_list_to_vector(vector<string> &vect, const char *list[])
{
        for (const char **p = list; *p != NULL; ++p)
                vect.push_back(string(*p));
}

void output_flag(const char flag)
{
        cout << "flag *" << (char)toupper(flag) << ":" << endl;
}

vector<string> new_list(const char *const str, ...)
{
        vector<string> lst;
        va_list strs;
        const char *next_str;

        if (str == NULL) return lst;
        lst.push_back(string(str));
        
        va_start(strs, str);
        while ((next_str = va_arg(strs, const char *const)) != NULL)
                lst.push_back(string(next_str));
        // Crashed? Perhaps you forgot to use NULL as the last
        // argument to this function when you called it.
        
        va_end(strs);

        return lst;
}
vector<string> combine(const vector<string> &v1, const vector<string> &v2)
{
        vector<string> nv(v1);
        nv.insert(nv.end(), v2.begin(), v2.end());
        return nv;
}
vector<string> combine(const vector<string> &v1, const vector<string> &v2,
                       const vector<string> &v3)
{
        vector<string> nv(v1);
        nv.insert(nv.end(), v2.begin(), v2.end());
        nv.insert(nv.end(), v3.begin(), v3.end());
        return nv;
}

void output_suffix_list(Suffix *lst, int n)
{
        while (n > 0)
        {
                output_suffix_group(lst->pattern, lst->replaced,
                                    lst->replacements);
                                    
                --n;
                ++lst;
        }
}
void output_suffix_list_to_front(Suffix *lst, int n)
{
        while (n > 0)
        {
                output_suffix_group(to_front_vowel(lst->pattern),
                                    to_front_vowel(lst->replaced),
                                    to_front_vowel(lst->replacements));
                --n;
                ++lst;
        }
}

vector<string> add_and_suffix(const vector<string> which,
                              const vector<string> suffixes)
{
        return combine(which, suffix_strings(which, suffixes));
}
