// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
	\brief Implements the OpenEXR reader K-3D object
	\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/ibitmap_read_format.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/ifile_format.h>
#include <k3dsdk/module.h>
#include <k3dsdk/path.h>

#include <ImfInputFile.h>

namespace
{

/////////////////////////////////////////////////////////////////////////////
// exr_reader

class exr_reader :
	public k3d::ifile_format,
	public k3d::ibitmap_read_format,
	public k3d::ideletable
{
public:
	exr_reader()
	{
	}

	unsigned long priority()
	{
		return 128;
	}

	bool query_can_handle(const k3d::filesystem::path& File)
	{
		// See if we can read it ...
		try
		{
			Imf::InputFile file(File.native_filesystem_string().c_str());
		}
		catch (const std::exception &exc)
		{
			return false;
		}

		return true;
	}

	bool read_file(const k3d::filesystem::path& File, k3d::bitmap& Bitmap)
	{
		k3d::log() << info << "Read " << File.native_console_string() << " using OpenEXRREader" << std::endl;
		
		try 
		{
 			Imf::InputFile file(File.native_filesystem_string().c_str());

 			Imath::Box2i dw = file.header().dataWindow();
   			const int width  = dw.max.x - dw.min.x + 1;
   			const int height = dw.max.y - dw.min.y + 1;
			
   			Bitmap.reset(width, height);
  			k3d::pixel* pixels = Bitmap.data();

  			Imf::FrameBuffer frameBuffer;

			frameBuffer.insert("R",
					   Imf::Slice(Imf::HALF,
						      (char *) &pixels[0].red,
						      sizeof(*pixels) * 1,
						      sizeof(*pixels) * width,
						      1, 1,
						      0.0));
			
			frameBuffer.insert("G",
					   Imf::Slice(Imf::HALF,
						      (char *) &pixels[0].green,
						      sizeof(*pixels) * 1,
						      sizeof(*pixels) * width,
						      1, 1,
						      0.0));
			
			frameBuffer.insert("B",
					   Imf::Slice(Imf::HALF,
						      (char *) &pixels[0].blue,
						      sizeof(*pixels) * 1,
						      sizeof(*pixels) * width,
						      1, 1,
						      0.0));
			
			frameBuffer.insert("A",
					   Imf::Slice(Imf::HALF,
						      (char *) &pixels[0].alpha,
						      sizeof(*pixels) * 1,
						      sizeof(*pixels) * width,
						      1, 1,
						      0.0));

			file.setFrameBuffer(frameBuffer);
 			file.readPixels(dw.min.y, dw.max.y);
		}
		catch (const std::exception &exc)
		{
			k3d::log() << error << "OpenEXRReader: Caught exception " << exc.what() << std::endl;
			return false;
		}

		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::application_plugin<exr_reader>,
			k3d::interface_list<k3d::ibitmap_read_format> > factory(
			k3d::uuid(0xdb5255b7, 0xb9c243a4, 0x81eb2645, 0x1d80ecac),
			"OpenEXRReader",
			_("OpenEXR (*.exr)"),
			"Bitmap BitmapReader");

		return factory;
	}
};

}//namespace

namespace libk3dopenexr
{

/////////////////////////////////////////////////////////////////////////////
// exr_reader_factory

k3d::iplugin_factory& exr_reader_factory()
{
	return ::exr_reader::get_factory();
}

} // namespace libk3dopenexr

